import '../../packages/ui/toast.js';
import { showToast } from '../../packages/ui/toast.js';
import { goToBoards, goToAuth } from '../../packages/services/navigation.js';
import { escapeHtml } from '../../packages/utils/format.js';

const app = document.getElementById('app');

init().catch(error => {
  console.error(error);
  if (app) {
    app.removeAttribute('data-loading');
    app.innerHTML = renderCard('error', 'Activation impossible', "Une erreur inattendue empêche l'activation.");
  }
});

async function init() {
  const search = new URLSearchParams(window.location.search);
  const token = search.get('token');
  if (!token) {
    render('error', 'Lien invalide', 'Le lien utilisé est incomplet ou expiré.');
    return;
  }
  render('loading');
  try {
    const res = await fetch(`/api/activation?token=${encodeURIComponent(token)}`, {
      credentials: 'include',
    });
    const payload = await res.json().catch(() => ({}));
    if (!res.ok || payload.status !== 'ok') {
      const message = mapError(payload.message);
      render('error', 'Activation impossible', message);
      return;
    }
    render('success', 'Licence activée ✅', `Votre rôle effectif est désormais ${payload.role ?? 'standard'}.`);
  } catch (error) {
    console.error(error);
    render('error', 'Erreur réseau', 'Impossible de contacter le serveur.');
  }
}

function render(state, title, message) {
  if (!app) return;
  app.removeAttribute('data-loading');
  switch (state) {
    case 'loading':
      app.innerHTML = '<div class="loading">Activation en cours…</div>';
      break;
    case 'success':
      app.innerHTML = renderCard('success', title, message);
      break;
    case 'error':
      app.innerHTML = renderCard('error', title, message);
      break;
    default:
      app.innerHTML = renderCard('error', 'Activation impossible', message ?? 'Lien invalide.');
  }
  bindEvents();
}

function renderCard(kind, title, message) {
  return `
    <div class="activation-card surface-card surface-card--glass surface-card--bordered ${kind}">
      <h1>${escapeHtml(title)}</h1>
      <p>${escapeHtml(message)}</p>
      <div class="activation-actions">
        <button class="btn" data-action="go-boards">Aller aux boards</button>
        <button class="btn ghost" data-action="go-login">Connexion</button>
      </div>
    </div>
  `;
}

function bindEvents() {
  if (!app) return;
  const boardsBtn = app.querySelector('[data-action="go-boards"]');
  if (boardsBtn) boardsBtn.addEventListener('click', () => goToBoards());
  const loginBtn = app.querySelector('[data-action="go-login"]');
  if (loginBtn) loginBtn.addEventListener('click', () => goToAuth());
}

function mapError(code) {
  switch (code) {
    case 'TOKEN_NOT_FOUND':
      return 'Le lien est invalide ou a déjà été utilisé.';
    case 'TOKEN_USED':
      return 'Ce lien a déjà été utilisé.';
    case 'TOKEN_EXPIRED':
      return 'Ce lien est expiré.';
    case 'TOKEN_INVALID':
      return 'Lien d’activation incorrect.';
    case 'invalid-signature':
      return 'Code de licence invalide.';
    case 'expired':
      return 'Licence expirée.';
    case 'already-used':
      return 'Licence déjà utilisée.';
    default:
      return 'Activation impossible.';
  }
}

