import '../../packages/ui/menu.js';
import '../../packages/ui/modal.js';
import '../../packages/ui/toast.js';
import { confirmDialog, promptDialog } from '../../packages/ui/dialog.js';
import { showToast } from '../../packages/ui/toast.js';
import { ensureAuthenticated } from '../../packages/services/session.js';
import { requestJson, setCsrfProvider } from '../../packages/services/http.js';
import {
  renderBrand as renderGlobalBrand,
  renderUserMenu as renderGlobalUserMenu,
  renderTopbar as renderGlobalTopbar,
  renderActionsGroup as renderTopbarActionsGroup,
} from '../../packages/ui/global-topbar.js';
import { goToBoards, goToBoard, goToModules, goToAuth } from '../../packages/services/navigation.js';
import { openFileLibrary } from '../../packages/ui/file-library.js';
import { escapeHtml, escapeAttr, formatDateTime } from '../../packages/utils/format.js';

const app = document.getElementById('app');
let csrfToken = '';

let state = {
  session: null,
  view: 'notifications',
  // New rich notifications model
  notifCategories: [],
  notifCatSelected: null,
  notifRich: [],
  showCreateCategory: false,
  showEditCategory: false,
  users: [],
  userDetails: null,
  licenses: [],
  security: { ips: [] },
  loading: false,
  // Actions admin details
  selectedActionId: null,
  actionDetails: null,
};

init().catch((error) => {
  console.error(error);
  if (app) {
    app.removeAttribute('data-loading');
    app.innerHTML = '<div class="admin-loading">Erreur de chargement.</div>';
  }
});

async function init() {
  const session = await ensureAuthenticated();
  if (!session) return;
  if (!isAdmin(session.user)) {
    if (app) {
      app.innerHTML = '<div class="admin-app forbidden">Accès réservé aux administrateurs.</div>';
    }
    return;
  }

  csrfToken = session.csrf;
  try { setCsrfProvider(() => csrfToken); } catch (_) {}
  setState({ session, view: 'notifications' });
  await loadView('notifications');
}

function setState(patch) {
  state = { ...state, ...patch };
  render();
}

function render() {
  if (!app) return;
  if (!state.session) {
    app.innerHTML = '<div class="admin-loading">Initialisation…</div>';
    return;
  }

  app.removeAttribute('data-loading');

  const loadingBanner = state.loading ? '<div class="admin-loading">Chargement…</div>' : '';

  app.innerHTML = `
    ${renderAppTopbar(state.session)}
    <div class="admin-shell">
      ${renderNav(state.view)}
      <section class="admin-panel surface-card surface-card--glass surface-card--bordered stack stack--gap-lg">
        ${renderView(state.view, state)}
      </section>
    </div>
    ${loadingBanner}
  `;

  bindEvents();
}

function renderAppTopbar(session) {
  if (!session) {
    return '';
  }
  const brand = renderGlobalBrand({ subtitle: "Console d'administration" });
  const menu = renderGlobalUserMenu(session.user, {
    items: [
      { id: 'open-boards', label: 'Mes boards', icon: '📋' },
      { id: 'open-modules', label: 'Modules', icon: '🧩' },
      { id: 'open-file-library', label: 'Mes fichiers', icon: '📁' },
    ],
    footer: [{ id: 'logout', label: 'Déconnexion', kind: 'danger', icon: '🚪' }],
  });
  return renderGlobalTopbar({
    className: 'app-topbar--compact app-topbar--wrap',
    brand,
    actions: [menu],
  });
}

function renderNav(activeView) {
  const items = [
    ['notifications', 'Notifications'],
    ['actions', 'Actions'],
    ['users', 'Utilisateurs'],
    ['licenses', 'Licences'],
    ['security', 'Sécurité'],
  ];
  return `
    <nav class="admin-nav surface-card surface-card--glass surface-card--bordered stack stack--gap-sm" aria-label="Navigation administration">
      ${items.map(([view, label]) => {
        const isActive = activeView === view;
        const aria = isActive ? ' aria-current="page"' : '';
        return `<button type="button" class="nav-btn focus-ring${isActive ? ' active' : ''}" data-view="${escapeHtml(view)}"${aria}>${escapeHtml(label)}</button>`;
      }).join('')}
    </nav>
  `;
}

function renderView(view, currentState) {
  switch (view) {
    case 'notifications':
      return renderNotificationsRich(currentState);
    case 'actions':
      return renderActionsCatalog(currentState);
    case 'users':
      return renderUsers(currentState.users);
    case 'user-details':
      return renderUserDetails(currentState.userDetails);
    case 'licenses':
      return renderLicenses(currentState.licenses);
    case 'security':
      return renderSecurity(currentState.security);
    default:
      return '<p>Vue indisponible.</p>';
  }
}

function renderActionsCatalog(s) {
  const list = Array.isArray(s.actionsCatalog) ? s.actionsCatalog : [];
  const details = s.actionDetails || null;
  return `
    <div class="admin-section stack stack--gap-sm">
      <div class="panel-header"><h2>Catalogue d'actions</h2></div>
      <section class="surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <h3>Créer une action</h3>
        <form class="admin-form" data-form="action-create">
          <div class="form-grid form-grid--columns">
            <label>action_id
              <input name="action_id" placeholder="subscribe_category" required />
            </label>
            <label>kind
              <select name="kind">
                <option value="User.SubscribeCategory">User.SubscribeCategory</option>
                <option value="User.UnsubscribeCategory">User.UnsubscribeCategory</option>
                <option value="User.SetCategoryFrequencyOverride">User.SetCategoryFrequencyOverride</option>
                <option value="UserData.SetKey">UserData.SetKey</option>
                <option value="User.SubscribeAndSetPref">User.SubscribeAndSetPref</option>
                <option value="OpenDialog">OpenDialog</option>
              </select>
            </label>
          </div>
          <div><button class="btn" type="submit">Créer</button></div>
        </form>
      </section>
      <section class="surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <h3>Créer une version</h3>
        <form class="admin-form" data-form="action-version-create">
          <div class="form-grid form-grid--columns">
            <label>Mode
              <div class="btn-group" role="group" aria-label="Mode de saisie">
                <button class="btn ghost" type="button" data-version-mode="guided" aria-pressed="true">Guidé</button>
                <button class="btn ghost" type="button" data-version-mode="json" aria-pressed="false">JSON</button>
              </div>
            </label>
          </div>
          <div class="form-grid form-grid--columns">
            <label>action_id
              <select name="action_id" required>
                ${list.map(a => `<option value="${escapeAttr(String(a.action_id))}">${escapeHtml(String(a.action_id))} — ${escapeHtml(String(a.kind))}</option>`).join('')}
              </select>
            </label>
            <label>version
              <input name="version" type="number" min="1" required />
            </label>
            <label>status
              <select name="status">
                <option value="draft">draft</option>
                <option value="active" selected>active</option>
                <option value="deprecated">deprecated</option>
                <option value="disabled">disabled</option>
              </select>
            </label>
          </div>
          <div data-section="guided-def" class="stack stack--gap-sm">
            <div class="form-grid form-grid--columns">
              <label>Label (fr)
                <input name="def_label_fr" placeholder="Ex: J’aime…" />
              </label>
              <label>Audit tag
                <input name="def_audit_tag" placeholder="ex: like_and_subscribe" />
              </label>
            </div>
            <div data-guided-kind-fields class="stack stack--gap-sm">
              <!-- champs spécifiques au kind (auto) -->
            </div>
            <details class="surface-card surface-card--muted surface-card--bordered"><summary>Prévisualisation JSON</summary>
              <pre style="white-space:pre-wrap;word-break:break-word;"><code data-guided-preview></code></pre>
            </details>
          </div>
          <label class="form-field">definition (JSON)
            <textarea name="definition" placeholder='{"label":{"fr":"S’abonner"},"payloadSchema":{},"defaultParams":{},"capabilities":[],"auditTag":"notif_subscribe"}'></textarea>
          </label>
          <div><button class="btn" type="submit">Créer la version</button></div>
        </form>
      </section>
      <section class="surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <h3>Liste des actions</h3>
        ${list.length ? `<table class="admin-table"><thead><tr><th>action_id</th><th>kind</th><th>created</th></tr></thead><tbody>
          ${list.map(a => `<tr data-action="select-action" data-id="${escapeAttr(String(a.action_id))}" style="cursor:pointer"><td>${escapeHtml(a.action_id)}</td><td>${escapeHtml(a.kind)}</td><td>${formatDateTime((a.created_at||0)*1000)}</td></tr>`).join('')}
        </tbody></table>` : '<p class="admin-empty surface-card surface-card--muted surface-card--bordered">Aucune action.</p>'}
      </section>
      ${details ? `
      <section class="surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <div class="panel-header"><h3>Détails: ${escapeHtml(details.action_id)} <small class="muted">(${escapeHtml(details.kind)})</small></h3></div>
        ${(Array.isArray(details.versions) && details.versions.length) ? `
          <table class="admin-table"><thead><tr><th>Version</th><th>Status</th><th>Etag</th><th>Modifié</th><th></th><th></th></tr></thead><tbody>
            ${details.versions.map(v => `
              <tr>
                <td>${escapeHtml(String(v.version))}</td>
                <td>
                  <select data-action="set-version-status" data-action-id="${escapeAttr(String(details.action_id))}" data-version="${escapeAttr(String(v.version))}">
                    ${['draft','active','deprecated','disabled'].map(s => `<option value="${s}" ${String(v.status)===s?'selected':''}>${s}</option>`).join('')}
                  </select>
                </td>
                <td><code>${escapeHtml(String(v.etag))}</code></td>
                <td>${v.updated_at ? formatDateTime(Number(v.updated_at)*1000) : ''}</td>
                <td><button class="btn ghost" data-action="apply-version-status" data-action-id="${escapeAttr(String(details.action_id))}" data-version="${escapeAttr(String(v.version))}">Mettre à jour</button></td>
                <td>${String(v.status||'').toLowerCase()==='active' ? '<span class="muted">—</span>' : `<button class="btn ghost danger" data-action="delete-action-version" data-action-id="${escapeAttr(String(details.action_id))}" data-version="${escapeAttr(String(v.version))}">Supprimer</button>`}</td>
              </tr>
            `).join('')}
          </tbody></table>
        ` : '<p class="admin-empty surface-card surface-card--muted surface-card--bordered">Aucune version.</p>'}
        <div class="admin-actions">
          <button class="btn danger" data-action="delete-action" data-id="${escapeAttr(String(details.action_id))}">Supprimer l\'action</button>
        </div>
      </section>
      ` : ''}
    </div>
  `;
}

function renderNotificationsRich(s) {
  const cats = s.notifCategories ?? [];
  const selected = Number(s.notifCatSelected ?? 0) || (cats[0]?.id ?? null);
  const items = (s.notifRich ?? []).filter(n => !selected || n.category_id === selected);
  const tabs = cats.length ? `<div class="admin-tabs">${cats.map(c => {
    const mode = String(c.dispatch_mode || 'BROADCAST').toUpperCase() === 'PERSONALIZED' ? 'P' : 'BR';
    const fk = String(c.frequency_kind || 'IMMEDIATE').toUpperCase();
    const fp = typeof c.frequency_param === 'number' ? c.frequency_param : null;
    const freq = fk === 'IMMEDIATE' ? 'IM' : (fk === 'EVERY_N_DAYS' ? `D${fp ?? ''}` : (fk === 'WEEKLY' ? `W${fp ?? ''}` : `M${fp ?? ''}`));
    const ov = c.allow_user_override ? '⚙️' : '';
    const badges = `<span class=\"badge badge--muted\" title=\"Mode\">${mode}</span> <span class=\"badge\" title=\"Fréquence\">${freq}</span> ${ov}`;
    return `
    <button class="notifrich-tab${selected===c.id?' is-active':''}" data-action="select-category" data-id="${c.id}">
      <span class="label">${escapeHtml(c.name)}<br><small class="meta-line">${badges}</small></span>
      <span class="edit" title="Modifier" data-action="edit-category-open" data-id="${c.id}">✎</span>
      <span class="close" title="Supprimer" data-action="delete-category" data-id="${c.id}">×</span>
    </button>`;
  }).join('')}</div>` : '<p class="admin-empty surface-card surface-card--muted surface-card--bordered">Aucune catégorie.</p>';
  const selectedCat = cats.find(c => c.id === selected) || null;
  const table = items.length ? `
    <table class="admin-table">
      <thead>
        <tr>
          <th>#</th><th>Titre</th><th>Actif</th><th>Poids</th><th>Créée</th><th></th>
        </tr>
      </thead>
      <tbody>
        ${items.map(n => `
          <tr>
            <td>${n.id}</td>
            <td>${escapeHtml(n.title)}</td>
            <td>${n.active ? '<span class="badge badge--success">Active</span>' : '<span class="badge badge--muted">Inactif</span>'}</td>
            <td>${n.weight ?? 0}</td>
            <td>${formatDateTime(n.created_at)}</td>
            <td>
              <div class="admin-actions">
                <button class="btn ghost" data-action="edit-notification-rich" data-id="${n.id}">Modifier</button>
                <button class="btn ghost" data-action="delete-notification-rich" data-id="${n.id}">Supprimer</button>
              </div>
            </td>
          </tr>
        `).join('')}
      </tbody>
    </table>
  ` : '<p class="admin-empty surface-card surface-card--muted surface-card--bordered">Aucune notification dans cette catégorie.</p>';

  return `
    <div class="admin-section stack stack--gap-sm">
      <div class="panel-header">
        <h2>Notifications (riches)</h2>
      </div>
      <section class="surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <div class="panel-header">
          <h3>Catégories</h3>
          <div class="admin-actions">
            <button class="btn" data-action="toggle-create-category">${s.showCreateCategory ? 'Fermer' : 'Créer'}</button>
          </div>
        </div>
        ${tabs}
        ${s.showCreateCategory ? `<form class=\"admin-form\" data-form=\"category\">
          <div class="form-grid form-grid--columns">
            <label>Slug
              <input name="slug" placeholder="ex: news" required />
            </label>
            <label>Nom
              <input name="name" placeholder="Nom visible" required />
            </label>
            <label>Ordre
              <input name="sort_order" type="number" value="0" />
            </label>
            <label style="flex-direction:row;align-items:center;gap:8px;">
              <input type="checkbox" name="active" checked /> Active
            </label>
          </div>
          <div class="form-grid form-grid--columns">
            <label>Audience
              <select name="audience_mode">
                <option value="EVERYONE">EVERYONE</option>
                <option value="SUBSCRIBERS">SUBSCRIBERS</option>
              </select>
            </label>
            <label>Mode
              <select name="dispatch_mode">
                <option value="BROADCAST">BROADCAST</option>
                <option value="PERSONALIZED">PERSONALIZED</option>
              </select>
            </label>
            <label>Fréquence
              <select name="frequency_kind">
                <option value="IMMEDIATE">IMMEDIATE</option>
                <option value="EVERY_N_DAYS">EVERY_N_DAYS</option>
                <option value="WEEKLY">WEEKLY</option>
                <option value="MONTHLY">MONTHLY</option>
              </select>
            </label>
            <label>Paramètre (selon fréquence)
              <input name="frequency_param" type="number" placeholder="ex: 1..7 / 14 / 21" />
            </label>
            <label>Ancre (timestamp UTC)
              <input name="anchor_ts" type="number" placeholder="optionnel" />
            </label>
            <div class="admin-hint">Ancre = minuit Europe/Paris converti en UTC (évite la dérive).</div>
            <label style="flex-direction:row;align-items:center;gap:8px;">
              <input type="checkbox" name="allow_user_override" /> Autoriser override (PERSONALIZED uniquement)
            </label>
          </div>
          <div>
            <button class="btn" type="submit">Créer la catégorie</button>
          </div>
        </form>` : ''}
        ${selectedCat && s.showEditCategory ? `
        <form class="admin-form" data-form="category-edit">
          <div class="panel-header">
            <h4>Modifier la catégorie sélectionnée</h4>
            <div class="admin-actions"><button class="btn ghost" type="button" data-action="close-edit-category">×</button></div>
          </div>
          <div class="form-grid form-grid--columns">
            <input type="hidden" name="id" value="${selectedCat.id}" />
            <label>Slug
              <input name="slug" value="${escapeAttr(selectedCat.slug)}" required />
            </label>
            <label>Nom
              <input name="name" value="${escapeAttr(selectedCat.name)}" required />
            </label>
            <label>Ordre
              <input name="sort_order" type="number" value="${Number(selectedCat.sort_order ?? 0)}" />
            </label>
            <label style="flex-direction:row;align-items:center;gap:8px;">
              <input type="checkbox" name="active" ${selectedCat.active ? 'checked' : ''} /> Active
            </label>
          </div>
          <div class="form-grid form-grid--columns">
            <label>Audience
              <select name="audience_mode">
                <option value="EVERYONE" ${String(selectedCat.audience_mode||'EVERYONE').toUpperCase()==='EVERYONE'?'selected':''}>EVERYONE</option>
                <option value="SUBSCRIBERS" ${String(selectedCat.audience_mode||'').toUpperCase()==='SUBSCRIBERS'?'selected':''}>SUBSCRIBERS</option>
              </select>
            </label>
            <label>Mode
              <select name="dispatch_mode">
                <option value="BROADCAST" ${String(selectedCat.dispatch_mode||'BROADCAST').toUpperCase()==='BROADCAST'?'selected':''}>BROADCAST</option>
                <option value="PERSONALIZED" ${String(selectedCat.dispatch_mode||'').toUpperCase()==='PERSONALIZED'?'selected':''}>PERSONALIZED</option>
              </select>
            </label>
            <label>Fréquence
              <select name="frequency_kind">
                <option value="IMMEDIATE" ${String(selectedCat.frequency_kind||'IMMEDIATE').toUpperCase()==='IMMEDIATE'?'selected':''}>IMMEDIATE</option>
                <option value="EVERY_N_DAYS" ${String(selectedCat.frequency_kind||'').toUpperCase()==='EVERY_N_DAYS'?'selected':''}>EVERY_N_DAYS</option>
                <option value="WEEKLY" ${String(selectedCat.frequency_kind||'').toUpperCase()==='WEEKLY'?'selected':''}>WEEKLY</option>
                <option value="MONTHLY" ${String(selectedCat.frequency_kind||'').toUpperCase()==='MONTHLY'?'selected':''}>MONTHLY</option>
              </select>
            </label>
            <label>Paramètre (selon fréquence)
              <input name="frequency_param" type="number" value="${selectedCat.frequency_param ?? ''}" placeholder="ex: 1..7 / 14 / 21" />
            </label>
            <label>Ancre (timestamp UTC)
              <input name="anchor_ts" type="number" value="${selectedCat.anchor_ts ?? ''}" placeholder="optionnel" />
            </label>
            <div class="admin-hint">Ancre = minuit Europe/Paris converti en UTC (évite la dérive).</div>
            <label style="flex-direction:row;align-items:center;gap:8px;">
              <input type="checkbox" name="allow_user_override" ${selectedCat.allow_user_override ? 'checked' : ''} /> Autoriser override (PERSONALIZED uniquement)
            </label>
          </div>
          <div>
            <button class="btn" type="submit">Mettre à jour</button>
          </div>
        </form>` : ''}
      </section>

      <section class="surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <h3>Nouvelle notification</h3>
        <form class="admin-form" data-form="notification-rich">
          <div class="form-grid form-grid--columns">
            <input type="hidden" name="id" value="" />
            <input type="hidden" name="category_id" value="${selected ?? ''}" />
            <label>Titre
              <input name="title" placeholder="Titre" required />
            </label>
            <label>Émetteur
              <input name="emitter" placeholder="Optionnel" />
            </label>
            <label>Poids
              <input name="weight" type="number" value="0" />
            </label>
            <label>Sequence index
              <input name="sequence_index" type="number" placeholder="auto si vide" />
            </label>
            <label style="flex-direction:row;align-items:center;gap:8px;">
              <input type="checkbox" name="active" checked /> Active
            </label>
          </div>
          <div class="form-grid form-grid--columns">
            <label>Gabarit
              <select name="template_id">
                <option value="">— Choisir un gabarit —</option>
                ${(s.notifTemplates || []).map(t => `<option value="${escapeAttr(String(t.id))}">${escapeHtml(String(t.name ?? t.id))}</option>`).join('')}
              </select>
            </label>
            <div style="display:flex;align-items:flex-end;gap:8px;flex-wrap:wrap;">
              <button class="btn ghost" type="button" data-action="load-template">Charger le gabarit</button>
              <button class="btn ghost" type="button" data-action="toggle-act-palette">Action picker</button>
              <button class="btn ghost" type="button" data-action="toggle-act-json">Action avancée (JSON)</button>
            </div>
          </div>
          <div class="admin-actions-toolbox stack stack--gap-sm">
            <section data-section="actions-palette" class="surface-card surface-card--muted surface-card--bordered stack stack--gap-sm" style="display:none;">
              <div class="panel-header">
                <h4>Actions (catalogue) — Palette</h4>
                <div class="admin-actions">
                  <button class="btn ghost" type="button" data-action="close-act-palette">Fermer</button>
                </div>
              </div>
              <div class="form-grid form-grid--columns">
                <label>Id local
                  <input name="act_local_id" placeholder="ex: sub" />
                </label>
                <label>Action
                  <select name="act_action_id">
                    ${(Array.isArray(s.actionsCatalog) ? s.actionsCatalog : []).map(a => `<option value="${escapeAttr(String(a.action_id))}">${escapeHtml(String(a.action_id))} — ${escapeHtml(String(a.kind))}</option>`).join('')}
                  </select>
                </label>
                <label>Version
                  <select name="act_version"></select>
                </label>
              </div>
              <label class="form-field">Paramètres (JSON)
                <textarea name="act_params_json" placeholder='{}'></textarea>
              </label>
              <div class="stack stack--gap-sm">
                <div class="muted">Paramètres (formulaire)</div>
                <div data-act-params-form class="stack stack--gap-sm"></div>
              </div>
              <div><button class="btn" type="button" data-action="add-action-ref">Ajouter</button></div>
              ${(() => {
                const list = Array.isArray(s.notifActions) ? s.notifActions : [];
                if (!list.length) return '<p class="admin-empty surface-card surface-card--muted surface-card--bordered">Aucune action ajoutée.</p>';
                return `<table class=\"admin-table\"><thead><tr><th>id</th><th>ref</th><th>version</th><th>shortcode</th><th></th></tr></thead><tbody>${list.map((a,i)=>{
                  const lid = String(a?.id ?? a?.ref ?? '');
                  const ref = String(a?.ref ?? '');
                  const ver = Number(a?.version ?? 0) || 0;
                  const lbl = (a && a.label && typeof a.label.fr === 'string') ? a.label.fr : lid || ref || 'Action';
                  const sc = `[sb:action id=\"${escapeAttr(lid)}\"]${escapeHtml(lbl)}[/sb:action]`;
                  return `<tr><td>${escapeHtml(lid)}</td><td>${escapeHtml(ref)}</td><td>${ver}</td><td><code>${sc}</code></td><td><button class=\"btn ghost danger\" type=\"button\" data-action=\"remove-action-ref\" data-index=\"${i}\">Retirer</button></td></tr>`;
                }).join('')}</tbody></table>`;
              })()}
            </section>
            <section data-section="actions-json" class="surface-card surface-card--muted surface-card--bordered stack stack--gap-sm" style="display:none;">
              <div class="panel-header">
                <h4>Actions (catalogue) — JSON</h4>
                <div class="admin-actions">
                  <button class="btn ghost" type="button" data-action="close-act-json">Fermer</button>
                </div>
              </div>
              <label class="form-field">Références (actions_ref_json)
                <textarea name="actions_ref_json" placeholder='{"actions":[]}'></textarea>
              </label>
              <label class="form-field">Snapshot‑light (actions_snapshot_json)
                <textarea name="actions_snapshot_json" placeholder='{"version":1,"policy":"snapshot","actions":[]}'></textarea>
              </label>
            </section>
          </div>
          <label class="form-field">HTML
            <textarea name="content_html" placeholder="<div>...</div>" required></textarea>
          </label>
          <label class="form-field">CSS
            <textarea name="content_css" placeholder="/* styles */"></textarea>
          </label>
          
          <div>
            <button class="btn primary" type="submit" data-role="notifrich-submit">Créer la notification</button>
          </div>
        </form>
        <div id="notifrich-preview" class="surface-card surface-card--muted surface-card--bordered" style="padding:8px;">
          <div class="muted">Prévisualisation à partir des champs ci‑dessus.</div>
        </div>
        <div class="stack stack--gap-sm">
          ${table}
        </div>
      </section>
    </div>
  `;
}

function renderUsers(users) {
  return `
    <div class="admin-section stack stack--gap-sm">
      <div class="panel-header">
        <h2>Utilisateurs</h2>
      </div>
      <form class="admin-form surface-card surface-card--muted surface-card--bordered" data-form="user">
        <div class="form-grid form-grid--columns">
          <label>Email
            <input name="email" type="email" required />
          </label>
          <label>Mot de passe
            <input name="password" type="password" required />
          </label>
          <label>Rôle
            <input name="role" placeholder="standard, admin…" />
          </label>
          <label>Pseudo
            <input name="pseudo" placeholder="Optionnel" />
          </label>
        </div>
        <div>
          <button class="btn primary" type="submit">Créer l'utilisateur</button>
        </div>
      </form>
      ${users.length ? `
        <table class="admin-table">
          <thead>
            <tr>
              <th>#</th>
              <th>Email</th>
              <th>Pseudo</th>
              <th>Rôle</th>
              <th>Créé</th>
              <th>Bloqué</th>
              <th>Boards</th>
              <th></th>
            </tr>
          </thead>
          <tbody>
            ${users.map(renderUserRow).join('')}
          </tbody>
        </table>
      ` : '<p class="admin-empty surface-card surface-card--muted surface-card--bordered">Aucun utilisateur additionnel.</p>'}
    </div>
  `;
}

function renderUserRow(user) {
  return `
    <tr>
      <td>${user.id}</td>
      <td>${escapeHtml(user.email)}</td>
      <td>${escapeHtml(user.pseudo ?? '')}</td>
      <td>${escapeHtml(user.role ?? 'standard')}</td>
      <td>${formatDateTime(user.created_at)}</td>
      <td>${user.blocked ? '<span class="badge badge--warning">Oui</span>' : '<span class="badge badge--success">Non</span>'}</td>
      <td>${user.board_count ?? 0}</td>
      <td>
        <div class="admin-actions">
          <button class="btn ghost" data-action="view-user-data" data-id="${user.id}">Fiche</button>
          <button class="btn ghost" data-action="edit-user" data-id="${user.id}">Modifier</button>
          <button class="btn ghost" data-action="delete-user" data-id="${user.id}">Supprimer</button>
        </div>
      </td>
    </tr>
  `;
}

function renderUserDetails(userData) {
  if (!userData) {
    return '<div class="admin-loading">Chargement de la fiche…</div>';
  }
  const p = userData.profile || {};
  const subs = Array.isArray(userData.subscriptions) ? userData.subscriptions : [];
  const nus = userData.nus || { unread: 0, seen: 0, archived: 0, deleted: 0 };
  const kv = Array.isArray(userData.user_data) ? userData.user_data : [];
  const ach = Array.isArray(userData.achievements) ? userData.achievements : [];
  const subsTable = subs.length ? `
    <table class="admin-table">
      <thead><tr><th>Catégorie</th><th>Mode</th><th>Abonné</th><th>Override</th><th>Anchor</th><th>Dernier</th></tr></thead>
      <tbody>
        ${subs.map(s => `<tr>
          <td>${escapeHtml(s.name ?? s.slug ?? String(s.category_id))}</td>
          <td>${escapeHtml(String(s.dispatch_mode || ''))}</td>
          <td>${s.subscribed ? 'Oui' : 'Non'}</td>
          <td>${s.override_kind ? escapeHtml(`${s.override_kind}(${s.override_param ?? ''})`) : '—'}</td>
          <td>${s.cycle_anchor_ts ? formatDateTime(s.cycle_anchor_ts * 1000) : '—'}</td>
          <td>${s.last_delivered_notification_id ?? '—'}</td>
        </tr>`).join('')}
      </tbody>
    </table>
  ` : '<p class="admin-empty surface-card surface-card--muted surface-card--bordered">Aucun abonnement.</p>';
  const kvList = kv.length ? `<ul class="admin-list">${kv.map(e => `<li><code>${escapeHtml(e.namespace)}/${escapeHtml(e.key)}</code> = <pre>${escapeHtml(JSON.stringify(e.value))}</pre></li>`).join('')}</ul>` : '<p class="admin-empty">Aucune donnée personnalisée.</p>';
  const achList = ach.length ? `<ul class="admin-list">${ach.map(a => `<li><strong>${escapeHtml(a.code)}</strong> — ${formatDateTime(a.earned_at * 1000)}</li>`).join('')}</ul>` : '<p class="admin-empty">Aucun trophée.</p>';
  return `
    <div class="admin-section stack stack--gap-md">
      <div class="panel-header">
        <h2>Fiche utilisateur</h2>
        <div class="admin-actions">
          <button class="btn ghost" data-action="back-users">Retour</button>
        </div>
      </div>
      <section class="surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <h3>Profil</h3>
        <div class="app-grid app-grid--metadata">
          <div><div>ID</div><strong>${p.id}</strong></div>
          <div><div>Email</div><strong>${escapeHtml(p.email ?? '')}</strong></div>
          <div><div>Pseudo</div><strong>${escapeHtml(p.pseudo ?? '')}</strong></div>
          <div><div>Rôle</div><strong>${escapeHtml(p.role ?? 'standard')}</strong></div>
          <div><div>Créé</div><strong>${formatDateTime((p.created_at ?? 0) * 1000)}</strong></div>
          <div><div>Vérifié</div><strong>${p.email_verified_at ? formatDateTime(p.email_verified_at * 1000) : '—'}</strong></div>
        </div>
      </section>
      <section class="surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <h3>Abonnements</h3>
        ${subsTable}
      </section>
      <section class="surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <h3>Notifications — métriques</h3>
        <div class="app-grid app-grid--metadata">
          <div><div>Non lues</div><strong>${nus.unread ?? 0}</strong></div>
          <div><div>Vues</div><strong>${nus.seen ?? 0}</strong></div>
          <div><div>Archivées</div><strong>${nus.archived ?? 0}</strong></div>
          <div><div>Supprimées</div><strong>${nus.deleted ?? 0}</strong></div>
        </div>
      </section>
      <section class="surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <h3>Données utilisateur (user_data)</h3>
        ${kvList}
      </section>
      <section class="surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <h3>Trophées</h3>
        ${achList}
      </section>
    </div>
  `;
}

function renderLicenses(licenses) {
  return `
    <div class="admin-section stack stack--gap-sm">
      <div class="panel-header">
        <h2>Licences</h2>
      </div>
      <form class="admin-form surface-card surface-card--muted surface-card--bordered" data-form="license">
        <div class="form-grid form-grid--columns">
          <label>Code
            <input name="code" placeholder="Optionnel (auto si vide)" />
          </label>
          <label>Rôle
            <input name="role" placeholder="standard, premium…" required />
          </label>
          <label>Packs (séparés par des virgules)
            <input name="packs" placeholder="packA, packB" />
          </label>
          <label>Expiration
            <input name="expiresAt" type="date" />
          </label>
        </div>
        <div>
          <button class="btn primary" type="submit">Créer la licence</button>
        </div>
      </form>
      ${licenses.length ? `
        <table class="admin-table">
          <thead>
            <tr>
              <th>#</th>
              <th>Code</th>
              <th>Rôle</th>
              <th>Packs</th>
              <th>Assignée à</th>
              <th>Expiration</th>
              <th></th>
            </tr>
          </thead>
          <tbody>
            ${licenses.map(renderLicenseRow).join('')}
          </tbody>
        </table>
      ` : '<p class="admin-empty surface-card surface-card--muted surface-card--bordered">Aucune licence disponible.</p>'}
    </div>
  `;
}

function renderLicenseRow(license) {
  return `
    <tr>
      <td>${license.id}</td>
      <td>${escapeHtml(license.code)}</td>
      <td>${escapeHtml(license.role)}</td>
      <td>${escapeHtml((license.packs ?? []).join(', '))}</td>
      <td>${license.assigned_email ? escapeHtml(license.assigned_email) : '<span class="badge badge--muted">Libre</span>'}</td>
      <td>${license.expires_at ? formatDateTime(license.expires_at * 1000) : '-'}</td>
      <td>
        <div class="admin-actions">
          <button class="btn ghost" data-action="assign-license" data-id="${license.id}">Assigner</button>
          <button class="btn ghost" data-action="unassign-license" data-id="${license.id}">Libérer</button>
          <button class="btn ghost" data-action="link-license" data-id="${license.id}">Lien</button>
          <button class="btn ghost" data-action="delete-license" data-id="${license.id}">Supprimer</button>
        </div>
      </td>
    </tr>
  `;
}

function renderSecurity(security) {
  const ips = security?.ips ?? [];
  return `
    <div class="admin-section stack stack--gap-sm">
      <div class="panel-header">
        <h2>Sécurité</h2>
        <div class="admin-actions">
          <button class="btn ghost" data-action="clear-all-ip">Purger les IP</button>
          <button class="btn ghost" data-action="reset-schema">Réinitialiser le schéma</button>
        </div>
      </div>
      ${ips.length ? `
        <div class="security-list">
          ${ips.map(renderSecurityCard).join('')}
        </div>
      ` : '<p class="admin-empty surface-card surface-card--muted surface-card--bordered">Aucune IP bannie actuellement.</p>'}
    </div>
  `;
}

function renderSecurityCard(record) {
  return `
    <div class="security-card surface-card surface-card--muted surface-card--bordered">
      <div>
        <strong>${escapeHtml(record.ip)}</strong>
        <div class="meta">
          <span>Échecs: ${record.fails ?? 0}</span>
          <span>Banni jusqu'au: ${record.banned_until ? formatDateTime(record.banned_until * 1000) : '—'}</span>
        </div>
      </div>
      <div class="security-actions">
        <button class="btn ghost" data-action="clear-ip" data-ip="${record.ip}">Débloquer</button>
      </div>
    </div>
  `;
}

function bindEvents() {
  if (!app) return;
  app.querySelectorAll('[data-view]').forEach(btn => {
    btn.addEventListener('click', () => changeView(btn.dataset.view));
  });

  const logoutBtn = app.querySelector('[data-action="logout"]');
  if (logoutBtn) logoutBtn.addEventListener('click', logout);
  const goBoards = app.querySelector('[data-action="go-boards"]');
  if (goBoards) goBoards.addEventListener('click', () => goToBoards());
  const goBoard = app.querySelector('[data-action="go-board"]');
  if (goBoard) goBoard.addEventListener('click', () => goToBoard());
  const goModules = app.querySelector('[data-action="go-modules"]');
  if (goModules) goModules.addEventListener('click', () => goToModules());
  const openFiles = app.querySelector('[data-action="open-file-library"]');
  if (openFiles) openFiles.addEventListener('click', async (e) => { e.preventDefault(); try { await openFileLibrary({ title: 'Mes fichiers', showSelect: false }); } catch (_) {} });
  app.querySelectorAll('[data-action="open-boards"]').forEach(btn => {
    btn.addEventListener('click', event => {
      event.preventDefault();
      goToBoards();
    });
  });
  app.querySelectorAll('[data-action="open-modules"]').forEach(btn => {
    btn.addEventListener('click', event => {
      event.preventDefault();
      goToModules();
    });
  });

  // New rich notifications bindings
  const catForm = app.querySelector('[data-form="category"]');
  if (catForm) catForm.addEventListener('submit', handleCategoryCreate);
  const catEditForm = app.querySelector('[data-form="category-edit"]');
  if (catEditForm) catEditForm.addEventListener('submit', handleCategoryUpdate);
  const toggleCreateBtn = app.querySelector('[data-action="toggle-create-category"]');
  if (toggleCreateBtn) toggleCreateBtn.addEventListener('click', () => setState({ showCreateCategory: !state.showCreateCategory }));
  const richForm = app.querySelector('[data-form="notification-rich"]');
  if (richForm) {
    richForm.addEventListener('submit', handleNotificationRichSubmit);
    setupRichPreview(richForm);
    // Réappliquer un éventuel brouillon capturé avant re-render (préserve Titre/HTML/CSS/JSON)
    try {
      if (state && state.notifDraft) {
        applyNotifDraft(richForm, state.notifDraft);
        try { state.notifDraft = null; } catch (_) {}
      }
      // Première ouverture (création): charger un brouillon persistant si disponible
      const idNow = Number(richForm.querySelector('input[name="id"]')?.value || 0) || 0;
      if (!idNow && (!state || !state.notifDraft)) {
        const { draft, actions } = loadDraftFromStorage();
        if (draft) {
          applyNotifDraft(richForm, draft);
          if (Array.isArray(actions) && actions.length) setState({ notifActions: actions });
        }
      }
    } catch (_) {}
    // Palette actions — bindings
    const actSel = richForm.querySelector('select[name="act_action_id"]');
    if (actSel) {
      actSel.addEventListener('change', () => onSelectActionId(richForm));
      if (actSel.value) { onSelectActionId(richForm).catch(console.error); }
    }
    const addAct = richForm.querySelector('[data-action="add-action-ref"]');
    if (addAct) addAct.addEventListener('click', (e) => { e.preventDefault(); addActionRef(richForm); });
    richForm.querySelectorAll('[data-action="remove-action-ref"]').forEach(btn => btn.addEventListener('click', (e) => { e.preventDefault(); removeActionRef(Number(btn.getAttribute('data-index'))); }));
    const loadTplBtn = richForm.querySelector('[data-action="load-template"]');
  if (loadTplBtn) {
      loadTplBtn.addEventListener('click', () => {
        const select = richForm.querySelector('select[name="template_id"]');
        const id = select?.value || '';
        if (!id) return;
        loadTemplateById(id).catch(err => { console.error(err); showToast('Impossible de charger le gabarit', { kind: 'error' }); });
      });
    }
    // Toggle action sections
    const btnPal = richForm.querySelector('[data-action="toggle-act-palette"]');
    if (btnPal) btnPal.addEventListener('click', (e) => { e.preventDefault(); toggleSection(richForm, 'actions-palette'); });
    const btnJson = richForm.querySelector('[data-action="toggle-act-json"]');
    if (btnJson) btnJson.addEventListener('click', (e) => { e.preventDefault(); toggleSection(richForm, 'actions-json'); });
    const closePal = richForm.querySelector('[data-action="close-act-palette"]');
    if (closePal) closePal.addEventListener('click', (e) => { e.preventDefault(); showSection(richForm, 'actions-palette', false); });
    const closeJson = richForm.querySelector('[data-action="close-act-json"]');
    if (closeJson) closeJson.addEventListener('click', (e) => { e.preventDefault(); showSection(richForm, 'actions-json', false); });
    // Auto-ouvrir la palette si des actions existent
    try { if (Array.isArray(state.notifActions) && state.notifActions.length) showSection(richForm, 'actions-palette', true); } catch (_) {}
    // Persister le brouillon au fil de la saisie
    richForm.addEventListener('input', () => {
      try {
        const d = captureNotifDraft(richForm);
        const list = Array.isArray(state.notifActions) ? state.notifActions : [];
        saveDraftToStorage(d, list);
      } catch (_) {}
    });
  }
  // ——— helpers: show/hide action sections ———
  function showSection(scope, key, show = true) {
    const sel = `[data-section="${key}"]`;
    const el = scope.querySelector(sel);
    if (el) el.style.display = show ? 'block' : 'none';
  }
  function toggleSection(scope, key) {
    const sel = `[data-section="${key}"]`;
    const el = scope.querySelector(sel);
    if (!el) return;
    const isHidden = (getComputedStyle(el).display === 'none') || el.style.display === 'none' || !el.style.display;
    // Close the other toolbox sections to avoid clutter
    ['actions-palette','actions-json'].filter(k => k !== key).forEach(k => showSection(scope, k, false));
    showSection(scope, key, isHidden);
  }
  app.querySelectorAll('[data-action="select-category"]').forEach(btn => btn.addEventListener('click', (e) => { e.stopPropagation?.(); selectCategory(Number(btn.dataset.id)); }));
  app.querySelectorAll('[data-action="delete-category"]').forEach(btn => btn.addEventListener('click', (e) => { e.stopPropagation?.(); deleteCategory(Number(btn.dataset.id)); }));
  app.querySelectorAll('[data-action="edit-category-open"]').forEach(btn => btn.addEventListener('click', (e) => { e.stopPropagation?.(); openEditCategory(Number(btn.dataset.id)); }));
  const closeEditBtn = app.querySelector('[data-action="close-edit-category"]');
  if (closeEditBtn) closeEditBtn.addEventListener('click', () => setState({ showEditCategory: false }));
  app.querySelectorAll('[data-action="edit-notification-rich"]').forEach(btn => btn.addEventListener('click', () => editNotificationRich(Number(btn.dataset.id))));
  app.querySelectorAll('[data-action="delete-notification-rich"]').forEach(btn => btn.addEventListener('click', () => deleteNotificationRich(Number(btn.dataset.id))));

  // Actions catalogue bindings
  const actionCreate = app.querySelector('[data-form="action-create"]');
  if (actionCreate) {
    actionCreate.addEventListener('submit', handleActionCreate);
    // Auto-suggest action_id from kind when empty
    const kindSel = actionCreate.querySelector('select[name="kind"]');
    const idInput = actionCreate.querySelector('input[name="action_id"]');
    if (kindSel && idInput) {
      // Prefill on load for current kind if empty
      if (!idInput.value?.trim()) {
        try { idInput.value = suggestActionId(kindSel.value); } catch (_) {}
      }
      // Track manual edits to avoid overriding user's input
      idInput.addEventListener('input', () => { idInput.dataset.dirty = '1'; });
      kindSel.addEventListener('change', () => {
        if (idInput.dataset.dirty !== '1') {
          idInput.value = suggestActionId(kindSel.value);
        }
      });
    }
  }
  // Delete entire action (guarded)
  app.querySelectorAll('[data-action="delete-action"]').forEach(btn => btn.addEventListener('click', async () => {
    const id = btn.getAttribute('data-id') || '';
    if (!id) return;
    if (!(await confirmDialog({ title: `Supprimer l'action`, message: `Supprimer l'action ${id} ?`, okLabel: 'Supprimer', cancelLabel: 'Annuler' }))) return;
    try {
      await requestJSON('/api/admin/actions/catalog/delete', { method: 'POST', body: { action_id: id } });
      showToast('Action supprimée', { kind: 'success' });
      // Clear details panel and refresh list
      setState({ selectedActionId: null, actionDetails: null });
      await loadView('actions');
    } catch (e) {
      const code = e?.payload?.code || e?.payload?.error || e?.message || '';
      if (String(code) === 'ACTION_IN_USE') showToast('Action référencée par une notification', { kind: 'warning' }); else showToast('Suppression impossible', { kind: 'error' });
    }
  }));
  const actionVersionCreate = app.querySelector('[data-form="action-version-create"]');
  if (actionVersionCreate) {
    actionVersionCreate.addEventListener('submit', handleActionVersionCreate);
    const sel = actionVersionCreate.querySelector('select[name="action_id"]');
    if (sel) {
      sel.addEventListener('change', () => { prefillActionVersionForm(actionVersionCreate).catch(console.error); });
      if (sel.value) { prefillActionVersionForm(actionVersionCreate).catch(console.error); }
    }
    // Toggle mode buttons
    const btnGuided = actionVersionCreate.querySelector('[data-version-mode="guided"]');
    const btnJson = actionVersionCreate.querySelector('[data-version-mode="json"]');
    const jsonArea = actionVersionCreate.querySelector('textarea[name="definition"]');
    const guidedSection = actionVersionCreate.querySelector('[data-section="guided-def"]');
    function setVersionMode(mode) {
      const guided = mode === 'guided';
      if (guidedSection) guidedSection.style.display = guided ? 'block' : 'none';
      if (jsonArea) jsonArea.parentElement.style.display = guided ? 'none' : 'block';
      if (btnGuided) btnGuided.setAttribute('aria-pressed', guided ? 'true' : 'false');
      if (btnJson) btnJson.setAttribute('aria-pressed', guided ? 'false' : 'true');
      actionVersionCreate.dataset.versionMode = mode;
      if (guided) updateGuidedPreview(actionVersionCreate).catch(console.error);
    }
    if (btnGuided) btnGuided.addEventListener('click', () => setVersionMode('guided'));
    if (btnJson) btnJson.addEventListener('click', () => setVersionMode('json'));
    setVersionMode('guided');
  }
  // Actions list interactions (select action, update version status)
  app.querySelectorAll('tr[data-action="select-action"]').forEach(row => row.addEventListener('click', async () => {
    const id = row.getAttribute('data-id');
    if (!id) return;
    try {
      const details = await fetchActionDetails(id);
      setState({ selectedActionId: id, actionDetails: details });
    } catch (e) { console.error(e); }
  }));
  app.querySelectorAll('[data-action="apply-version-status"]').forEach(btn => btn.addEventListener('click', async (e) => {
    e.preventDefault();
    const actionId = btn.getAttribute('data-action-id');
    const version = Number(btn.getAttribute('data-version')||0);
    const select = btn.closest('tr')?.querySelector('select[data-action="set-version-status"]');
    const status = select?.value || '';
    if (!actionId || !Number.isFinite(version) || !status) return;
    try {
      await updateActionVersionStatus(actionId, version, status);
      showToast('Statut mis à jour', { kind: 'success' });
      const details = await fetchActionDetails(actionId);
      setState({ actionDetails: details });
    } catch (e) { console.error(e); }
  }));

  const userForm = app.querySelector('[data-form="user"]');
  if (userForm) userForm.addEventListener('submit', handleUserCreate);
  app.querySelectorAll('[data-action="edit-user"]').forEach(btn => btn.addEventListener('click', () => editUser(Number(btn.dataset.id))));
  app.querySelectorAll('[data-action="view-user-data"]').forEach(btn => btn.addEventListener('click', () => openUserData(Number(btn.dataset.id))));
  app.querySelectorAll('[data-action="delete-user"]').forEach(btn => btn.addEventListener('click', () => deleteUser(Number(btn.dataset.id))));

  const licenseForm = app.querySelector('[data-form="license"]');
  if (licenseForm) licenseForm.addEventListener('submit', handleLicenseCreate);
  app.querySelectorAll('[data-action="assign-license"]').forEach(btn => btn.addEventListener('click', () => assignLicense(Number(btn.dataset.id))));
  app.querySelectorAll('[data-action="unassign-license"]').forEach(btn => btn.addEventListener('click', () => unassignLicense(Number(btn.dataset.id))));
  app.querySelectorAll('[data-action="link-license"]').forEach(btn => btn.addEventListener('click', () => linkLicense(Number(btn.dataset.id))));
  app.querySelectorAll('[data-action="delete-license"]').forEach(btn => btn.addEventListener('click', () => deleteLicense(Number(btn.dataset.id))));

  app.querySelectorAll('[data-action="clear-ip"]').forEach(btn => btn.addEventListener('click', () => clearIp(btn.dataset.ip)));
  const clearAllBtn = app.querySelector('[data-action="clear-all-ip"]');
  if (clearAllBtn) clearAllBtn.addEventListener('click', clearAllIps);
  const resetSchemaBtn = app.querySelector('[data-action="reset-schema"]');
  if (resetSchemaBtn) resetSchemaBtn.addEventListener('click', resetSchema);

  const backUsers = app.querySelector('[data-action="back-users"]');
  if (backUsers) backUsers.addEventListener('click', async () => { await changeView('users'); });

  // Delete action version (admin)
  app.querySelectorAll('[data-action="delete-action-version"]').forEach(btn => btn.addEventListener('click', async (e) => {
    e.preventDefault();
    const actionId = btn.getAttribute('data-action-id');
    const version = Number(btn.getAttribute('data-version')||0);
    if (!actionId || !Number.isFinite(version)) return;
    const ok = await confirmDialog({ title: 'Supprimer la version', message: `Supprimer la version ${version} de ${actionId} ?`, okLabel: 'Supprimer', cancelLabel: 'Annuler' });
    if (!ok) return;
    try {
      await deleteActionVersion(actionId, version);
      showToast('Version supprimée', { kind: 'success' });
      const details = await fetchActionDetails(actionId);
      setState({ actionDetails: details });
    } catch (e) { console.error(e); }
  }));
}

async function changeView(view) {
  if (state.view === view && !state.loading) {
    await loadView(view);
    return;
  }
  setState({ view });
  await loadView(view);
}

async function loadView(view) {
  setState({ loading: true });
  try {
    const updates = {};
    switch (view) {
      case 'notifications':
        updates.notifCategories = await fetchNotificationCategories();
        updates.actionsCatalog = await fetchActionsCatalog();
        try {
          updates.notifTemplates = await fetchNotificationTemplates();
        } catch (_) {
          updates.notifTemplates = [];
        }
        if (!updates.notifCategories.length) {
          updates.notifCatSelected = null;
          updates.notifRich = [];
        } else {
          const current = Number(state.notifCatSelected ?? 0) || updates.notifCategories[0].id;
          updates.notifCatSelected = current;
          updates.notifRich = await fetchRichNotifications(current);
        }
        break;
      case 'actions':
        updates.actionsCatalog = await fetchActionsCatalog();
        // Also load categories to offer guided selects for categoryId
        try { updates.notifCategories = await fetchNotificationCategories(); } catch (_) { updates.notifCategories = []; }
        break;
      case 'users':
        updates.users = await fetchUsers();
        updates.userDetails = null;
        break;
      case 'user-details':
        // keep current userDetails as is; it is set by openUserData
        break;
      case 'licenses':
        updates.licenses = await fetchLicenses();
        break;
      case 'security':
        updates.security = await fetchSecurity();
        break;
      default:
        break;
    }
    setState({ loading: false, ...updates });
  } catch (error) {
    console.error(error);
    showToast('Impossible de charger les données', { kind: 'error' });
    setState({ loading: false });
  }
}

async function fetchNotificationCategories() {
  const payload = await requestJSON('/api/admin/notification-categories');
  return payload.categories ?? [];
}

async function fetchRichNotifications(categoryId) {
  const q = Number.isFinite(categoryId) && categoryId > 0 ? `?categoryId=${encodeURIComponent(categoryId)}` : '';
  const payload = await requestJSON('/api/admin/notifications/rich' + q);
  return payload.notifications ?? [];
}

async function fetchUsers() {
  const payload = await requestJSON('/api/admin/users');
  return payload.users ?? [];
}

async function fetchUserData(userId) {
  const payload = await requestJSON(`/api/admin/users/${encodeURIComponent(userId)}/data`);
  return payload;
}

async function fetchLicenses() {
  const payload = await requestJSON('/api/admin/licenses');
  return payload.licenses ?? [];
}

async function handleActionCreate(event) {
  event.preventDefault();
  const form = event.currentTarget;
  const data = new FormData(form);
  const body = {
    action_id: data.get('action_id')?.toString().trim(),
    kind: data.get('kind')?.toString().trim(),
  };
  if (!body.action_id && body.kind) {
    // Auto-suggest a stable, readable id if missing
    body.action_id = suggestActionId(body.kind);
    try { const inp = form.querySelector('input[name="action_id"]'); if (inp) inp.value = body.action_id; } catch (_) {}
  }
  if (!body.action_id || !body.kind) { showToast('action_id et kind requis', { kind:'warning' }); return; }
  await requestJSON('/api/admin/actions/catalog', { method: 'POST', body });
  showToast('Action créée', { kind: 'success' });
  form.reset();
  await loadView('actions');
}

async function handleActionVersionCreate(event) {
  event.preventDefault();
  const form = event.currentTarget;
  const data = new FormData(form);
  const action_id = data.get('action_id')?.toString().trim();
  const version = Number(data.get('version') ?? 0);
  const status = data.get('status')?.toString().trim() || 'draft';
  const definitionRaw = data.get('definition')?.toString() || '';
  if (!action_id || !Number.isFinite(version) || version <= 0) { showToast('action_id + version requis', { kind:'warning' }); return; }
  let definition = {};
  if (definitionRaw) {
    try { definition = JSON.parse(definitionRaw); }
    catch (e) { showToast('definition JSON invalide', { kind:'warning' }); return; }
  }
  await requestJSON('/api/admin/actions/catalog/version', { method: 'POST', body: { action_id, version, status, definition } });
  showToast('Version créée', { kind: 'success' });
  form.reset();
  await loadView('actions');
}

function suggestActionId(kind) {
  const now = new Date();
  const pad2 = (n) => String(n).padStart(2, '0');
  const ts = `${now.getFullYear()}${pad2(now.getMonth()+1)}${pad2(now.getDate())}_${pad2(now.getHours())}${pad2(now.getMinutes())}${pad2(now.getSeconds())}`;
  let base = String(kind || 'action').toLowerCase();
  base = base.replace(/[^a-z0-9_.:-]+/g, '.').replace(/\.{2,}/g, '.').replace(/^\.|\.$/g, '');
  let id = `${base}.${ts}`;
  if (id.length < 2) id = `act.${ts}`;
  if (id.length > 64) id = id.slice(0, 64);
  return id;
}

async function prefillActionVersionForm(form) {
  try {
    const sel = form.querySelector('select[name="action_id"]');
    const verInput = form.querySelector('input[name="version"]');
    const defTa = form.querySelector('textarea[name="definition"]');
    const actionId = sel?.value || '';
    if (!actionId) return;
    const payload = await requestJSON(`/api/admin/actions/catalog/${encodeURIComponent(actionId)}`, { method: 'GET' });
    const action = payload?.action || {};
    const versions = Array.isArray(action.versions) ? action.versions : [];
    const maxVer = versions.reduce((m, v) => Math.max(m, Number(v?.version || 0)), 0);
    if (verInput) verInput.value = String(maxVer + 1 || 1);
    const kind = String(action.kind || '');
    form.dataset.actionKind = kind;
    if (defTa && !defTa.value.trim()) {
      const def = defaultDefinitionForKind(kind);
      defTa.value = JSON.stringify(def, null, 2);
    }
    // Build guided fields for this kind
    buildGuidedFieldsForKind(form, kind);
    // Initialize preview from guided
    await updateGuidedPreview(form);
  } catch (e) {
    console.error(e);
  }
}

function defaultDefinitionForKind(kind) {
  switch (String(kind || '')) {
    case 'UserData.SetKey':
      return {
        label: { fr: 'Enregistrer préférence' },
        payloadSchema: {
          type: 'object',
          additionalProperties: false,
          required: ['namespace', 'key', 'value'],
          properties: {
            namespace: { const: 'prefs' },
            key: { type: 'string', maxLength: 128 },
            value: {},
          }
        },
        defaultParams: { namespace: 'prefs' },
        capabilities: [],
        auditTag: 'user_pref_set'
      };
    case 'User.SubscribeCategory':
      return {
        label: { fr: 'S’abonner' },
        payloadSchema: {
          type: 'object',
          additionalProperties: false,
          required: ['categoryId'],
          properties: {
            categoryId: { type: 'integer', minimum: 1 },
          }
        },
        defaultParams: {},
        capabilities: [],
        auditTag: 'user_subscribe'
      };
    case 'User.UnsubscribeCategory':
      return {
        label: { fr: 'Se désabonner' },
        payloadSchema: {
          type: 'object',
          additionalProperties: false,
          required: ['categoryId'],
          properties: {
            categoryId: { type: 'integer', minimum: 1 },
          }
        },
        defaultParams: {},
        capabilities: [],
        auditTag: 'user_unsubscribe'
      };
    case 'User.SetCategoryFrequencyOverride':
      return {
        label: { fr: 'Modifier fréquence' },
        payloadSchema: {
          type: 'object',
          additionalProperties: false,
          required: ['categoryId', 'kind', 'param'],
          properties: {
            categoryId: { type: 'integer', minimum: 1 },
            kind: { type: 'string', enum: ['EVERY_N_DAYS','WEEKLY','MONTHLY'] },
            param: { type: 'integer', minimum: 1 },
          }
        },
        defaultParams: {},
        capabilities: [],
        auditTag: 'user_freq_override'
      };
    case 'OpenDialog':
      return {
        label: { fr: 'Ouvrir un dialogue' },
        payloadSchema: {
          type: 'object',
          additionalProperties: false,
          required: ['schemaRef'],
          properties: {
            schemaRef: { type: 'string', minLength: 1 },
          }
        },
        defaultParams: {},
        capabilities: [],
        auditTag: 'open_dialog'
      };
    case 'User.SubscribeAndSetPref':
      return {
        label: { fr: 'J’aime…' },
        payloadSchema: {
          type: 'object',
          additionalProperties: false,
          required: ['categoryId', 'key', 'value'],
          properties: {
            categoryId: { type: 'integer', minimum: 1 },
            namespace: { const: 'prefs' },
            key: { type: 'string' },
            value: {},
          }
        },
        defaultParams: { namespace: 'prefs' },
        capabilities: [],
        auditTag: 'like_and_subscribe'
      };
    default:
      return {
        label: { fr: 'Action' },
        payloadSchema: { type: 'object', additionalProperties: true },
        defaultParams: {},
        capabilities: [],
        auditTag: 'action'
      };
  }
}

function renderParamsForm(form, schema, defaults = {}) {
  const host = form.querySelector('[data-act-params-form]');
  const ta = form.querySelector('textarea[name="act_params_json"]');
  if (!host) return;
  const s = (schema && typeof schema === 'object') ? schema : {};
  const props = (s.properties && typeof s.properties === 'object') ? s.properties : {};
  const required = Array.isArray(s.required) ? s.required : [];
  const entries = Object.entries(props);
  if (!entries.length) { host.innerHTML = '<div class="surface-card surface-card--muted surface-card--bordered" style="padding:8px;">Aucun schéma: utilisez le JSON libre si nécessaire.</div>'; return; }
  const fieldsHtml = entries.map(([name, def]) => {
    const isReq = required.includes(name); // informative only; no HTML5 required to avoid blocking main form submit
    const val = defaults && Object.prototype.hasOwnProperty.call(defaults, name) ? defaults[name] : (def.const ?? '');
    const label = `${escapeHtml(name)}${isReq ? ' *' : ''}`;
    if (Object.prototype.hasOwnProperty.call(def, 'const')) {
      return `<label>${label}<input data-act-param name="${escapeAttr(name)}" value="${escapeAttr(String(val))}" readonly /></label>`;
    }
    const t = String(def.type || '').toLowerCase();
    if (Array.isArray(def.enum) && def.enum.length) {
      const opts = def.enum.map(opt => {
        const sel = String(opt) === String(val) ? ' selected' : '';
        return `<option value="${escapeAttr(String(opt))}"${sel}>${escapeHtml(String(opt))}</option>`;
      }).join('');
      return `<label>${label}<select data-act-param name="${escapeAttr(name)}">${opts}</select></label>`;
    }
    if (t === 'integer' || t === 'number') {
      // Special-case categoryId: prefer select of existing categories when available
      if (name === 'categoryId' && Array.isArray(state?.notifCategories) && state.notifCategories.length) {
        const opts = state.notifCategories.map(c => {
          const sel = String(c.id) === String(val) ? ' selected' : '';
          return `<option value="${escapeAttr(String(c.id))}"${sel}>${escapeHtml(String(c.slug || 'cat'))} — ${escapeHtml(String(c.name || ''))} (#${escapeHtml(String(c.id))})</option>`;
        }).join('');
        return `<label>${label}<select data-act-param name="${escapeAttr(name)}" data-type="integer">${opts}</select></label>`;
      }
      return `<label>${label}<input data-act-param type="number" name="${escapeAttr(name)}" value="${escapeAttr(String(val ?? ''))}" /></label>`;
    }
    if (t === 'boolean') {
      const checked = String(val) === 'true' || val === true ? 'checked' : '';
      return `<label>${label}<input data-act-param type="checkbox" name="${escapeAttr(name)}" ${checked} /></label>`;
    }
    // default text (string or unknown)
    return `<label>${label}<input data-act-param name="${escapeAttr(name)}" value="${escapeAttr(val != null ? String(val) : '')}" /></label>`;
  }).join('');
  host.innerHTML = `<div class="form-grid form-grid--columns">${fieldsHtml}</div>`;
  const inputs = host.querySelectorAll('[data-act-param]');
  function syncParamsJson() {
    const obj = {};
    inputs.forEach(el => {
      const n = el.getAttribute('name');
      if (!n) return;
      const dt = el.getAttribute('data-type');
      if (el.type === 'checkbox') {
        obj[n] = el.checked;
      } else if (el.type === 'number' || dt === 'integer') {
        const v = el.value;
        obj[n] = v === '' ? null : Number(v);
      } else {
        const v = el.value.trim();
        obj[n] = v;
      }
    });
    if (ta) ta.value = JSON.stringify(obj);
  }
  inputs.forEach(el => el.addEventListener('input', syncParamsJson));
  syncParamsJson();
}

function updateParamsFormFromSelectedVersion(form) {
  const select = form.querySelector('select[name="act_version"]');
  const ta = form.querySelector('textarea[name="act_params_json"]');
  const versions = select? select._versions || [] : [];
  const ver = Number(select?.value || 0);
  const vdef = versions.find(v => Number(v.version) === ver);
  const schema = vdef?.definition?.payloadSchema || {};
  const defaults = vdef?.definition?.defaultParams || {};
  renderParamsForm(form, schema, defaults);
  if (ta) ta.value = JSON.stringify(defaults || {});
}

function buildGuidedFieldsForKind(form, kind) {
  const host = form.querySelector('[data-guided-kind-fields]');
  if (!host) return;
  const v = String(kind || '');
  let html = '';
  if (v === 'UserData.SetKey') {
    html = `
      <div class="form-grid form-grid--columns">
        <label>namespace
          <input name="def_ns" value="prefs" />
        </label>
        <label>key
          <input name="def_key" placeholder="likes.history" />
        </label>
        <label>value (bool/texte/JSON)
          <input name="def_value" placeholder="true" />
        </label>
      </div>
    `;
  } else if (v === 'User.SubscribeAndSetPref') {
    const cats = Array.isArray(state?.notifCategories) ? state.notifCategories : [];
    const catField = cats.length
      ? `<label>categoryId
            <select name="def_categoryId">${cats.map(c => `<option value="${escapeAttr(String(c.id))}">${escapeHtml(String(c.slug || 'cat'))} — ${escapeHtml(String(c.name || ''))} (#${escapeHtml(String(c.id))})</option>`).join('')}</select>
         </label>`
      : `<label>categoryId
            <input name="def_categoryId" type="number" min="1" placeholder="Ex: 42" />
         </label>`;
    html = `
      <div class="form-grid form-grid--columns">
        ${catField}
        <label>namespace
          <input name="def_ns" value="prefs" />
        </label>
        <label>key
          <input name="def_key" placeholder="likes.history" />
        </label>
        <label>value (bool/texte/JSON)
          <input name="def_value" placeholder="true" />
        </label>
      </div>
    `;
  } else if (v === 'User.SubscribeCategory' || v === 'User.UnsubscribeCategory') {
    const cats = Array.isArray(state?.notifCategories) ? state.notifCategories : [];
    const catField = cats.length
      ? `<label>categoryId
            <select name="def_categoryId">${cats.map(c => `<option value="${escapeAttr(String(c.id))}">${escapeHtml(String(c.slug || 'cat'))} — ${escapeHtml(String(c.name || ''))} (#${escapeHtml(String(c.id))})</option>`).join('')}</select>
         </label>`
      : `<label>categoryId
            <input name="def_categoryId" type="number" min="1" placeholder="Ex: 42" />
         </label>`;
    html = `
      <div class="form-grid form-grid--columns">
        ${catField}
      </div>
    `;
  } else if (v === 'User.SetCategoryFrequencyOverride') {
    const cats = Array.isArray(state?.notifCategories) ? state.notifCategories : [];
    const catField = cats.length
      ? `<label>categoryId
            <select name="def_categoryId">${cats.map(c => `<option value="${escapeAttr(String(c.id))}">${escapeHtml(String(c.slug || 'cat'))} — ${escapeHtml(String(c.name || ''))} (#${escapeHtml(String(c.id))})</option>`).join('')}</select>
         </label>`
      : `<label>categoryId
            <input name="def_categoryId" type="number" min="1" placeholder="Ex: 42" />
         </label>`;
    const fkSel = `<label>kind
      <select name="def_freqKind">
        <option value="EVERY_N_DAYS">EVERY_N_DAYS</option>
        <option value="WEEKLY">WEEKLY</option>
        <option value="MONTHLY">MONTHLY</option>
      </select>
    </label>`;
    const param = `<label>param
      <input name="def_param" type="number" min="1" placeholder="Ex: 7" />
    </label>`;
    html = `
      <div class="form-grid form-grid--columns">
        ${catField}
        ${fkSel}
        ${param}
      </div>
    `;
  } else if (v === 'OpenDialog') {
    html = `
      <div class="form-grid form-grid--columns">
        <label>schemaRef
          <input name="def_schemaRef" placeholder="ex: feedback.v1" />
        </label>
      </div>
    `;
  } else {
    html = `<div class="surface-card surface-card--muted surface-card--bordered" style="padding:8px;">Aucun formulaire guidé pour ce kind. Utilisez le mode JSON.</div>`;
  }
  host.innerHTML = html;
  // Bind change events to update preview
  host.querySelectorAll('input,select,textarea').forEach(el => el.addEventListener('input', () => updateGuidedPreview(form).catch(console.error)));
}

async function updateGuidedPreview(form) {
  if (!form || form.dataset.versionMode !== 'guided') return;
  const kind = form.dataset.actionKind || '';
  const preview = form.querySelector('[data-guided-preview]');
  const txt = form.querySelector('textarea[name="definition"]');
  const labelFr = form.querySelector('input[name="def_label_fr"]')?.value?.trim() || '';
  const auditTag = form.querySelector('input[name="def_audit_tag"]')?.value?.trim() || '';
  const guided = readGuidedInputs(form);
  const def = materializeDefinition(kind, labelFr, auditTag, guided);
  const json = JSON.stringify(def, null, 2);
  if (preview) preview.textContent = json;
  if (txt) txt.value = json;
}

function readGuidedInputs(form) {
  const ns = form.querySelector('input[name="def_ns"]')?.value?.trim();
  const key = form.querySelector('input[name="def_key"]')?.value?.trim();
  const valRaw = form.querySelector('input[name="def_value"]')?.value?.trim();
  const catId = form.querySelector('input[name="def_categoryId"]')?.value?.trim();
  const catSel = form.querySelector('select[name="def_categoryId"]')?.value?.trim();
  const freqKind = form.querySelector('select[name="def_freqKind"]')?.value?.trim();
  const param = form.querySelector('input[name="def_param"]')?.value?.trim();
  const schemaRef = form.querySelector('input[name="def_schemaRef"]')?.value?.trim();
  return { ns, key, valRaw, catId: catSel || catId, freqKind, param, schemaRef };
}

function materializeDefinition(kind, labelFr, auditTag, guided) {
  const base = { label: {}, payloadSchema: { type: 'object', additionalProperties: false, required: [], properties: {} }, defaultParams: {}, capabilities: [] };
  if (labelFr) base.label.fr = labelFr; else base.label.fr = 'Action';
  if (auditTag) base.auditTag = auditTag;
  const v = String(kind || '');
  if (v === 'UserData.SetKey') {
    base.payloadSchema.required = ['namespace','key','value'];
    base.payloadSchema.properties = {
      namespace: { const: 'prefs' },
      key: { type: 'string', maxLength: 128 },
      value: {},
    };
    base.defaultParams = { namespace: guided.ns || 'prefs' };
  } else if (v === 'User.SubscribeAndSetPref') {
    base.payloadSchema.required = ['categoryId','key','value'];
    base.payloadSchema.properties = {
      categoryId: { type: 'integer', minimum: 1 },
      namespace: { const: 'prefs' },
      key: { type: 'string' },
      value: {},
    };
    const cat = Number(guided.catId || 0);
    base.defaultParams = { namespace: guided.ns || 'prefs' };
    if (Number.isFinite(cat) && cat > 0) base.defaultParams.categoryId = cat;
    if (guided.key) base.defaultParams.key = guided.key;
    if (guided.valRaw) base.defaultParams.value = parseLooseJson(guided.valRaw);
    return base;
  } else if (v === 'User.SubscribeCategory' || v === 'User.UnsubscribeCategory') {
    base.payloadSchema.required = ['categoryId'];
    base.payloadSchema.properties = { categoryId: { type: 'integer', minimum: 1 } };
    const cat = Number(guided.catId || 0);
    if (Number.isFinite(cat) && cat > 0) base.defaultParams.categoryId = cat;
    return base;
  } else if (v === 'User.SetCategoryFrequencyOverride') {
    base.payloadSchema.required = ['categoryId','kind','param'];
    base.payloadSchema.properties = {
      categoryId: { type: 'integer', minimum: 1 },
      kind: { type: 'string', enum: ['EVERY_N_DAYS','WEEKLY','MONTHLY'] },
      param: { type: 'integer', minimum: 1 },
    };
    const cat = Number(guided.catId || 0);
    if (Number.isFinite(cat) && cat > 0) base.defaultParams.categoryId = cat;
    const k = String(guided.freqKind || '').toUpperCase();
    if (k) base.defaultParams.kind = k;
    const p = Number(guided.param || 0);
    if (Number.isFinite(p) && p > 0) base.defaultParams.param = p;
    return base;
  } else if (v === 'OpenDialog') {
    base.payloadSchema.required = ['schemaRef'];
    base.payloadSchema.properties = { schemaRef: { type: 'string', minLength: 1 } };
    if (guided.schemaRef) base.defaultParams.schemaRef = guided.schemaRef;
    return base;
  }
  if (guided.key) base.defaultParams.key = guided.key;
  if (guided.valRaw) base.defaultParams.value = parseLooseJson(guided.valRaw);
  return base;
}

function parseLooseJson(input) {
  const s = String(input || '').trim();
  if (s === '') return '';
  if (s === 'true') return true;
  if (s === 'false') return false;
  if (!Number.isNaN(Number(s)) && /^-?\d+(?:\.\d+)?$/.test(s)) return Number(s);
  try { return JSON.parse(s); } catch (_) { return s; }
}

// --- Helpers: actions <-> shortcodes consistency + local draft persistence ---
function extractActionIdsFromHtml(html) {
  const ids = new Set();
  try {
    const re = /\[sb:action\s+id=\"([^\"]+)\"\]([\s\S]*?)\[\/sb:action\]/g;
    let m;
    while ((m = re.exec(String(html || ''))) !== null) {
      if (m[1]) ids.add(String(m[1]));
    }
  } catch (_) {}
  return Array.from(ids);
}

const DRAFT_LS_KEY = 'sb.admin.notifrich.draft.v1';
const ACTIONS_LS_KEY = 'sb.admin.notifrich.actions.v1';
function saveDraftToStorage(draft, actions) {
  try {
    if (draft) localStorage.setItem(DRAFT_LS_KEY, JSON.stringify(draft));
    if (Array.isArray(actions)) localStorage.setItem(ACTIONS_LS_KEY, JSON.stringify(actions));
  } catch (_) {}
}
function loadDraftFromStorage() {
  try {
    const raw = localStorage.getItem(DRAFT_LS_KEY);
    const rawA = localStorage.getItem(ACTIONS_LS_KEY);
    const draft = raw ? JSON.parse(raw) : null;
    const actions = rawA ? JSON.parse(rawA) : null;
    return { draft, actions: Array.isArray(actions) ? actions : [] };
  } catch (_) { return { draft: null, actions: [] }; }
}
function clearDraftStorage() {
  try { localStorage.removeItem(DRAFT_LS_KEY); localStorage.removeItem(ACTIONS_LS_KEY); } catch (_) {}
}

// --- Helpers: préservation du brouillon de formulaire (notification riche) ---
function captureNotifDraft(form) {
  if (!form) return null;
  const getVal = (sel) => { const el = form.querySelector(sel); return el ? el.value : ''; };
  const getNum = (sel) => { const v = Number(getVal(sel)); return Number.isFinite(v) ? v : 0; };
  const isChecked = (sel) => { const el = form.querySelector(sel); return !!(el && el.checked); };
  return {
    id: getNum('input[name="id"]') || undefined,
    category_id: getNum('input[name="category_id"]'),
    title: getVal('input[name="title"]').toString(),
    emitter: getVal('input[name="emitter"]').toString(),
    weight: getNum('input[name="weight"]'),
    sequence_index: getNum('input[name="sequence_index"]') || undefined,
    active: isChecked('input[name="active"]'),
    content_html: getVal('textarea[name="content_html"]'),
    content_css: getVal('textarea[name="content_css"]'),
    actions_ref_json: getVal('textarea[name="actions_ref_json"]'),
    actions_snapshot_json: getVal('textarea[name="actions_snapshot_json"]'),
  };
}

function applyNotifDraft(form, draft) {
  if (!form || !draft) return;
  const setVal = (sel, v) => { const el = form.querySelector(sel); if (el) el.value = v ?? ''; };
  const setChecked = (sel, v) => { const el = form.querySelector(sel); if (el) el.checked = !!v; };
  try {
    if (draft.id) setVal('input[name="id"]', draft.id);
    setVal('input[name="category_id"]', draft.category_id ?? '');
    setVal('input[name="title"]', draft.title ?? '');
    setVal('input[name="emitter"]', draft.emitter ?? '');
    setVal('input[name="weight"]', String(draft.weight ?? 0));
    if (typeof draft.sequence_index !== 'undefined') setVal('input[name="sequence_index"]', String(draft.sequence_index));
    setChecked('input[name="active"]', !!draft.active);
    setVal('textarea[name="content_html"]', draft.content_html ?? '');
    setVal('textarea[name="content_css"]', draft.content_css ?? '');
    if (typeof draft.actions_ref_json === 'string') setVal('textarea[name="actions_ref_json"]', draft.actions_ref_json);
    if (typeof draft.actions_snapshot_json === 'string') setVal('textarea[name="actions_snapshot_json"]', draft.actions_snapshot_json);
    // Déclencher un rafraîchissement de l'aperçu
    const htmlTa = form.querySelector('textarea[name="content_html"]');
    if (htmlTa) {
      const ev = new Event('input', { bubbles: true });
      htmlTa.dispatchEvent(ev);
    }
    // Mettre le bouton en mode Enregistrer si on édite
    const submitBtn = form.querySelector('[data-role="notifrich-submit"]');
    if (submitBtn && draft.id) submitBtn.textContent = 'Enregistrer';
  } catch (_) {}
}

async function fetchSecurity() {
  const payload = await requestJSON('/api/admin/security/ip');
  return { ips: payload.ips ?? [] };
}

async function fetchActionsCatalog() {
  const payload = await requestJSON('/api/admin/actions/catalog');
  return payload.actions ?? [];
}

async function fetchActionDetails(actionId) {
  const payload = await requestJSON(`/api/admin/actions/catalog/${encodeURIComponent(actionId)}`, { method: 'GET' });
  return payload.action ?? null;
}

async function updateActionVersionStatus(action_id, version, status) {
  return await requestJSON('/api/admin/actions/catalog/version/status', { method: 'POST', body: { action_id, version, status } });
}

async function deleteActionVersion(action_id, version) {
  return await requestJSON('/api/admin/actions/catalog/version/delete', { method: 'POST', body: { action_id, version } });
}

async function fetchNotificationTemplates() {
  const res = await fetch('/assets/notifications/templates/manifest.json', { credentials: 'include', headers: { 'Accept': 'application/json' } });
  if (!res.ok) throw new Error('TEMPLATES_MANIFEST_HTTP_' + res.status);
  const ct = (res.headers.get('Content-Type') || res.headers.get('content-type') || '').toLowerCase();
  if (!ct.includes('application/json')) throw new Error('TEMPLATES_MANIFEST_CT');
  const json = await res.json();
  const list = Array.isArray(json?.templates) ? json.templates : [];
  return list.filter(t => t && typeof t === 'object' && typeof t.id === 'string');
}

async function handleCategoryCreate(event) {
  event.preventDefault();
  const form = event.currentTarget;
  const data = new FormData(form);
  const body = {
    slug: data.get('slug')?.toString().trim(),
    name: data.get('name')?.toString().trim(),
    sort_order: Number(data.get('sort_order') ?? 0),
    active: data.get('active') === 'on',
    audience_mode: (data.get('audience_mode')?.toString().trim() || 'EVERYONE').toUpperCase(),
    dispatch_mode: (data.get('dispatch_mode')?.toString().trim() || 'BROADCAST').toUpperCase(),
    frequency_kind: (data.get('frequency_kind')?.toString().trim() || 'IMMEDIATE').toUpperCase(),
    frequency_param: (data.get('frequency_param')?.toString().trim() ?? ''),
    anchor_ts: (data.get('anchor_ts')?.toString().trim() ?? ''),
    allow_user_override: data.get('allow_user_override') === 'on',
  };
  if (!body.slug || !body.name) {
    showToast('Slug et nom requis.', { kind: 'warning' });
    return;
  }
  // Normalisation + validations strictes
  if (body.dispatch_mode === 'PERSONALIZED' && body.frequency_kind === 'IMMEDIATE') {
    showToast('PERSONALIZED × IMMEDIATE interdit.', { kind: 'warning' });
    return;
  }
  // frequency_param: convertir en nombre selon le kind
  const fpRaw = body.frequency_param;
  let fp = null;
  if (body.frequency_kind === 'EVERY_N_DAYS') {
    const v = Number(fpRaw);
    if (![1,2,3,4,5,6,7,14,21].includes(v)) {
      showToast('Paramètre EVERY_N_DAYS invalide (attendu: 1..7,14,21).', { kind: 'warning' });
      return;
    }
    fp = v;
  } else if (body.frequency_kind === 'WEEKLY') {
    const v = Number(fpRaw);
    if (!Number.isInteger(v) || v < 1 || v > 7) {
      showToast('Paramètre WEEKLY invalide (1..7).', { kind: 'warning' });
      return;
    }
    fp = v;
  } else if (body.frequency_kind === 'MONTHLY') {
    const v = Number(fpRaw);
    if (!Number.isInteger(v) || v < 1 || v > 28) {
      showToast('Paramètre MONTHLY invalide (1..28).', { kind: 'warning' });
      return;
    }
    fp = v;
  }
  body.frequency_param = fp;
  // anchor_ts optionnel → nombre ou null
  const anchor = Number(body.anchor_ts);
  body.anchor_ts = Number.isFinite(anchor) && anchor > 0 ? anchor : null;
  await requestJSON('/api/admin/notification-categories', { method: 'POST', body });
  showToast('Catégorie créée', { kind: 'success' });
  form.reset();
  setState({ showCreateCategory: false });
  await loadView('notifications');
}

async function handleCategoryUpdate(event) {
  event.preventDefault();
  const form = event.currentTarget;
  const data = new FormData(form);
  const id = Number(data.get('id') ?? 0);
  if (!Number.isFinite(id) || id <= 0) {
    showToast('Catégorie invalide', { kind: 'warning' });
    return;
  }
  const body = {
    id,
    slug: data.get('slug')?.toString().trim(),
    name: data.get('name')?.toString().trim(),
    sort_order: Number(data.get('sort_order') ?? 0),
    active: data.get('active') === 'on',
    audience_mode: (data.get('audience_mode')?.toString().trim() || 'EVERYONE').toUpperCase(),
    dispatch_mode: (data.get('dispatch_mode')?.toString().trim() || 'BROADCAST').toUpperCase(),
    frequency_kind: (data.get('frequency_kind')?.toString().trim() || 'IMMEDIATE').toUpperCase(),
    frequency_param: (data.get('frequency_param')?.toString().trim() ?? ''),
    anchor_ts: (data.get('anchor_ts')?.toString().trim() ?? ''),
    allow_user_override: data.get('allow_user_override') === 'on',
  };
  if (!body.slug || !body.name) {
    showToast('Slug et nom requis.', { kind: 'warning' });
    return;
  }
  if (body.dispatch_mode === 'PERSONALIZED' && body.frequency_kind === 'IMMEDIATE') {
    showToast('PERSONALIZED × IMMEDIATE interdit.', { kind: 'warning' });
    return;
  }
  const fpRaw = body.frequency_param;
  let fp = null;
  if (body.frequency_kind === 'EVERY_N_DAYS') {
    const v = Number(fpRaw);
    if (![1,2,3,4,5,6,7,14,21].includes(v)) {
      showToast('Paramètre EVERY_N_DAYS invalide (attendu: 1..7,14,21).', { kind: 'warning' });
      return;
    }
    fp = v;
  } else if (body.frequency_kind === 'WEEKLY') {
    const v = Number(fpRaw);
    if (!Number.isInteger(v) || v < 1 || v > 7) {
      showToast('Paramètre WEEKLY invalide (1..7).', { kind: 'warning' });
      return;
    }
    fp = v;
  } else if (body.frequency_kind === 'MONTHLY') {
    const v = Number(fpRaw);
    if (!Number.isInteger(v) || v < 1 || v > 28) {
      showToast('Paramètre MONTHLY invalide (1..28).', { kind: 'warning' });
      return;
    }
    fp = v;
  }
  body.frequency_param = fp;
  const anchor = Number(body.anchor_ts);
  body.anchor_ts = Number.isFinite(anchor) && anchor > 0 ? anchor : null;
  await requestJSON('/api/admin/notification-categories/update', { method: 'POST', body });
  showToast('Catégorie mise à jour', { kind: 'success' });
  setState({ showEditCategory: false });
  await loadView('notifications');
}

async function handleNotificationRichSubmit(event) {
  event.preventDefault();
  const form = event.currentTarget;
  const data = new FormData(form);
  const body = {
    id: Number(data.get('id') ?? 0) || undefined,
    category_id: Number(data.get('category_id') ?? 0),
    title: data.get('title')?.toString().trim(),
    emitter: data.get('emitter')?.toString().trim() || null,
    weight: Number(data.get('weight') ?? 0),
    sequence_index: (Number(data.get('sequence_index') ?? 0) || undefined),
    active: data.get('active') === 'on',
    content_html: data.get('content_html')?.toString(),
    content_css: data.get('content_css')?.toString() || '',
  };
  // Optional actions (JSON)
  const refJson = (data.get('actions_ref_json')?.toString() || '').trim();
  const snapJson = (data.get('actions_snapshot_json')?.toString() || '').trim();
  if (Array.isArray(state.notifActions) && state.notifActions.length) {
    body.actions_ref_json = { actions: state.notifActions.map(a => ({ ref: a.ref, version: a.version, params: a.params || {} })) };
    body.actions_snapshot_json = { version: 1, policy: 'snapshot', actions: state.notifActions.map(a => ({ id: a.id, ref: a.ref, version: a.version, label: a.label || {}, params: a.params || {}, versionEtag: a.versionEtag || '' })) };
  } else if (refJson) {
    try { body.actions_ref_json = JSON.parse(refJson); } catch (e) { showToast('actions_ref_json invalide (JSON)', { kind: 'warning' }); return; }
  }
  if (snapJson) {
    try { body.actions_snapshot_json = JSON.parse(snapJson); } catch (e) { showToast('actions_snapshot_json invalide (JSON)', { kind: 'warning' }); return; }
  }
  // Edition: si aucune action fournie (ni palette, ni JSON), expliciter un manifeste vide
  if (body.id && !body.actions_ref_json && !body.actions_snapshot_json && !refJson && !snapJson) {
    body.actions_ref_json = { actions: [] };
    body.actions_snapshot_json = { version: 1, policy: 'snapshot', actions: [] };
  }
  // Validation stricte: chaque shortcode présent dans l'HTML doit référencer une action connue (id OU ref)
  const htmlIds = extractActionIdsFromHtml(body.content_html || '');
  if (htmlIds.length) {
    const manifest = (body.actions_snapshot_json && typeof body.actions_snapshot_json === 'object') ? body.actions_snapshot_json : null;
    const refs = (body.actions_ref_json && typeof body.actions_ref_json === 'object') ? body.actions_ref_json : null;
    const knownIds = new Set();
    const knownRefs = new Set();
    try {
      const snapActs = Array.isArray(manifest?.actions) ? manifest.actions : [];
      const refActs = Array.isArray(refs?.actions) ? refs.actions : [];
      snapActs.forEach(a => { if (a && typeof a === 'object') { if (a.id) knownIds.add(String(a.id)); if (a.ref) knownRefs.add(String(a.ref)); } });
      refActs.forEach(a => { if (a && typeof a === 'object' && a.ref) knownRefs.add(String(a.ref)); });
    } catch (_) {}
    const missing = htmlIds.filter(id => !(knownIds.has(id) || knownRefs.has(id)));
    if (missing.length) {
      showToast(`Actions manquantes pour shortcodes: ${missing.join(', ')}`, { kind: 'warning' });
      return;
    }
  }
  if (!Number.isFinite(body.category_id) || body.category_id <= 0 || !body.title || !body.content_html) {
    showToast('Veuillez choisir une catégorie et renseigner Titre + HTML.', { kind: 'warning' });
    return;
  }
  if (body.id) {
    await requestJSON('/api/admin/notifications/rich/update', { method: 'POST', body });
    showToast('Notification mise à jour', { kind: 'success' });
  } else {
    await requestJSON('/api/admin/notifications/rich', { method: 'POST', body });
    showToast('Notification créée', { kind: 'success' });
  }
  try { showSection(form, 'actions-palette', false); showSection(form, 'actions-json', false); } catch (_) {}
  form.reset();
  // Nettoyer le brouillon persistant après succès
  clearDraftStorage();
  setState({ notifRich: await fetchRichNotifications(state.notifCatSelected) });
}

async function onSelectActionId(form) {
  const actionId = form.querySelector('select[name="act_action_id"]').value;
  if (!actionId) return;
  try {
    const payload = await requestJSON(`/api/admin/actions/catalog/${encodeURIComponent(actionId)}`, { method: 'GET' });
    const versions = Array.isArray(payload?.action?.versions) ? payload.action.versions : [];
    const select = form.querySelector('select[name="act_version"]');
    if (select) {
      select.innerHTML = versions.map(v => `<option value="${escapeAttr(String(v.version))}">${escapeHtml(String(v.version))} — ${escapeHtml(String(v.status||''))}</option>`).join('');
      select._versions = versions;
      select.addEventListener('change', () => updateParamsFormFromSelectedVersion(form));
    }
    // Prefill params with defaultParams of active version
    const active = versions.find(v => String(v.status||'').toLowerCase()==='active') || versions[0];
    if (select && active) { try { select.value = String(active.version); } catch (_) {} }
    const defParams = active?.definition?.defaultParams || {};
    const ta = form.querySelector('textarea[name="act_params_json"]');
  if (ta) ta.value = JSON.stringify(defParams);
  // Render params form from payloadSchema
  renderParamsForm(form, active?.definition?.payloadSchema || {}, defParams);
} catch (e) { console.error(e); }
}

async function addActionRef(form) {
  // Capturer le brouillon avant tout re-render pour préserver les champs du formulaire
  let draft = null;
  try { draft = captureNotifDraft(form); } catch (_) {}
  const id = form.querySelector('input[name="act_local_id"]').value.trim();
  const ref = form.querySelector('select[name="act_action_id"]').value;
  const version = Number(form.querySelector('select[name="act_version"]').value || 0);
  const paramsRaw = form.querySelector('textarea[name="act_params_json"]').value.trim();
  if (!id || !ref || !Number.isFinite(version) || version <= 0) { showToast('Renseigner id, action, version', { kind: 'warning' }); return; }
  let params = {};
  if (paramsRaw) { try { params = JSON.parse(paramsRaw); } catch (_) { showToast('Params JSON invalide', { kind:'warning' }); return; } }
  try {
    const det = await requestJSON(`/api/admin/actions/catalog/${encodeURIComponent(ref)}`, { method: 'GET' });
    const ver = (det?.action?.versions || []).find(v => Number(v.version) === version);
    const entry = {
      id,
      ref,
      version,
      params,
      label: ver?.definition?.label || {},
      versionEtag: ver?.etag || ''
    };
    const list = Array.isArray(state.notifActions) ? state.notifActions.slice() : [];
    list.push(entry);
    setState({ notifActions: list, ...(draft ? { notifDraft: draft } : {}) });
    try { saveDraftToStorage(draft || captureNotifDraft(form), list); } catch (_) {}
    try { showSection(form, 'actions-palette', false); } catch (_) {}
  } catch (e) { console.error(e); showToast('Impossible d\'ajouter l\'action', { kind:'error' }); }
}

function removeActionRef(index) {
  // Capturer le brouillon du formulaire courant avant re-render
  let draft = null;
  try { const f = document.querySelector('[data-form="notification-rich"]'); if (f) draft = captureNotifDraft(f); } catch (_) {}
  const list = Array.isArray(state.notifActions) ? state.notifActions.slice() : [];
  if (index >= 0 && index < list.length) list.splice(index, 1);
  setState({ notifActions: list, ...(draft ? { notifDraft: draft } : {}) });
  try { saveDraftToStorage(draft, list); } catch (_) {}
}

function setupRichPreview(form) {
  function update() {
    const html = form.querySelector('textarea[name="content_html"]')?.value || '';
    const css = form.querySelector('textarea[name="content_css"]')?.value || '';
    const origin = window.location && window.location.origin ? window.location.origin : '';
    const nonce = Math.random().toString(36).slice(2) + Date.now().toString(36);
    const csp = `default-src 'none'; img-src ${origin} data:; media-src 'none'; style-src 'unsafe-inline'; script-src 'nonce-${nonce}'; connect-src 'none';`;
    const pre = `function __sb_expand(){ try {var body=document.body; if(!body) return; var html=body.innerHTML; html=html.replace(/\\[sb:action\\s+id=\\\"([^\\\"]+)\\\"\\]([\\s\\S]*?)\\[\\/sb:action\\]/g,function(_,id,txt){ var lab=String(txt).replace(/<[^>]+>/g,' ').replace(/\\s+/g,' ').trim(); return '<button data-sb-action-id=\\"'+id+'\\" aria-label=\\"'+lab+'\\">'+txt+'</button>';}); body.innerHTML=html;} catch(_){} } document.addEventListener('DOMContentLoaded', function(){ __sb_expand(); });`;
    const srcdoc = `<!doctype html><html><head><meta http-equiv="Content-Security-Policy" content="${escapeAttr(csp)}"><style>${css}</style></head><body>${html}<script nonce="${escapeAttr(nonce)}">${pre}</script></body></html>`;
    const container = document.getElementById('notifrich-preview');
    if (!container) return;
    container.innerHTML = `<iframe class="notifrich-iframe" sandbox="allow-scripts" referrerpolicy="no-referrer" srcdoc="${escapeAttr(srcdoc)}" style="width:100%;height:560px;border:1px solid #e5e7eb;border-radius:6px;"></iframe>`;
  }
  form.addEventListener('input', (e) => {
    const target = e.target;
    if (!(target instanceof HTMLElement)) return;
    if (target.matches('textarea[name="content_html"], textarea[name="content_css"], input[name="title"]')) {
      update();
    }
  });
  // initial
  update();
}

async function selectCategory(id) {
  if (!Number.isFinite(id) || id <= 0) return;
  setState({ notifCatSelected: id, loading: true });
  const items = await fetchRichNotifications(id);
  setState({ notifRich: items, loading: false });
}

function openEditCategory(id) {
  if (!Number.isFinite(id) || id <= 0) return;
  setState({ notifCatSelected: id, showEditCategory: true });
}

async function deleteCategory(id) {
  if (!Number.isFinite(id) || id <= 0) return;
  const ok = await confirmDialog({
    title: 'Supprimer la catégorie',
    message: 'Cette action va supprimer la catégorie ET toutes les données liées (contenus, abonnements, états utilisateurs, métadonnées scheduler). Opération définitive. Confirmer ?',
    okLabel: 'Supprimer',
    cancelLabel: 'Annuler',
  });
  if (!ok) return;
  await requestJSON('/api/admin/notification-categories/delete', { method: 'POST', body: { id } });
  showToast('Catégorie supprimée', { kind: 'success' });
  await loadView('notifications');
}

async function loadTemplateById(templateId) {
  const list = state.notifTemplates || [];
  const entry = list.find(t => t.id === templateId);
  if (!entry) throw new Error('TEMPLATE_NOT_FOUND');
  const base = String(entry.path || '').replace(/\/$/, '');
  const files = entry.files || {};
  const htmlUrl = files.html ? `${base}/${files.html}` : null;
  const cssUrl = files.css ? `${base}/${files.css}` : null;
  const jsUrl = files.js ? `${base}/${files.js}` : null;
  const [html, css, js] = await Promise.all([
    htmlUrl ? fetchText(htmlUrl) : Promise.resolve(''),
    cssUrl ? fetchText(cssUrl) : Promise.resolve(''),
    jsUrl ? fetchText(jsUrl) : Promise.resolve(''),
  ]);
  const form = document.querySelector('[data-form="notification-rich"]');
  if (!form) return;
  const setVal = (sel, v) => { const el = form.querySelector(sel); if (el) el.value = v ?? ''; };
  setVal('input[name="id"]', '');
  setVal('textarea[name="content_html"]', html);
  setVal('textarea[name="content_css"]', css);
  
  const submitBtn = form.querySelector('[data-role="notifrich-submit"]');
  if (submitBtn) submitBtn.textContent = 'Créer la notification';
  const origin = window.location && window.location.origin ? window.location.origin : '';
  const nonce = Math.random().toString(36).slice(2) + Date.now().toString(36);
  const csp = `default-src 'none'; img-src ${origin} data:; media-src 'none'; style-src 'unsafe-inline'; script-src 'nonce-${nonce}'; connect-src 'none';`;
  const pre = `function __sb_expand(){ try {var body=document.body; if(!body) return; var html=body.innerHTML; html=html.replace(/\\[sb:action\\s+id=\\\"([^\\\"]+)\\\"\\]([\\s\\S]*?)\\[\\/sb:action\\]/g,function(_,id,txt){ var lab=String(txt).replace(/<[^>]+>/g,' ').replace(/\\s+/g,' ').trim(); return '<button data-sb-action-id=\\"'+id+'\\" aria-label=\\"'+lab+'\\">'+txt+'</button>';}); body.innerHTML=html;} catch(_){} } document.addEventListener('DOMContentLoaded', function(){ __sb_expand(); });`;
  const srcdoc = `<!doctype html><html><head><meta http-equiv=\"Content-Security-Policy\" content=\"${escapeAttr(csp)}\"><style>${css}</style></head><body>${html}<script nonce=\"${escapeAttr(nonce)}\">${pre}</script></body></html>`;
  const container = document.getElementById('notifrich-preview');
  if (container) container.innerHTML = `<iframe class="notifrich-iframe" sandbox="allow-scripts" referrerpolicy="no-referrer" srcdoc="${escapeAttr(srcdoc)}" style="width:100%;height:560px;border:1px solid #e5e7eb;border-radius:6px;"></iframe>`;
  showToast('Gabarit chargé', { kind: 'success' });
}

async function fetchText(url) {
  const res = await fetch(url, { credentials: 'include', headers: { 'Accept': 'text/plain,*/*' } });
  if (!res.ok) throw new Error('HTTP_' + res.status);
  return await res.text();
}

async function editNotificationRich(id) {
  try {
    const payload = await requestJSON(`/api/admin/notifications/rich/${id}`, { method: 'GET' });
    const n = payload?.notification;
    if (!n) { showToast('Notification introuvable', { kind: 'error' }); return; }
    const form = document.querySelector('[data-form="notification-rich"]');
    if (!form) return;
    const setVal = (sel, v) => { const el = form.querySelector(sel); if (el) el.value = v ?? ''; };
    const setChecked = (sel, v) => { const el = form.querySelector(sel); if (el) el.checked = !!v; };
    // Remplir immédiatement une première fois (avant re-render)
    setVal('input[name="id"]', n.id);
    setVal('input[name="category_id"]', n.category_id);
    setVal('input[name="title"]', n.title);
    setVal('input[name="emitter"]', n.emitter ?? '');
    setVal('input[name="weight"]', String(n.weight ?? 0));
    setChecked('input[name="active"]', !!n.active);
    setVal('textarea[name="content_html"]', n.content_html ?? '');
    setVal('textarea[name="content_css"]', n.content_css ?? '');
    const refStr = (n.actions_ref_json ? (typeof n.actions_ref_json === 'string' ? n.actions_ref_json : JSON.stringify(n.actions_ref_json)) : '');
    const snapStr = (n.actions_snapshot_json ? (typeof n.actions_snapshot_json === 'string' ? n.actions_snapshot_json : JSON.stringify(n.actions_snapshot_json)) : '');
    if (refStr) setVal('textarea[name="actions_ref_json"]', refStr);
    if (snapStr) setVal('textarea[name="actions_snapshot_json"]', snapStr);
    try {
      let refs = typeof n.actions_ref_json === 'string' ? JSON.parse(n.actions_ref_json) : (n.actions_ref_json || {});
      let snaps = typeof n.actions_snapshot_json === 'string' ? JSON.parse(n.actions_snapshot_json) : (n.actions_snapshot_json || {});
      // Tolérance double-encodage éventuel
      if (typeof refs === 'string') { try { const tmp = JSON.parse(refs); if (tmp && typeof tmp === 'object') refs = tmp; } catch (_) {} }
      if (typeof snaps === 'string') { try { const tmp = JSON.parse(snaps); if (tmp && typeof tmp === 'object') snaps = tmp; } catch (_) {} }
      const refsArr = Array.isArray(refs?.actions) ? refs.actions : [];
      const snapsArr = Array.isArray(snaps?.actions) ? snaps.actions : [];
      let joined = refsArr.map(r => {
        const s = snapsArr.find(x => x && x.ref === r.ref && Number(x.version) === Number(r.version));
        return { id: (s?.id || r.ref), ref: r.ref, version: Number(r.version||0), params: r.params || {}, label: s?.label || {}, versionEtag: s?.versionEtag || '' };
      });
      // Fallback: si aucune référence mais un snapshot existe, reconstruire la liste depuis le snapshot
      if ((!joined || !joined.length) && snapsArr.length) {
        joined = snapsArr.map(s => ({ id: s.id || s.ref, ref: s.ref, version: Number(s.version||0), params: s.params || {}, label: s.label || {}, versionEtag: s.versionEtag || '' }));
      }
      // Capturer un brouillon pour réappliquer après re-render
      const draft = {
        id: n.id,
        category_id: n.category_id,
        title: n.title ?? '',
        emitter: n.emitter ?? '',
        weight: n.weight ?? 0,
        sequence_index: n.sequence_index ?? undefined,
        active: !!n.active,
        content_html: n.content_html ?? '',
        content_css: n.content_css ?? '',
        actions_ref_json: refStr,
        actions_snapshot_json: snapStr,
      };
      setState({ notifActions: joined, notifDraft: draft });
    } catch (_) {}
    // refresh preview
    const origin = window.location && window.location.origin ? window.location.origin : '';
    const nonce = Math.random().toString(36).slice(2) + Date.now().toString(36);
    const csp = `default-src 'none'; img-src ${origin} data:; media-src 'none'; style-src 'unsafe-inline'; script-src 'nonce-${nonce}'; connect-src 'none';`;
    const pre = `function __sb_expand(){ try {var body=document.body; if(!body) return; var html=body.innerHTML; html=html.replace(/\\[sb:action\\s+id=\\\"([^\\\"]+)\\\"\\]([\\s\\S]*?)\\[\\/sb:action\\]/g,function(_,id,txt){ var lab=String(txt).replace(/<[^>]+>/g,' ').replace(/\\s+/g,' ').trim(); return '<button data-sb-action-id=\\"'+id+'\\" aria-label=\\"'+lab+'\\">'+txt+'</button>';}); body.innerHTML=html;} catch(_){} } document.addEventListener('DOMContentLoaded', function(){ __sb_expand(); });`;
    const srcdoc = `<!doctype html><html><head><meta http-equiv=\"Content-Security-Policy\" content=\"${escapeAttr(csp)}\"><style>${n.content_css ?? ''}</style></head><body>${n.content_html ?? ''}<script nonce=\"${escapeAttr(nonce)}\">${pre}</script></body></html>`;
    const container = document.getElementById('notifrich-preview');
    if (container) container.innerHTML = `<iframe class="notifrich-iframe" sandbox="allow-scripts" referrerpolicy="no-referrer" srcdoc="${escapeAttr(srcdoc)}" style="width:100%;height:560px;border:1px solid #e5e7eb;border-radius:6px;"></iframe>`;
    const submitBtn = form.querySelector('[data-role="notifrich-submit"]');
    if (submitBtn) submitBtn.textContent = 'Enregistrer';
    showToast('Formulaire chargé en mode édition', { kind: 'success' });
  } catch (e) {
    console.error(e);
    showToast('Impossible de charger la notification', { kind: 'error' });
  }
}

async function deleteNotificationRich(id) {
  if (!(await confirmDialog({ title: 'Supprimer la notification', message: 'Supprimer cette notification ?', okLabel: 'Supprimer', cancelLabel: 'Annuler' }))) return;
  await requestJSON('/api/admin/notifications/rich/delete', { method: 'POST', body: { id } });
  showToast('Notification supprimée', { kind: 'success' });
  setState({ notifRich: await fetchRichNotifications(state.notifCatSelected) });
}

// legacy edit/delete notification removed (rich notifications replace it)

async function handleUserCreate(event) {
  event.preventDefault();
  const form = event.currentTarget;
  const data = new FormData(form);
  const body = {
    email: data.get('email')?.toString().trim(),
    password: data.get('password')?.toString(),
    role: data.get('role')?.toString().trim() || 'standard',
    pseudo: data.get('pseudo')?.toString().trim() || null,
  };
  if (!body.email || !body.password) {
    showToast('Email et mot de passe sont requis.', { kind: 'warning' });
    return;
  }
  await requestJSON('/api/admin/users', { method: 'POST', body });
  form.reset();
  showToast('Utilisateur créé', { kind: 'success' });
  await loadView('users');
}

async function editUser(id) {
  const user = state.users.find(item => item.id === id);
  if (!user) {
    showToast('Utilisateur introuvable', { kind: 'error' });
    return;
  }
  const pseudo = await promptDialog({ title: 'Modifier l\'utilisateur', label: 'Pseudo', defaultValue: user.pseudo ?? '' });
  if (pseudo === null) return;
  const role = await promptDialog({ title: 'Modifier l\'utilisateur', label: 'Rôle', defaultValue: user.role ?? 'standard' });
  if (role === null) return;
  const blockedInput = await promptDialog({ title: 'Modifier l\'utilisateur', label: 'Bloqué ? (oui/non)', defaultValue: user.blocked ? 'oui' : 'non' });
  if (blockedInput === null) return;

  const payload = {
    id,
    pseudo: pseudo.trim() || null,
    role: role.trim() || 'standard',
    blocked: blockedInput.trim().toLowerCase().startsWith('o'),
  };

  await requestJSON('/api/admin/users/update', { method: 'POST', body: payload });
  showToast('Profil mis à jour', { kind: 'success' });
  await loadView('users');
}

async function deleteUser(id) {
  if (!(await confirmDialog({ title: 'Supprimer l\'utilisateur', message: 'Supprimer cet utilisateur et ses boards ?', okLabel: 'Supprimer', cancelLabel: 'Annuler' }))) return;
  await requestJSON('/api/admin/users/delete', { method: 'POST', body: { id } });
  showToast('Utilisateur supprimé', { kind: 'success' });
  await loadView('users');
}

async function openUserData(id) {
  if (!Number.isFinite(id) || id <= 0) return;
  setState({ loading: true });
  try {
    const data = await fetchUserData(id);
    setState({ view: 'user-details', userDetails: data, loading: false });
  } catch (e) {
    console.error(e);
    showToast('Impossible de charger la fiche', { kind: 'error' });
    setState({ loading: false });
  }
}

async function handleLicenseCreate(event) {
  event.preventDefault();
  const form = event.currentTarget;
  const data = new FormData(form);
  const packsRaw = data.get('packs')?.toString() ?? '';
  const expiresInput = data.get('expiresAt')?.toString() ?? '';
  let expiresAt = null;
  if (expiresInput) {
    const ts = new Date(expiresInput).getTime();
    expiresAt = Number.isNaN(ts) ? null : Math.floor(ts / 1000);
  }
  const body = {
    code: data.get('code')?.toString().trim() || undefined,
    role: data.get('role')?.toString().trim() || 'standard',
    packs: packsRaw.split(',').map(x => x.trim()).filter(Boolean),
    expiresAt,
  };
  await requestJSON('/api/admin/licenses', { method: 'POST', body });
  form.reset();
  showToast('Licence créée', { kind: 'success' });
  await loadView('licenses');
}

async function assignLicense(licenseId) {
  const userIdInput = await promptDialog({ title: 'Assigner la licence', label: 'ID utilisateur', defaultValue: '' });
  if (userIdInput === null) return;
  const userId = Number(userIdInput);
  if (!Number.isFinite(userId) || userId <= 0) {
    showToast('Identifiant invalide', { kind: 'warning' });
    return;
  }
  await requestJSON('/api/admin/licenses/assign', { method: 'POST', body: { licenseId, userId } });
  showToast('Licence assignée', { kind: 'success' });
  await loadView('licenses');
}

async function unassignLicense(licenseId) {
  await requestJSON('/api/admin/licenses/unassign', { method: 'POST', body: { licenseId } });
  showToast('Licence libérée', { kind: 'success' });
  await loadView('licenses');
}

async function linkLicense(licenseId) {
  const userIdInput = await promptDialog({ title: 'Lien d\'activation', label: 'ID utilisateur', defaultValue: '' });
  if (userIdInput === null) return;
  const userId = Number(userIdInput);
  if (!Number.isFinite(userId) || userId <= 0) {
    showToast('Identifiant invalide', { kind: 'warning' });
    return;
  }
  const ttlInput = await promptDialog({ title: 'Lien d\'activation', label: 'Durée de validité en secondes', defaultValue: '86400' });
  if (ttlInput === null) return;
  const ttlSeconds = Number(ttlInput) || 86400;
  const payload = await requestJSON('/api/admin/licenses/link', { method: 'POST', body: { licenseId, userId, ttlSeconds } });
  const activationUrl = new URL(payload.url ?? '', window.location.origin).toString();
  try {
    await navigator.clipboard.writeText(activationUrl);
    showToast('Lien copié dans le presse-papier', { kind: 'success' });
  } catch (_) {
    showToast(`Lien généré: ${activationUrl}`, { kind: 'success' });
  }
}

async function deleteLicense(licenseId) {
  if (!(await confirmDialog({ title: 'Supprimer la licence', message: 'Supprimer cette licence ?', okLabel: 'Supprimer', cancelLabel: 'Annuler' }))) return;
  await requestJSON('/api/admin/licenses/delete', { method: 'POST', body: { licenseId } });
  showToast('Licence supprimée', { kind: 'success' });
  await loadView('licenses');
}

async function clearIp(ip) {
  await requestJSON('/api/admin/security/ip/clear', { method: 'POST', body: { ip } });
  showToast(`IP ${ip} débloquée`, { kind: 'success' });
  await loadView('security');
}

async function clearAllIps() {
  if (!(await confirmDialog({ title: 'Purger les IP bloquées', message: 'Purger toutes les IP bloquées ?', okLabel: 'Purger', cancelLabel: 'Annuler' }))) return;
  await requestJSON('/api/admin/security/ip/clear-all', { method: 'POST' });
  showToast('Liste IP purgée', { kind: 'success' });
  await loadView('security');
}

async function resetSchema() {
  if (!(await confirmDialog({ title: 'Réinitialiser le schéma', message: 'Réinitialiser le schéma de base de données ?', okLabel: 'Réinitialiser', cancelLabel: 'Annuler' }))) return;
  await requestJSON('/api/admin/security/reset-schema', { method: 'POST' });
  showToast('Schéma réinitialisé', { kind: 'success' });
}

async function logout() {
  try { await requestJson('/api/commands', { method: 'POST', body: { type: 'Account.Logout', payload: {} } }); } catch (_) {}
  goToAuth();
}

async function requestJSON(url, { method = 'GET', body } = {}) {
  try {
    return await requestJson(url, { method, body });
  } catch (e) {
    const message = e?.payload?.error ?? e?.payload?.message ?? e?.message ?? 'Action impossible';
    showToast(message, { kind: 'error' });
    throw e;
  }
}

function isAdmin(user) {
  const role = (user?.role ?? 'standard').toLowerCase();
  return role === 'admin' || role === 'superadmin';
}
