import { goToBoards } from '../../packages/services/navigation.js';
import { requestJson } from '../../packages/services/http.js';

const form = document.getElementById("authForm");
const tabs = document.querySelectorAll(".tab");
const errorBox = document.getElementById("authError");
let mode = "login";

init();

function init() {
  tabs.forEach(tab => tab.addEventListener("click", () => switchMode(tab.dataset.mode)));
  form.addEventListener("submit", submitForm);
}

function switchMode(nextMode) {
  mode = nextMode;
  tabs.forEach(tab => tab.classList.toggle("active", tab.dataset.mode === mode));
  form.querySelector("button[type=submit]").textContent = mode === "login" ? "Se connecter" : "Créer un compte";
  hideError();
}

async function submitForm(event) {
  event.preventDefault();
  hideError();
  const data = new FormData(form);
  const payload = {
    email: data.get("email"),
    password: data.get("password"),
  };
  try {
    const endpoint = `/api/auth/${mode === "login" ? "login" : "register"}`;
    const body = await requestJson(endpoint, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: payload,
    });
    if (!body || body.ok !== true) {
      const code = body?.error || body?.code || body?.message || "REQUEST_FAILED";
      showError(mapAuthError(code));
      return;
    }
    goToBoards();
  } catch (error) {
    console.error(error);
    const code = (error && typeof error === 'object')
      ? (error.payload?.error || error.payload?.code || error.message)
      : String(error || 'REQUEST_FAILED');
    if (code === 'COMMAND_RESPONSE_INVALID') {
      showError('Réponse invalide du serveur');
      return;
    }
    showError(mapAuthError(code));
  }
}

function showError(message) {
  errorBox.textContent = message;
  errorBox.hidden = false;
}

function hideError() {
  errorBox.hidden = true;
  errorBox.textContent = "";
}

function mapAuthError(code) {
  const c = String(code || '').toUpperCase();
  switch (c) {
    case 'INVALID_PAYLOAD':
      return 'Email invalide ou mot de passe trop court.';
    case 'EMAIL_ALREADY_EXISTS':
      return 'Cet email est déjà utilisé.';
    case 'INVALID_CREDENTIALS':
    case 'UNAUTHORIZED':
      return 'Identifiants invalides.';
    case 'ACCOUNT_BLOCKED':
      return 'Ce compte est bloqué.';
    case 'RATE_LIMITED':
      return 'Trop de tentatives. Réessayez plus tard.';
    case 'CSRF_TOKEN_INVALID':
      return 'Session expirée. Actualisez la page.';
    default:
      return 'Une erreur s\'est produite.';
  }
}
