import { showToast } from '../../../packages/ui/toast.js';
import { escapeHtml, formatTime, formatTimeFromMs } from './ui/format.js';

let store = null;
let getCsrfToken = null; // reserved
let confirmAutosave = null;
let reloadBoard = null;

// Internal state
let pendingCommands = 0;
let latestAutosaveDetails = null;
let autosaveSettleTimer = null;
const AUTOSAVE_CONFIRM_DELAY = 650;
const AUTOSAVE_MAX_ATTEMPTS = 3;
let autosaveConfirmTimer = null;
let autosaveConfirmAttempts = 0;
let autosaveConfirmInFlight = false;
let autosaveDirty = false;
let autosavePendingDetails = null;
let lastAutosaveToastAt = 0;

export function initAutosave(ctx) {
  store = ctx.store;
  getCsrfToken = ctx.getCsrfToken || (() => '');
  confirmAutosave = ctx.confirmAutosave; // (boardId, revision, csrf) => Promise
  reloadBoard = ctx.reloadBoard || null;
}

export function getPendingCount() { return pendingCommands; }
export function isAutosavePending() { return pendingCommands > 0; }

export function mapAutosaveStatus(raw = null, options = {}) {
  const fallback = options.fallbackStatus ?? 'idle';
  if (!raw) {
    return {
      status: fallback,
      message: fallback === 'pending' ? 'Sauvegarde…' : (fallback === 'error' ? 'Erreur de sauvegarde' : 'Prêt'),
      details: null,
    };
  }
  const status = raw.status ?? fallback;
  let message;
  switch (status) {
    case 'pending': message = 'Sauvegarde…'; break;
    case 'dirty': message = 'Modifications en attente'; break;
    case 'saved':
      if (typeof raw.updatedAt === 'number') { message = `Enregistré à ${formatTimeFromMs(raw.updatedAt)}`; }
      else if (raw.savedAt) { message = `Enregistré à ${formatTime(raw.savedAt)}`; }
      else { message = 'Enregistré'; }
      break;
    case 'blocked': message = 'Action bloquée'; break;
    case 'failed': message = "Erreur d'enregistrement"; break;
    case 'aborted': message = 'Board introuvable'; break;
    default: message = 'Prêt'; break;
  }
  return { status, message, details: raw };
}

export function renderAutosavePanel(state) {
  if (!state) return '';
  // Only render when explicitly opened by the user
  if (!state.autosavePanelOpen) return '';
  const autosave = state.autosave ?? mapAutosaveStatus();
  const queueSize = pendingCommands;
  const hasPending = queueSize > 0 || autosave.status === 'pending';
  const details = autosave.details ?? {};
  const preEvents = Array.isArray(state?.lastEvents?.pre) ? state.lastEvents.pre : [];
  const postEvents = Array.isArray(state?.lastEvents?.post) ? state.lastEvents.post : [];
  const evaluated = state.trace ?? null;
  const version = state.rulesVersion ?? null;

  const commandType = state?.lastCommand?.command?.type ?? null;
  const savedAt = details?.savedAt ? formatTime(details.savedAt) : null;
  const duration = details?.durationMs ? `${details.durationMs} ms` : null;
  const revision = details?.revision ?? null;
  const errorMessage = state?.lastCommand?.error?.message ?? null;
  const timestamp = state?.lastCommand?.timestamp ? new Date(state.lastCommand.timestamp).toLocaleString('fr-FR') : null;
  const statusLabel = autosaveStatusLabel(autosave.status);

  return `
    <div class="autosave-overlay" data-action="close-autosave">
      <div class="autosave-panel" data-stop-propagation>
        <header class="autosave-panel__header">
          <div>
            <strong>Synchronisation</strong>
            ${timestamp ? `<span class="autosave-panel__meta">Dernière action ${escapeHtml(timestamp)}</span>` : ''}
          </div>
          <button class="btn ghost" data-action="close-autosave">Fermer</button>
        </header>
        <section class="autosave-panel__section">
          <div class="status-badge status-badge--${autosave.status}">${escapeHtml(statusLabel)}</div>
          <p class="autosave-panel__message">${escapeHtml(autosave.message)}</p>
          <ul class="autosave-panel__list">
            ${commandType ? `<li><strong>Commande :</strong> ${escapeHtml(commandType)}</li>` : ''}
            ${savedAt ? `<li><strong>Enregistré :</strong> ${escapeHtml(savedAt)}</li>` : ''}
            ${duration ? `<li><strong>Durée :</strong> ${escapeHtml(duration)}</li>` : ''}
            ${revision ? `<li><strong>Révision :</strong> ${escapeHtml(String(revision))}</li>` : ''}
            ${hasPending ? `<li><strong>Actions en attente :</strong> ${queueSize > 0 ? queueSize : 'Synchronisation en cours'}</li>` : ''}
            ${errorMessage ? `<li><strong>Erreur :</strong> ${escapeHtml(errorMessage)}</li>` : ''}
          </ul>
        </section>
        ${renderAutosaveEvents('Pré-événements', preEvents)}
        ${renderAutosaveEvents('Post-événements', postEvents)}
        ${renderAutosaveTrace(evaluated, version)}
      </div>
    </div>
  `;
}

export function beginAutosaveCycle() {
  pendingCommands += 1;
  if (pendingCommands === 1) {
    updateAutosaveStatus({ status: 'pending' });
  }
  autosaveDirty = true;
  if (autosaveConfirmTimer) {
    clearTimeout(autosaveConfirmTimer);
    autosaveConfirmTimer = null;
  }
}

export function completeAutosaveSuccess(details) {
  pendingCommands = Math.max(0, pendingCommands - 1);
  latestAutosaveDetails = details;
  if (pendingCommands === 0) {
    clearTimeout(autosaveSettleTimer);
    autosaveSettleTimer = setTimeout(() => {
      updateAutosaveStatus(details);
      handleAutosaveFeedback({ autosave: details });
      autosavePendingDetails = details;
      autosaveDirty = false;
      if (!autosaveConfirmInFlight) {
        autosaveConfirmTimer = setTimeout(() => { attemptAutosaveConfirm().catch(() => {}); }, AUTOSAVE_CONFIRM_DELAY);
      }
    }, 120);
  }
}

export function completeAutosaveError(details, options = {}) {
  pendingCommands = Math.max(0, pendingCommands - 1);
  latestAutosaveDetails = null;
  clearTimeout(autosaveSettleTimer);
  updateAutosaveStatus(details);
  if (autosaveConfirmTimer) {
    clearTimeout(autosaveConfirmTimer);
    autosaveConfirmTimer = null;
  }
  autosaveConfirmAttempts = 0;
  autosaveConfirmInFlight = false;
  autosavePendingDetails = null;
  if (options.toast !== false) {
    handleAutosaveFeedback({ autosave: details, error: { message: options.message ?? null } });
  }
}

function updateAutosaveStatus(raw) {
  store.setState(prev => {
    const previousDetails = prev.autosave?.details ?? {};
    const merged = { ...previousDetails, ...raw };
    return { ...prev, autosave: mapAutosaveStatus(merged) };
  });
}

function scheduleAutosaveConfirm(delay = AUTOSAVE_CONFIRM_DELAY) {
  if (autosaveConfirmInFlight || pendingCommands > 0) return;
  if (!store) return;
  const state = store.getState();
  const revision = autosavePendingDetails?.revision ?? state.boardRevision ?? state.boardMeta?.revision ?? null;
  if (!state.currentBoardId || !revision) return;
  if (autosaveConfirmTimer) clearTimeout(autosaveConfirmTimer);
  autosaveConfirmTimer = setTimeout(() => { attemptAutosaveConfirm().catch(() => {}); }, Math.max(0, delay));
}

async function attemptAutosaveConfirm() {
  if (autosaveConfirmInFlight) return;
  const state = store.getState();
  const revision = autosavePendingDetails?.revision ?? state.boardRevision ?? state.boardMeta?.revision ?? null;
  if (!state.currentBoardId || !revision) return;
  autosaveConfirmInFlight = true;
  try {
    const payload = await confirmAutosave(state.currentBoardId, revision, getCsrfToken());
    const updatedAt = typeof payload?.updatedAt === 'number' ? payload.updatedAt : (state.boardMeta?.updatedAt ?? Date.now());
    const history = Array.isArray(payload?.history)
      ? payload.history
      : Array.isArray(autosavePendingDetails?.history) ? autosavePendingDetails.history
      : Array.isArray(state.boardMeta?.history) ? state.boardMeta.history : [];
    const next = { status: 'saved', savedAt: new Date().toISOString(), updatedAt, revision: typeof payload?.revision === 'number' ? payload.revision : revision, history };
    updateAutosaveStatus(next);
    autosaveConfirmAttempts = 0;
    autosavePendingDetails = next;
    autosaveConfirmTimer = null;
  } catch (error) {
    const status = error?.status ?? 0;
    if (status === 409) {
      showToast('Le board a été modifié ailleurs. Rechargement…', { kind: 'warning' });
      try { if (typeof reloadBoard === 'function') { await reloadBoard(); } } catch (e) { /* ignore */ }
      autosaveConfirmAttempts = 0;
      autosavePendingDetails = null;
      autosaveConfirmTimer = null;
    } else if (autosaveConfirmAttempts < AUTOSAVE_MAX_ATTEMPTS) {
      autosaveConfirmAttempts += 1;
      const backoff = AUTOSAVE_CONFIRM_DELAY * autosaveConfirmAttempts;
      scheduleAutosaveConfirm(backoff);
    } else {
      autosaveConfirmTimer = null;
    }
  } finally {
    autosaveConfirmInFlight = false;
  }
}

// Expose a helper to confirm immediately when idle (e.g., on tab hide)
export function attemptAutosaveConfirmIfIdle() {
  if (autosaveConfirmInFlight) return;
  if (pendingCommands > 0) return;
  if (autosaveDirty) return;
  if (!store) return;
  const state = store.getState();
  const revision = autosavePendingDetails?.revision ?? state.boardRevision ?? state.boardMeta?.revision ?? null;
  if (!state.currentBoardId || !revision) return;
  attemptAutosaveConfirm().catch(() => {});
}

// Normalize autosave meta used across the app
export function normalizeAutosaveDetails(raw, options = {}) {
  const fallbackStatus = options.fallbackStatus ?? 'saved';
  const previous = options.previous ?? {};
  const status = raw?.status ?? fallbackStatus;

  let updatedAt = typeof raw?.updatedAt === 'number' ? raw.updatedAt : null;
  if (!updatedAt && status === 'saved') {
    updatedAt = Date.now();
  }
  if (!updatedAt && typeof previous.updatedAt === 'number') {
    updatedAt = previous.updatedAt;
  }

  let revision = typeof raw?.revision === 'number' ? raw.revision : null;
  if (!revision && updatedAt) {
    revision = updatedAt;
  }
  if (!revision && typeof previous.revision === 'number') {
    revision = previous.revision;
  }

  const previousHistory = Array.isArray(previous.history) ? previous.history : [];
  const history = Array.isArray(raw?.history) ? raw.history : previousHistory;

  return {
    ...raw,
    status,
    savedAt: raw?.savedAt ?? (status === 'saved' ? new Date().toISOString() : null),
    updatedAt,
    revision,
    history,
  };
}

function handleAutosaveFeedback(result) {
  if (!result || !result.autosave) return;
  const autosave = mapAutosaveStatus(result.autosave);
  const status = autosave.status;
  const errorMessage = result?.error?.message;
  if (status === 'saved') {
    const now = Date.now();
    if (now - lastAutosaveToastAt > 4000) {
      showToast('Modifications enregistrées', { kind: 'success' });
      lastAutosaveToastAt = now;
    }
  } else if ((status === 'blocked' || status === 'failed' || status === 'aborted') && !errorMessage) {
    const kind = status === 'blocked' ? 'warning' : 'error';
    const message = status === 'blocked' ? 'Action bloquée par les règles' : 'Enregistrement impossible';
    showToast(message, { kind });
  }
}

function renderAutosaveEvents(title, events) {
  if (!Array.isArray(events) || !events.length) return '';
  return `
    <section class="autosave-panel__section autosave-panel__section--events">
      <h4>${escapeHtml(title)}</h4>
      <ul class="autosave-panel__list">${events.map(ev => `<li>${escapeHtml(ev?.name ?? '')}</li>`).join('')}</ul>
    </section>`;
}

function renderAutosaveTrace(trace, version) {
  if (!trace) return '';
  // Prefer rich evaluated entries if present; fallback to simple steps
  const evaluated = Array.isArray(trace?.evaluated) ? trace.evaluated : null;
  let listHtml = '';
  if (evaluated) {
    listHtml = evaluated.slice(0, 50).map(entry => {
      const matched = entry?.matched ? 'matched' : 'skipped';
      const mutations = typeof entry?.mutations === 'number' ? `, mutations ${entry.mutations}` : '';
      const scope = entry?.scope ? ` <small>${escapeHtml(String(entry.scope))}</small>` : '';
      const id = escapeHtml(String(entry?.id ?? '—'));
      return `<li class="${matched}">${id}${scope}${mutations}</li>`;
    }).join('');
  } else {
    const steps = Array.isArray(trace?.steps) ? trace.steps : [];
    listHtml = steps.map(step => `<li>${escapeHtml(step?.id ?? '')}</li>`).join('');
  }
  return `
    <section class="autosave-panel__section autosave-panel__section--trace">
      <div class="autosave-panel__section-head">
        <h4>Règles évaluées</h4>
        ${version ? `<span class="autosave-panel__meta">v${escapeHtml(String(version))}</span>` : ''}
      </div>
      <ul class="autosave-panel__trace">${listHtml}</ul>
    </section>`;
}

function autosaveStatusLabel(status) {
  switch (status) {
    case 'pending': return 'En cours…';
    case 'dirty': return 'En attente';
    case 'saved': return 'Enregistré';
    case 'blocked': return 'Bloqué';
    case 'failed': return 'Erreur';
    case 'aborted': return 'Introuvable';
    default: return 'Prêt';
  }
}

// helpers moved to ./ui/format.js
