import { createDragManager } from '../../../packages/ui/drag-manager.js';
import { showToast } from '../../../packages/ui/toast.js';
import { findListLocation, findColumn, findColumnLocation, isContainer } from './domain-utils.js';
import { findItem as editorFindItem } from './editor.js';
import { moveNode } from './move-node.js';

let appRef = null;
let storeRef = null;
let sendCommandRef = null;
let dragManagerRef = null;
let dropIntent = null;
const dropIndicatorState = { element: null, kind: null, dropzone: null };
let lastDropDeniedAt = 0;

export function initDnd(ctx) {
  appRef = ctx.app ?? null;
  storeRef = ctx.store ?? null;
  sendCommandRef = ctx.sendCommand ?? null;
}

export function setupDragManager() {
  if (!appRef || !storeRef) {
    return;
  }
  if (dragManagerRef) {
    dragManagerRef.destroy();
  }
  dragManagerRef = createDragManager(appRef, {
    getData(el) {
      const type = el.dataset.type
        || (el.classList.contains('list') ? 'list'
          : (el.classList.contains('item') ? 'item'
            : (el.classList.contains('board-column') ? 'column' : 'node')));
      return {
        id: el.dataset.id || el.dataset.node || '',
        type,
        nodeId: el.dataset.node || el.dataset.id || '',
        columnId: el.dataset.column || '',
        workspaceId: el.dataset.workspace || '',
      };
    },
    onPreview: handlePreview,
    onDrop({ data }) {
      if (!data || !dropIntent || dropIntent.sourceId !== data.id) {
        notifyDropDenied();
        clearDropIndicator();
        dropIntent = null;
        return;
      }

      if (dropIntent.kind === 'item') {
        handleItemDrop(data);
      } else if (dropIntent.kind === 'list') {
        handleListDrop(data);
      } else if (dropIntent.kind === 'column') {
        handleColumnDrop(data);
      } else {
        notifyDropDenied();
      }

      clearDropIndicator();
      dropIntent = null;
    },
    getScrollContainers(_event, ctx) {
      const containers = [];
      const scroller = document.scrollingElement;
      if (scroller) containers.push(scroller);
      const main = appRef.querySelector('.board-main');
      if (main) containers.push(main);
      const columnRow = main?.querySelector('.column-row');
      if (columnRow) containers.push(columnRow);
      if (ctx?.dropzone instanceof HTMLElement) {
        containers.push(ctx.dropzone);
      }
      return containers;
    },
    autoScrollMargin: 72,
    autoScrollSpeed: 18,
    activationThreshold: 8,
    longPressDelay: 250,
  });
}

export function teardownDnd() {
  if (dragManagerRef) {
    dragManagerRef.destroy();
  }
  dragManagerRef = null;
  dropIntent = null;
  clearDropIndicator();
}

export function resetDndState() {
  dropIntent = null;
  clearDropIndicator();
}

export function moveItemWithKeyboard(itemId, delta) {
  if (!itemId || !delta || !storeRef || !sendCommandRef) {
    return;
  }
  const state = storeRef.getState();
  const info = editorFindItem(state.board, itemId);
  if (!info) {
    return;
  }
  const list = info.list;
  const order = list.order.filter(entry => entry.type === 'item').map(entry => entry.id);
  const currentIndex = order.indexOf(itemId);
  if (currentIndex === -1) {
    return;
  }
  let targetIndex = currentIndex + delta;
  targetIndex = Math.max(0, Math.min(order.length - 1, targetIndex));
  if (targetIndex === currentIndex) {
    return;
  }

  const position = targetIndex;
  moveNode({
    nodeId: itemId,
    toParentId: list.id,
    toIndex: position,
  });
}

export function moveListWithKeyboard(listId, delta) {
  if (!listId || !delta || !storeRef || !sendCommandRef) {
    return;
  }
  const state = storeRef.getState();
  const location = findListLocation(state.board, listId);
  if (!location) {
    return;
  }
  const column = findColumn(state.board, location.columnId);
  if (!column) {
    return;
  }

  const movableOrder = column.lists.map(entry => entry.id);
  const currentIndex = movableOrder.indexOf(listId);
  if (currentIndex === -1) {
    return;
  }
  let targetIndex = currentIndex + delta;
  targetIndex = Math.max(0, Math.min(movableOrder.length - 1, targetIndex));
  if (targetIndex === currentIndex) {
    return;
  }

  const orderedIds = column.lists.map(list => list.id);
  const candidateIds = orderedIds.filter(id => id !== listId);

  let beforeId;
  if (targetIndex < currentIndex) {
    beforeId = movableOrder[targetIndex];
  } else {
    beforeId = movableOrder[targetIndex + 1] ?? null;
  }

  let position;
  if (beforeId) {
    position = candidateIds.indexOf(beforeId);
    if (position === -1) {
      position = candidateIds.length;
    }
  } else {
    position = candidateIds.length;
  }

  // V3: lists are nodes; move as node
  moveNode({
    nodeId: listId,
    toParentId: location.columnId,
    toIndex: position,
  });
}

export function moveColumnWithKeyboard(columnId, delta) {
  if (!columnId || !delta || !storeRef || !sendCommandRef) {
    return;
  }
  const state = storeRef.getState();
  const location = findColumnLocation(state.board, columnId);
  if (!location) {
    return;
  }
  const workspace = state.board?.workspaces.find(ws => ws.id === location.workspaceId);
  if (!workspace) {
    return;
  }

  const order = workspace.columns.map(column => column.id);
  const currentIndex = order.indexOf(columnId);
  if (currentIndex === -1) {
    return;
  }
  let targetIndex = currentIndex + delta;
  targetIndex = Math.max(0, Math.min(order.length - 1, targetIndex));
  if (targetIndex === currentIndex) {
    return;
  }

  const candidateIds = order.filter(id => id !== columnId);
  let beforeId;
  if (targetIndex < currentIndex) {
    beforeId = order[targetIndex];
  } else {
    beforeId = order[targetIndex + 1] ?? null;
  }

  let position;
  if (beforeId) {
    position = candidateIds.indexOf(beforeId);
    if (position === -1) {
      position = candidateIds.length;
    }
  } else {
    position = candidateIds.length;
  }

  const promise = sendCommandRef('MoveColumn', {
    workspaceId: location.workspaceId,
    columnId,
    position,
  });
  if (promise?.catch) {
    promise.catch(console.error);
  }
}

function handlePreview(event) {
  if (!event) {
    return;
  }

  switch (event.type) {
    case 'start':
      clearDropIndicator();
      dropIntent = null;
      break;
    case 'end':
      clearDropIndicator();
      dropIntent = null;
      break;
    case 'leave':
      if (dropIndicatorState.dropzone === event.dropzone) {
        clearDropIndicator();
      }
      if (dropIntent && dropIntent.dropzone === event.dropzone) {
        dropIntent = null;
      }
      break;
    case 'enter':
    case 'update':
    case 'move':
      if (!event.dropzone || !event.data) {
        clearDropIndicator();
        dropIntent = null;
        return;
      }
      const dataType = event.data.type;
      if (dataType === 'column') {
        updateColumnDropPreview(event);
      } else if (dataType === 'list') {
        updateListDropPreview(event);
      } else {
        updateItemDropPreview(event);
      }
      break;
    default:
      break;
  }
}

function updateItemDropPreview(event) {
  const dropzone = event.dropzone;
  const dropzoneType = dropzone?.dataset?.dropzoneType ?? '';
  if (!dropzone || (dropzoneType !== 'item' && dropzoneType !== 'list')) {
    if (dropIntent && dropIntent.kind === 'item') {
      dropIntent = null;
    }
    clearDropIndicator();
    return;
  }

  const toParentId = dropzone.dataset.node;
  if (!toParentId) {
    clearDropIndicator();
    dropIntent = null;
    return;
  }

  if (dropzone.closest(`[data-node="${event.data.nodeId || event.data.id}"]`)) {
    clearDropIndicator();
    dropIntent = null;
    return;
  }

  const siblings = Array.from(dropzone.querySelectorAll(':scope > .list, :scope > .item'))
    .filter(el => el.dataset.node && el.dataset.node !== (event.data.nodeId || event.data.id));

  let insertBefore = null;
  for (const el of siblings) {
    const rect = el.getBoundingClientRect();
    if (event.clientY < rect.top + rect.height / 2) {
      insertBefore = el;
      break;
    }
  }

  const position = insertBefore ? siblings.findIndex(el => el === insertBefore) : siblings.length;
  const indicator = ensureDropIndicator('item');
  if (!indicator) {
    return;
  }
  const composer = dropzone.querySelector('.list-composer[data-open="true"]');
  const tail = dropzone.querySelector('.list-drop-tail');
  const fallback = composer || tail;
  placeIndicator(dropzone, insertBefore, fallback);

  dropIntent = {
    kind: 'item',
    sourceId: event.data.nodeId || event.data.id,
    toParentId,
    toIndex: position,
    dropzone,
  };
}

function updateListDropPreview(event) {
  const dropzone = event.dropzone;
  if (!dropzone || dropzone.dataset.dropzoneType !== 'list') {
    if (dropIntent && dropIntent.kind === 'list') {
      dropIntent = null;
    }
    clearDropIndicator();
    return;
  }

  const toParentId = dropzone.dataset.node;
  if (!toParentId) {
    clearDropIndicator();
    dropIntent = null;
    return;
  }

  const draggedId = event.data.nodeId || event.data.id;
  if (!draggedId) {
    clearDropIndicator();
    dropIntent = null;
    return;
  }

  if (dropzone.closest(`[data-node="${draggedId}"]`)) {
    clearDropIndicator();
    dropIntent = null;
    return;
  }

  const state = storeRef.getState();
  const board = state.board;
  if (!board || !board.nodes) {
    clearDropIndicator();
    dropIntent = null;
    return;
  }

  const parentNode = board.nodes[toParentId] || null;
  if (!parentNode || !isContainer(parentNode)) {
    clearDropIndicator();
    dropIntent = null;
    return;
  }

  const orderedIds = Array.isArray(parentNode.children) ? parentNode.children.slice() : [];
  const listIds = orderedIds
    .map(id => board.nodes[id])
    .filter(isContainer)
    .map(node => node.id)
    .filter(id => id !== draggedId);

  const candidateElements = listIds
    .map(id => dropzone.querySelector(`article.list[data-node="${id}"]`))
    .filter(Boolean);

  let beforeId = null;
  for (const el of candidateElements) {
    const rect = el.getBoundingClientRect();
    if (event.clientY < rect.top + rect.height / 2) {
      beforeId = el.dataset.list ?? null;
      break;
    }
  }

  let position;
  if (beforeId) {
    const candidateIndex = listIds.indexOf(beforeId);
    position = candidateIndex === -1 ? listIds.length : candidateIndex;
  } else {
    position = listIds.length;
  }

  const afterIds = listIds.slice(position);
  let referenceEl = null;
  if (beforeId) {
    referenceEl = dropzone.querySelector(`article.list[data-node="${beforeId}"]`);
  } else {
    for (const id of afterIds) {
      const candidate = dropzone.querySelector(`article.list[data-node="${id}"]`);
      if (candidate) {
        referenceEl = candidate;
        break;
      }
    }
  }

  const indicator = ensureDropIndicator('list');
  if (!indicator) {
    return;
  }
  const composer = dropzone.querySelector('.list-composer[data-open="true"]');
  const tail = dropzone.querySelector('.list-drop-tail');
  const fallback = composer || tail;
  placeIndicator(dropzone, referenceEl, fallback);

  dropIntent = {
    kind: 'list',
    sourceId: draggedId,
    toParentId,
    position,
    dropzone,
  };
}

function updateColumnDropPreview(event) {
  const dropzone = event.dropzone;
  if (!dropzone || dropzone.dataset.dropzoneType !== 'column') {
    if (dropIntent && dropIntent.kind === 'column') {
      dropIntent = null;
    }
    clearDropIndicator();
    return;
  }

  const workspaceId = dropzone.dataset.workspace;
  if (!workspaceId || event.data.workspaceId !== workspaceId) {
    clearDropIndicator();
    dropIntent = null;
    return;
  }

  const columnsDom = Array.from(dropzone.querySelectorAll('section.board-column[data-column]')).filter(el => el.dataset.column);
  const filteredDom = columnsDom.filter(el => el.dataset.column !== event.data.id);

  let insertBeforeEl = null;
  for (const el of filteredDom) {
    const rect = el.getBoundingClientRect();
    if (event.clientX < rect.left + rect.width / 2) {
      insertBeforeEl = el;
      break;
    }
  }

  const state = storeRef.getState();
  const workspace = state.board?.workspaces.find(ws => ws.id === workspaceId);
  if (!workspace) {
    clearDropIndicator();
    dropIntent = null;
    return;
  }

  const columnIds = workspace.columns.map(col => col.id).filter(id => id !== event.data.id);
  let position = columnIds.length;
  if (insertBeforeEl) {
    const idx = columnIds.indexOf(insertBeforeEl.dataset.column);
    position = idx === -1 ? columnIds.length : idx;
  }

  const indicator = ensureDropIndicator('column');
  if (!indicator) {
    return;
  }
  const fallback = dropzone.querySelector('[data-dnd-ignore="true"]');
  placeIndicator(dropzone, insertBeforeEl, fallback);

  dropIntent = {
    kind: 'column',
    sourceId: event.data.id,
    workspaceId,
    position,
    dropzone,
  };
}

function handleItemDrop(data) {
  if (!storeRef || !sendCommandRef) {
    return;
  }
  if (!dropIntent || dropIntent.kind !== 'item') {
    return;
  }
  const nodeId = data.nodeId || data.id;
  if (!nodeId) return;
  moveNode({ nodeId, toParentId: dropIntent.toParentId, toIndex: dropIntent.toIndex });
}

function handleListDrop(data) {
  if (!storeRef || !sendCommandRef) {
    return;
  }
  if (!dropIntent || dropIntent.kind !== 'list') {
    return;
  }

  const state = storeRef.getState();
  const board = state.board;
  if (!board || !board.nodes) {
    return;
  }

  const nodeId = data.nodeId || data.id;
  if (!nodeId) {
    return;
  }

  const node = board.nodes[nodeId] || null;
  if (!node || !isContainer(node)) {
    return;
  }

  const currentParentId = node.parentId || null;
  if (currentParentId === dropIntent.toParentId) {
    const siblings = Array.isArray(board.nodes[currentParentId]?.children)
      ? board.nodes[currentParentId].children
      : [];
    const containerSiblings = siblings
      .map(id => board.nodes[id] || null)
      .filter(isContainer)
      .map(n => n.id);
    const currentIndex = containerSiblings.indexOf(nodeId);
    if (currentIndex === dropIntent.position) {
      return;
    }
  }

  moveNode({
    nodeId,
    toParentId: dropIntent.toParentId,
    toIndex: dropIntent.position,
  });
}

function handleColumnDrop(data) {
  if (!storeRef || !sendCommandRef) {
    return;
  }
  const state = storeRef.getState();
  const location = findColumnLocation(state.board, data.id);
  if (!location || !dropIntent || dropIntent.kind !== 'column') {
    return;
  }

  if (location.workspaceId !== dropIntent.workspaceId) {
    return;
  }
  if (location.index === dropIntent.position) {
    return;
  }

  const outcome = sendCommandRef('MoveColumn', {
    workspaceId: dropIntent.workspaceId,
    columnId: data.id,
    position: dropIntent.position,
  });
  handleCommandRejection(outcome, 'Réorganisation de colonne refusée');
}

function ensureDropIndicator(kind) {
  if (dropIndicatorState.kind !== kind || !dropIndicatorState.element) {
    if (dropIndicatorState.element) {
      dropIndicatorState.element.remove();
    }
    const el = document.createElement('div');
    el.className = `drop-indicator drop-indicator--${kind}`;
    dropIndicatorState.element = el;
    dropIndicatorState.kind = kind;
  }
  return dropIndicatorState.element;
}

function placeIndicator(dropzone, reference, fallback) {
  if (!dropIndicatorState.element) {
    return;
  }
  let target = reference;
  if (!target && fallback instanceof HTMLElement) {
    target = fallback;
  }

  if (target && target.parentElement === dropzone) {
    if (dropIndicatorState.element.nextElementSibling !== target) {
      dropzone.insertBefore(dropIndicatorState.element, target);
    }
  } else {
    const shouldAppend =
      dropIndicatorState.element.parentElement !== dropzone ||
      dropIndicatorState.element.nextElementSibling !== null;
    if (shouldAppend) {
      dropzone.appendChild(dropIndicatorState.element);
    }
  }
  dropIndicatorState.dropzone = dropzone;
}

function clearDropIndicator() {
  if (dropIndicatorState.element) {
    dropIndicatorState.element.remove();
  }
  dropIndicatorState.element = null;
  dropIndicatorState.kind = null;
  dropIndicatorState.dropzone = null;
}

function notifyDropDenied() {
  const now = Date.now();
  if (now - lastDropDeniedAt < 1500) {
    return;
  }
  lastDropDeniedAt = now;
  showToast('Déplacement impossible à cet emplacement', { kind: 'warning' });
}

function handleCommandRejection(promise, fallbackMessage) {
  if (!promise || typeof promise.catch !== 'function') {
    return;
  }
  promise.catch(error => {
    console.error('DND_COMMAND_FAILED', error);
    const message = resolveCommandErrorMessage(error, fallbackMessage);
    showToast(message, { kind: 'warning' });
  });
}

function resolveCommandErrorMessage(error, fallbackMessage = 'Action refusée') {
  if (!error) {
    return fallbackMessage;
  }
  const payloadMessage = error?.payload?.error?.message
    || error?.payload?.message
    || error?.payload?.error
    || null;
  if (typeof payloadMessage === 'string' && payloadMessage.trim() !== '') {
    return payloadMessage;
  }
  const direct = typeof error?.message === 'string' ? error.message : null;
  if (direct && direct !== 'COMMAND_FAILED') {
    return direct;
  }
  return fallbackMessage;
}
