import { exportBoard, importBoard } from '../../../packages/services/boards.js';
import { normalizeFilterTags } from './filters.js';
import { mapAutosaveStatus as autoMapAutosaveStatus } from './autosave.js';
import { formatDateTime } from './ui/format.js';
import { replaceBoardQuery, refreshBoardsList } from './board-structure.js';
import { clearTagCaches as tagsClearTagCaches } from './tags.js';
import { resetDndState } from './dnd.js';

export function createImportExportController({ store, app, getCsrfToken, showToast, onBoardHydrated = () => {} }) {
  async function openExportDialog() {
    const state = store.getState();
    if (!state.currentBoardId) {
      showToast('Aucun board sélectionné', { kind: 'warning' });
      return;
    }

    store.setState(prev => ({
      ...prev,
      dialog: { kind: 'export', status: 'loading' },
      autosavePanelOpen: false,
    }));

    try {
      const board = await exportBoard(state.currentBoardId);
      store.setState(prev => ({
        ...prev,
        dialog: { kind: 'export', status: 'ready', board },
      }));
    } catch (error) {
      console.error(error);
      store.setState(prev => ({
        ...prev,
        dialog: { kind: 'export', status: 'error', message: error?.message ?? 'Export impossible' },
      }));
      showToast('Export impossible', { kind: 'error' });
    }
  }

  function openImportDialog() {
    const state = store.getState();
    const currentBoard = state.boards.find(b => String(b.id) === state.currentBoardId);
    const existing = state.dialog?.kind === 'import' ? state.dialog : null;
    const defaultNewTitle = `Import ${new Date().toLocaleDateString('fr-FR')}`;

    store.setState(prev => ({
      ...prev,
      dialog: {
        kind: 'import',
        mode: existing?.mode ?? 'replace',
        busy: false,
        error: null,
        form: existing?.form ?? {
          json: '',
          replaceTitle: currentBoard?.title ?? '',
          newTitle: defaultNewTitle,
          workspaceTitle: 'Workspace',
        },
      },
      autosavePanelOpen: false,
    }));
  }

  function setImportMode(mode) {
    store.setState(prev => {
      if (!prev.dialog || prev.dialog.kind !== 'import') {
        return prev;
      }
      if (prev.dialog.busy || prev.dialog.mode === mode) {
        return prev;
      }
      const form = readImportForm(prev.dialog.form);
      return {
        ...prev,
        dialog: {
          ...prev.dialog,
          mode,
          error: null,
          form,
        },
      };
    });
  }

  async function submitImportDialog() {
    const state = store.getState();
    const dialog = state.dialog;
    if (!dialog || dialog.kind !== 'import' || dialog.busy) {
      return;
    }

    const mode = dialog.mode ?? 'replace';
    const form = readImportForm(dialog.form);
    const boardId = state.currentBoardId;

    if (mode !== 'new' && mode !== 'reset' && !boardId) {
      showToast('Aucun board sélectionné', { kind: 'warning' });
      return;
    }

    let importData = null;
    if (mode !== 'reset') {
      if (!form.json || form.json.trim() === '') {
        updateImportDialogState({ busy: false, error: 'Veuillez coller le JSON exporté avant de continuer.', form });
        return;
      }
      const parsed = parseImportBundle(form.json);
      if (!parsed.ok) {
        updateImportDialogState({ busy: false, error: parsed.error, form });
        return;
      }
      importData = parsed;
    }

    updateImportDialogState({ busy: true, error: null, form });

    try {
      if (mode === 'replace') {
        if (!importData) {
          throw new Error('IMPORT_DATA_MISSING');
        }
        const payload = {
          boardId,
          state: importData.state,
          revision: state.boardRevision ?? state.boardMeta?.revision ?? null,
        };
        if (Array.isArray(importData.rules)) {
          payload.rules = importData.rules;
        }
        const title = form.replaceTitle?.trim();
        if (title) {
          payload.title = title;
        }
        const result = await importBoard(payload, getCsrfToken());
        await handleImportSuccess(result, { refreshBoards: true, newBoard: false });
      } else if (mode === 'new') {
        if (!importData) {
          throw new Error('IMPORT_DATA_MISSING');
        }
        const payload = {
          title: form.newTitle?.trim() || importData?.title?.trim() || `Board importé ${new Date().toLocaleTimeString('fr-FR')}`,
          state: importData.state,
        };
        if (Array.isArray(importData.rules)) {
          payload.rules = importData.rules;
        }
        const result = await importBoard(payload, getCsrfToken());
        await handleImportSuccess(result, { refreshBoards: true, newBoard: true });
      } else {
        const payload = {
          boardId,
          reset: true,
          workspaceTitle: form.workspaceTitle?.trim() || 'Workspace',
          revision: state.boardRevision ?? state.boardMeta?.revision ?? null,
        };
        const result = await importBoard(payload, getCsrfToken());
        await handleImportSuccess(result, { refreshBoards: true, newBoard: false });
      }
    } catch (error) {
      console.error(error);
      const message = error?.payload?.message ?? error?.message ?? 'Import impossible';
      updateImportDialogState({ busy: false, error: message, form });
      showToast('Import impossible', { kind: 'error' });
    }
  }

  async function handleImportSuccess(result, { refreshBoards, newBoard }) {
    const payload = result?.board ?? null;

    if (payload) {
      onBoardHydrated(payload);
      const updatedAt = typeof payload.updatedAt === 'number' ? payload.updatedAt : Date.now();
      const createdAt = typeof payload.createdAt === 'number' ? payload.createdAt : updatedAt;
      const history = Array.isArray(payload.history) ? payload.history : [];
      const autosave = autoMapAutosaveStatus({
        status: 'saved',
        savedAt: new Date().toISOString(),
        updatedAt,
        revision: payload.revision ?? updatedAt,
        history,
      });
      const importedFilter = normalizeFilterTags(payload.state?.tagFilter?.selected ?? []);

      tagsClearTagCaches();
      resetDndState();

      store.setState(prev => ({
        ...prev,
        currentBoardId: String(payload.id),
        board: payload.state,
        boardMeta: { updatedAt, revision: payload.revision ?? updatedAt, history },
        boardRevision: payload.revision ?? updatedAt,
        autosave,
        autosavePanelOpen: false,
        dialog: null,
        lastCommand: null,
        trace: null,
        lastEvents: { pre: [], post: [] },
        boards: newBoard
          ? [{
              id: payload.id,
              title: payload.title ?? `Board ${payload.id}`,
              updated_at: updatedAt,
              created_at: createdAt,
            }, ...prev.boards.filter(b => String(b.id) !== String(payload.id))]
          : prev.boards.map(board =>
              String(board.id) === String(payload.id)
                ? { ...board, title: payload.title ?? board.title, updated_at: updatedAt }
                : board
            ),
        filter: { tags: importedFilter },
      }));

      if (newBoard) {
        replaceBoardQuery(payload.id);
      }
      showToast('Board importé avec succès', { kind: 'success' });
    } else {
      store.setState(prev => ({
        ...prev,
        dialog: null,
      }));
    }

    if (refreshBoards) {
      try {
        await refreshBoardsList();
      } catch (error) {
        console.error(error);
      }
    }
  }

  function readImportForm(fallback = {}) {
    if (!app) {
      return {
        json: fallback.json ?? '',
        replaceTitle: fallback.replaceTitle ?? '',
        newTitle: fallback.newTitle ?? '',
        workspaceTitle: fallback.workspaceTitle ?? 'Workspace',
      };
    }
    const jsonField = app.querySelector('[data-ref="import-json"]');
    const replaceTitleField = app.querySelector('[data-ref="import-replace-title"]');
    const newTitleField = app.querySelector('[data-ref="import-new-title"]');
    const workspaceField = app.querySelector('[data-ref="import-workspace"]');
    return {
      json: jsonField ? jsonField.value : (fallback.json ?? ''),
      replaceTitle: replaceTitleField ? replaceTitleField.value : (fallback.replaceTitle ?? ''),
      newTitle: newTitleField ? newTitleField.value : (fallback.newTitle ?? ''),
      workspaceTitle: workspaceField ? workspaceField.value : (fallback.workspaceTitle ?? 'Workspace'),
    };
  }

  function copyExportJson() {
    if (!app) return;
    const field = app.querySelector('[data-ref="export-json"]');
    if (!(field instanceof HTMLTextAreaElement)) return;
    const value = field.value;
    navigator.clipboard?.writeText(value).then(() => {
      showToast('Export copié dans le presse-papier', { kind: 'success' });
    }).catch(() => {
      field.select();
      try {
        const ok = document.execCommand('copy');
        if (ok) {
          showToast('Export copié dans le presse-papier', { kind: 'success' });
        } else {
          throw new Error('execCommand failed');
        }
      } catch (error) {
        console.error(error);
        showToast('Impossible de copier le JSON', { kind: 'error' });
      } finally {
        field.setSelectionRange(field.value.length, field.value.length);
        field.blur();
      }
    });
  }

  function downloadExportJson() {
    const field = app?.querySelector('[data-ref="export-json"]');
    if (!(field instanceof HTMLTextAreaElement)) return;
    const value = field.value ?? '';
    const blob = new Blob([value], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    const state = store.getState();
    const board = state.boards.find(b => String(b.id) === state.currentBoardId);
    const filename = `${slugify(board?.title ?? 'board') || 'board'}-${Date.now()}.json`;
    link.href = url;
    link.download = filename;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
  }

  function updateImportDialogState({ busy, error, form }) {
    store.setState(prev => ({
      ...prev,
      dialog: prev.dialog?.kind === 'import'
        ? { ...prev.dialog, busy, error, form }
        : prev.dialog,
    }));
  }

  return {
    openExportDialog,
    openImportDialog,
    setImportMode,
    submitImportDialog,
    readImportForm,
    copyExportJson,
    downloadExportJson,
  };
}

export function computeImportPreview(json) {
  if (!json || json.trim() === '') {
    return { summary: null, error: null };
  }
  const result = parseImportBundle(json);
  if (!result.ok) {
    return { summary: null, error: result.error };
  }
  return { summary: result.summary, error: null };
}

export function parseImportBundle(json) {
  let payload;
  try {
    payload = JSON.parse(json);
  } catch (error) {
    return { ok: false, error: 'JSON invalide. Vérifiez le contenu collé.' };
  }

  if (!payload || typeof payload !== 'object') {
    return { ok: false, error: "Format d'export inconnu." };
  }

  let state = null;
  let title = null;
  let revision = null;
  let history = [];
  let rules = null;

  if (typeof payload.version === 'number' && payload.state) {
    state = payload.state;
    title = payload.board?.title ?? null;
    revision = payload.board?.revision ?? payload.board?.updatedAt ?? null;
    history = Array.isArray(payload.board?.history) ? payload.board.history : [];
    if (Array.isArray(payload.rules)) {
      rules = payload.rules;
    } else if (Array.isArray(payload.board?.rules)) {
      rules = payload.board.rules;
    }
  } else if (payload.state && typeof payload.state === 'object') {
    state = payload.state;
    title = payload.board?.title ?? null;
    revision = payload.board?.revision ?? payload.board?.updatedAt ?? null;
    history = Array.isArray(payload.board?.history) ? payload.board.history : [];
    if (Array.isArray(payload.rules)) {
      rules = payload.rules;
    } else if (Array.isArray(payload.board?.rules)) {
      rules = payload.board.rules;
    }
  } else {
    state = payload;
  }

  if (!state || typeof state !== 'object') {
    return { ok: false, error: 'Le JSON ne contient pas un board valide.' };
  }

  const summary = summarizeBoardState(state, revision);
  if (!summary) {
    return { ok: false, error: 'Le JSON ne contient pas un board valide.' };
  }
  if (Array.isArray(rules)) {
    summary.ruleCount = rules.length;
  }

  return {
    ok: true,
    state,
    title,
    revision,
    summary,
    history,
    rules,
  };
}

export function summarizeBoardState(state, revision = null) {
  if (!state || typeof state !== 'object') return null;
  // v3 summary based on nodes map
  if (state.nodes && typeof state.nodes === 'object') {
    let listCount = 0;
    let itemCount = 0;
    for (const id in state.nodes) {
      const n = state.nodes[id];
      const tags = Array.isArray(n?.tags) ? n.tags : [];
      const isList = tags.some(t => t && (t.k === 'type' || t.key === 'type') && (t.v === 'list' || t.value === 'list'));
      if (isList) listCount += 1; else itemCount += 1;
    }
    // root is counted above; keep as-is
    return {
      title: state.title ?? null,
      nodeCount: Object.keys(state.nodes).length,
      listCount,
      itemCount,
      revision,
    };
  }
  // legacy v2 fallback (for older bundles)
  if (!Array.isArray(state.workspaces)) return null;
  const workspaces = state.workspaces;
  let columnCount = 0;
  let listCount = 0;
  let itemCount = 0;
  const visitList = (list) => {
    if (!list || typeof list !== 'object') return;
    listCount += 1;
    if (Array.isArray(list.items)) itemCount += list.items.length;
    if (Array.isArray(list.lists)) list.lists.forEach(visitList);
  };
  workspaces.forEach((workspace) => {
    const columns = Array.isArray(workspace?.columns) ? workspace.columns : [];
    columnCount += columns.length;
    columns.forEach((column) => {
      const lists = Array.isArray(column?.lists) ? column.lists : [];
      lists.forEach(visitList);
    });
  });
  return { title: state.title ?? null, workspaceCount: workspaces.length, columnCount, listCount, itemCount, revision };
}

export function createBoardExportBundle(board) {
  const meta = board ?? {};
  const revision = typeof meta.revision === 'number'
    ? meta.revision
    : (typeof meta.updatedAt === 'number' ? meta.updatedAt : null);
  return {
    version: 3,
    exportedAt: meta.exportedAt ?? new Date().toISOString(),
    board: {
      id: meta.id ?? null,
      title: meta.title ?? null,
      revision,
      updatedAt: meta.updatedAt ?? null,
      createdAt: meta.createdAt ?? null,
      history: Array.isArray(meta.history) ? meta.history : [],
    },
    state: meta.state ?? null,
    rules: Array.isArray(meta.rules) ? meta.rules : [],
  };
}

function slugify(str) {
  const normalized = String(str ?? '')
    .normalize('NFD')
    .replace(/[\u0300-\u036f]/g, '')
    .trim()
    .toLowerCase();
  const value = normalized.replace(/[^a-z0-9]+/g, '-').replace(/^-+|-+$/g, '');
  return value;
}
// date helpers moved to ./ui/format.js
