import { showToast } from '../../../packages/ui/toast.js';
import { findNode, isContainer } from './domain-utils.js';
import { requestSelection } from './selection.js';

let storeRef = null;
let sendCommandRef = null;

export function initMoveNode({ store, sendCommand }) {
  storeRef = store ?? null;
  sendCommandRef = sendCommand ?? null;
}

export function moveNode({ nodeId, toParentId, toIndex }) {
  if (!nodeId || !toParentId || typeof toIndex !== 'number' || !sendCommandRef) {
    showToast('Déplacement impossible à cet emplacement', { kind: 'warning' });
    return null;
  }
  const outcome = sendCommandRef('MoveNode', { nodeId, toParentId, toIndex });
  handleCommandRejection(outcome, 'Déplacement impossible à cet emplacement');
  return outcome;
}

export function computeInsertion({ parentId, anchorNodeId = null, mode = 'prepend', movingNodeId = null }) {
  if (!storeRef) return null;
  const state = storeRef.getState();
  const board = state?.board ?? null;
  const parent = board?.nodes?.[parentId] ?? null;
  if (!parent || !isContainer(parent)) {
    return null;
  }
  const children = Array.isArray(parent.children) ? parent.children.slice() : [];

  if (mode === 'prepend') {
    return { toParentId: parentId, toIndex: 0 };
  }

  if (mode === 'before-anchor') {
    if (!anchorNodeId) return null;
    let idx = children.indexOf(anchorNodeId);
    if (idx === -1) return null;
    if (movingNodeId) {
      const currentIdx = children.indexOf(movingNodeId);
      if (currentIdx !== -1 && currentIdx < idx) {
        idx -= 1;
      }
    }
    return { toParentId: parentId, toIndex: Math.max(0, idx) };
  }

  return null;
}

export async function moveNodeViaSelection(nodeId) {
  if (!storeRef || !nodeId) {
    return;
  }
  const state = storeRef.getState();
  const board = state?.board ?? null;
  const boardId = state?.currentBoardId ?? null;
  const movingNode = findNode(board, nodeId);
  if (!boardId || !movingNode) {
    showToast('Élément introuvable', { kind: 'warning' });
    return;
  }

  try {
    const [target] = await requestSelection({
      label: 'Choisissez la cible du déplacement',
      allowedShapes: ['container', 'leaf'],
      multi: false,
    });
    if (!target) return;
    if (String(target.boardId) !== String(boardId)) {
      showToast('Déplacement limité au board courant', { kind: 'warning' });
      return;
    }
    if (target.nodeId === nodeId) {
      showToast('Sélectionne une autre position', { kind: 'warning' });
      return;
    }
    const targetNode = findNode(board, target.nodeId);
    if (!targetNode) {
      showToast('Cible introuvable', { kind: 'warning' });
      return;
    }

    let insertion = null;
    if ((targetNode?.sys?.shape ?? null) === 'container') {
      insertion = computeInsertion({
        parentId: target.nodeId,
        mode: 'prepend',
        movingNodeId: nodeId,
      });
    } else {
      const parentId = targetNode.parentId || null;
      insertion = computeInsertion({
        parentId,
        anchorNodeId: target.nodeId,
        mode: 'before-anchor',
        movingNodeId: nodeId,
      });
    }

    if (!insertion) {
      showToast('Cible de déplacement invalide', { kind: 'warning' });
      return;
    }
    if (wouldCreateCycle(board, nodeId, insertion.toParentId)) {
      showToast('Impossible de déplacer dans son propre sous-arbre', { kind: 'warning' });
      return;
    }

    const currentParentId = movingNode.parentId || null;
    if (currentParentId === insertion.toParentId) {
      const siblings = Array.isArray(board.nodes[currentParentId]?.children)
        ? board.nodes[currentParentId].children
        : [];
      const filtered = siblings.filter(id => id !== nodeId);
      let desiredIndex = insertion.toIndex;
      if (desiredIndex > filtered.length) {
        desiredIndex = filtered.length;
      }
      const currentIndex = siblings.indexOf(nodeId);
      if (currentIndex === desiredIndex) {
        showToast('Déjà à cet emplacement', { kind: 'info' });
        return;
      }
    }

    moveNode({ nodeId, ...insertion });
  } catch (error) {
    if (error?.name === 'SelectionCancelled') return;
    console.error('MOVE_NODE_VIA_SELECTION_FAILED', error);
    showToast('Déplacement annulé ou invalide', { kind: 'warning' });
  }
}

function wouldCreateCycle(board, movingNodeId, candidateParentId) {
  if (!board || !movingNodeId || !candidateParentId) return false;
  let cursor = candidateParentId;
  const nodes = board.nodes || {};
  while (cursor) {
    if (cursor === movingNodeId) return true;
    const next = nodes[cursor]?.parentId || null;
    if (!next || next === cursor) break;
    cursor = next;
  }
  return false;
}

function handleCommandRejection(promise, fallbackMessage) {
  if (!promise || typeof promise.catch !== 'function') {
    return;
  }
  promise.catch(error => {
    console.error('MOVE_NODE_COMMAND_FAILED', error);
    const message = resolveCommandErrorMessage(error, fallbackMessage);
    showToast(message, { kind: 'warning' });
  });
}

function resolveCommandErrorMessage(error, fallbackMessage = 'Action refusée') {
  if (!error) {
    return fallbackMessage;
  }
  const payloadMessage = error?.payload?.error?.message
    || error?.payload?.message
    || error?.payload?.error
    || null;
  if (typeof payloadMessage === 'string' && payloadMessage.trim() !== '') {
    return payloadMessage;
  }
  const direct = typeof error?.message === 'string' ? error.message : null;
  if (direct && direct !== 'COMMAND_FAILED') {
    return direct;
  }
  return fallbackMessage;
}
