import { showToast } from '../../../packages/ui/toast.js';
import { requestJson } from '../../../packages/services/http.js';

let store = null;
let csrfProvider = null;
let sendCommandRef = null;

let categoriesMeta = new Map();
let categoriesLoaded = false;
let categoriesLoading = null;
let applyingCategoryMeta = false;

let currentPickerState = null;

const allowedTagCache = new Map();
const ruleExplanationCache = new Map();
let ruleExplainAvailable = null;
let tagPickerOutsideHandler = null;
let tagPickerKeydownHandler = null;

export function initTags(ctx) {
  store = ctx.store;
  csrfProvider = ctx.getCsrfToken || (() => '');
  sendCommandRef = ctx.sendCommand || null;
  loadCategoriesMeta().catch(() => {});
}

export async function addTag(scope, targetId, anchor) {
  const state = store.getState();
  const node = state.board?.nodes?.[targetId];
  const currentTags = Array.isArray(node?.tags) ? node.tags : [];
  await showTagPicker(anchor, scope, targetId, currentTags);
}

export async function removeTag(scope, targetId, kind, key, options = {}) {
  const closePicker = options?.closePicker !== false;
  if (closePicker) {
    hideTagPicker();
  }
  const res = await send('RemoveTagV3', { nodeId: targetId, key });
  if (res) invalidateAllowedTagCache(scope, targetId);
  if (res && !closePicker) {
    await refreshTagPicker(scope, targetId);
  }
  return res;
}

export async function resolveRuleExplanation(reasonId) {
  if (!reasonId) return null;
  if (ruleExplanationCache.has(reasonId)) return ruleExplanationCache.get(reasonId);
  if (ruleExplainAvailable === false) { ruleExplanationCache.set(reasonId, null); return null; }
  try {
    const payload = await requestJson(`/api/rules/explain?reasonId=${encodeURIComponent(reasonId)}`, { method: 'GET' });
    const message = typeof payload?.message === 'string' ? payload.message.trim() : '';
    const resolved = message !== '' ? message : null;
    ruleExplainAvailable = true;
    ruleExplanationCache.set(reasonId, resolved);
    return resolved;
  } catch {
    if (ruleExplainAvailable === null) ruleExplainAvailable = false;
    ruleExplanationCache.set(reasonId, null);
    return null;
  }
}

async function showTagPicker(_anchor, scope, targetId, currentTags = [], config = {}) {
  await loadCategoriesMeta().catch(() => {});
  const state = store.getState();
  const node = state.board?.nodes?.[targetId] ?? null;

  const previous = currentPickerState && currentPickerState.scope === scope && currentPickerState.targetId === targetId
    ? currentPickerState
    : null;

  const searchValue = typeof config.searchValue === 'string'
    ? config.searchValue
    : (previous?.searchValue ?? '');
  const optionsScrollTop = typeof config.optionsScrollTop === 'number'
    ? config.optionsScrollTop
    : (previous?.optionsScrollTop ?? 0);
  const sidebarScrollTop = typeof config.sidebarScrollTop === 'number'
    ? config.sidebarScrollTop
    : (previous?.sidebarScrollTop ?? 0);
  hideTagPicker({ clearContext: false });

  let stateRecord = null;

  const existingKeys = buildExistingTagKeySet(currentTags);
  let options = [];
  const appendUnique = (source) => {
    if (!Array.isArray(source)) return;
    const existing = new Set(options.map(opt => `${opt?.kind ?? 'user'}:${opt?.key ?? ''}`));
    source.forEach(option => {
      const key = `${option?.kind ?? 'user'}:${option?.key ?? ''}`;
      if (!key || existing.has(key)) return;
      options.push(option);
      existing.add(key);
    });
  };
  try {
    const allowed = await fetchAllowedTags(scope, targetId);
    if (Array.isArray(allowed)) {
      options = allowed.slice();
    }
  } catch (error) {
    console.error('ALLOWED_TAGS_FETCH_FAILED', error);
  }
  appendUnique(buildTagOptions(state, currentTags));

  const systemGroups = buildSystemTagGroups(currentTags, state, node);
  const normalizedOptions = normalizeTagOptions(options, state, existingKeys);

  const picker = document.createElement('div');
  picker.className = 'tag-picker';
  picker.setAttribute('role', 'dialog');
  picker.dataset.scope = scope;
  picker.dataset.targetId = targetId;
  picker.addEventListener('click', e => e.stopPropagation());

  const header = document.createElement('header');
  header.className = 'tag-picker__header';
  const title = document.createElement('h3');
  title.className = 'tag-picker__title';
  title.textContent = 'Ajouter un tag';
  const closeBtn = document.createElement('button');
  closeBtn.type = 'button';
  closeBtn.className = 'tag-picker__close';
  closeBtn.setAttribute('aria-label', 'Fermer');
  closeBtn.textContent = '×';
  closeBtn.addEventListener('click', () => hideTagPicker());
  header.append(title, closeBtn);
  picker.appendChild(header);

  const searchInput = document.createElement('input');
  searchInput.type = 'search';
  searchInput.className = 'tag-picker__search';
  searchInput.placeholder = 'Rechercher un tag';
  picker.appendChild(searchInput);

  const content = document.createElement('div');
  content.className = 'tag-picker__content';
  picker.appendChild(content);

  const layout = document.createElement('div');
  layout.className = 'tag-picker__layout';
  content.appendChild(layout);

  const optionsPane = document.createElement('div');
  optionsPane.className = 'tag-picker__options';
  layout.appendChild(optionsPane);

  const emptyMessage = document.createElement('p');
  emptyMessage.className = 'tag-picker__empty';
  emptyMessage.textContent = 'Aucun tag disponible';
  optionsPane.appendChild(emptyMessage);

  const sections = groupOptionsBySection(normalizedOptions);
  const sectionRecords = [];
  const optionEntries = [];

  sections.forEach(section => {
    const sectionEl = document.createElement('section');
    sectionEl.className = 'tag-picker__section';
    const heading = document.createElement('h4');
    heading.className = 'tag-picker__section-title';
    heading.textContent = section.name;
    sectionEl.append(heading);
    const list = document.createElement('div');
    list.className = 'tag-picker__list';
    sectionEl.append(list);
    const record = { element: sectionEl, buttons: [] };
    sectionRecords.push(record);
    section.options.forEach(option => {
      const button = createTagOptionButton(option, scope, targetId);
      record.buttons.push(button);
      optionEntries.push({ button, option, section: section.name });
      list.append(button);
    });
    optionsPane.append(sectionEl);
  });

  const createForm = document.createElement('form');
  createForm.className = 'tag-picker__create';
  createForm.innerHTML = `
    <label class="tag-picker__create-label" for="tag-picker-create-input">Créer un tag utilisateur</label>
    <div class="tag-picker__create-row">
      <input id="tag-picker-create-input" class="tag-picker__input" type="text" name="label" placeholder="Nom du tag" autocomplete="off" />
      <button type="submit" class="tag-picker__submit btn primary">Ajouter</button>
    </div>
  `;
  const createInput = createForm.querySelector('#tag-picker-create-input');
  createForm.addEventListener('submit', async (e) => {
    e.preventDefault();
    const rawLabel = (createInput?.value ?? '').trim();
    if (!rawLabel) {
      createInput?.focus();
      return;
    }
    const key = slugify(rawLabel);
    if (!key) {
      showToast('Nom de tag invalide', { kind: 'error' });
      createInput?.focus();
      return;
    }
    try {
      await addTagCommand(scope, targetId, { kind: 'user', key, label: rawLabel });
      hideTagPicker();
    } catch (err) {
      console.error(err);
      showToast('Impossible d\'ajouter le tag', { kind: 'error' });
    }
  });

  optionsPane.append(createForm);

  let sidebar = null;
  if (systemGroups.length > 0) {
    sidebar = renderSystemTagsSidebar(systemGroups, scope, targetId, {
      onRemoved: () => {
        if (stateRecord) {
          stateRecord.sidebarScrollTop = sidebar.scrollTop;
        }
      },
    });
    layout.appendChild(sidebar);
  }

  document.body.appendChild(picker);
  picker.style.left = '0px';
  picker.style.top = '0px';

  const margin = 12;
  const vw = window.innerWidth || document.documentElement.clientWidth || 1024;
  const vh = window.innerHeight || document.documentElement.clientHeight || 768;

  let pr = picker.getBoundingClientRect();
  const maxWidth = Math.max(320, vw - margin * 2);
  if (pr.width > maxWidth) {
    picker.style.width = `${maxWidth}px`;
    picker.style.maxWidth = `${maxWidth}px`;
    pr = picker.getBoundingClientRect();
  }
  const maxHeight = Math.max(260, vh - margin * 2);
  if (pr.height > maxHeight) {
    picker.style.maxHeight = `${maxHeight}px`;
    pr = picker.getBoundingClientRect();
  }

  let left = Math.round((vw - pr.width) / 2);
  let top = Math.round((vh - pr.height) / 2);
  left = Math.min(Math.max(left, margin), Math.max(margin, vw - pr.width - margin));
  top = Math.min(Math.max(top, margin), Math.max(margin, vh - pr.height - margin));

  picker.style.left = `${left}px`;
  picker.style.top = `${top}px`;

  searchInput.value = searchValue;
  setTimeout(() => (searchInput.value ? searchInput : (createInput ?? searchInput)).focus(), 0);

  tagPickerOutsideHandler = event => { if (!picker.contains(event.target)) hideTagPicker(); };
  document.addEventListener('click', tagPickerOutsideHandler, true);
  tagPickerKeydownHandler = event => { if (event.key === 'Escape') hideTagPicker(); };
  document.addEventListener('keydown', tagPickerKeydownHandler, true);

  stateRecord = {
    scope,
    targetId,
    searchValue: searchInput.value,
    optionsScrollTop,
    sidebarScrollTop,
  };
  currentPickerState = stateRecord;

  const updateVisibility = () => {
    const query = searchInput.value.toLowerCase().trim();
    let visibleCount = 0;
    optionEntries.forEach(entry => {
      const match = query === '' || entry.option.searchable.includes(query);
      entry.button.style.display = match ? '' : 'none';
      if (match) visibleCount += 1;
    });
    sectionRecords.forEach(record => {
      const anyVisible = record.buttons.some(btn => btn.style.display !== 'none');
      record.element.style.display = anyVisible ? '' : 'none';
    });
    emptyMessage.style.display = visibleCount === 0 ? '' : 'none';
  };

  updateVisibility();
  searchInput.addEventListener('input', () => {
    if (stateRecord) {
      stateRecord.searchValue = searchInput.value;
    }
    updateVisibility();
  });

  optionsPane.scrollTop = optionsScrollTop;
  optionsPane.addEventListener('scroll', () => {
    if (stateRecord) {
      stateRecord.optionsScrollTop = optionsPane.scrollTop;
    }
  });

  if (sidebar) {
    sidebar.scrollTop = sidebarScrollTop;
    sidebar.addEventListener('scroll', () => {
      if (stateRecord) {
        stateRecord.sidebarScrollTop = sidebar.scrollTop;
      }
    });
  }
}

// Allow host to clear caches when switching boards/imports
export function clearTagCaches() {
  allowedTagCache.clear();
  ruleExplanationCache.clear();
  ruleExplainAvailable = null;
  invalidateCategoryMeta();
}

export function hideTagPicker(options = {}) {
  const clearContext = options?.clearContext !== false;
  const open = document.querySelector('.tag-picker');
  if (open) open.remove();
  if (tagPickerOutsideHandler) { document.removeEventListener('click', tagPickerOutsideHandler, true); tagPickerOutsideHandler = null; }
  if (tagPickerKeydownHandler) { document.removeEventListener('keydown', tagPickerKeydownHandler, true); tagPickerKeydownHandler = null; }
  if (clearContext) {
    currentPickerState = null;
  }
}

async function refreshTagPicker(scope, targetId) {
  if (!currentPickerState) return;
  if (currentPickerState.scope !== scope || currentPickerState.targetId !== targetId) {
    return;
  }
  const state = store?.getState?.();
  const node = state?.board?.nodes?.[targetId] ?? null;
  if (!node) {
    hideTagPicker();
    return;
  }
  const currentTags = Array.isArray(node.tags) ? node.tags : [];
  const searchValue = currentPickerState.searchValue ?? '';
  const optionsScrollTop = currentPickerState.optionsScrollTop ?? 0;
  const sidebarScrollTop = currentPickerState.sidebarScrollTop ?? 0;
  await showTagPicker(null, scope, targetId, currentTags, {
    searchValue,
    optionsScrollTop,
    sidebarScrollTop,
  });
}

function buildTagOptions(state, currentTags) {
  const options = [];
  const seen = new Set();
  const push = (kind, key, extras = {}) => {
    if (!key) return;
    const id = `${kind}:${key}`;
    if (seen.has(id)) return;
    seen.add(id);
    options.push({ kind, key, ...extras });
  };

  (state.modules?.tags?.system ?? []).forEach(tag => push('system', tag.key));
  (state.modules?.tags?.user ?? []).forEach(tag => push('user', tag.key, { label: tag.label, icon: tag.icon, color: tag.color }));
  collectBoardTagKeys(state.board).forEach(tag => push(tag.kind, tag.key));
  return options;
}

async function addTagCommand(scope, targetId, tag) {
  const payload = buildTagPayload(tag);
  const res = await send('AddTagV3', { nodeId: targetId, tag: payload });
  // Invalidate cache so subsequent openings reflect the new state
  try { invalidateAllowedTagCache(scope, targetId); } catch (_) {}
  return res;
}

function buildTagPayload(tag) {
  const key = String(tag?.key ?? '').trim();
  if (!key) {
    throw new Error('TAG_KEY_MISSING');
  }
  const kind = tag?.kind === 'system' ? 'system' : 'user';
  const payload = { key, kind };
  if (typeof tag?.label === 'string' && tag.label.trim() !== '') {
    payload.label = tag.label.trim();
  }
  if (typeof tag?.icon === 'string' && tag.icon.trim() !== '') {
    payload.icon = tag.icon.trim();
  }
  if (typeof tag?.color === 'string' && tag.color.trim() !== '') {
    payload.color = tag.color.trim();
  }
  if (typeof tag?.themeColor === 'string' && tag.themeColor.trim() !== '') {
    payload.color = payload.color ?? tag.themeColor.trim();
  }
  if (tag?.category) {
    payload.category = tag.category;
  }
  return payload;
}

async function fetchAllowedTags(scope, targetId) {
  const state = store.getState(); if (!state.currentBoardId) return [];
  const cacheKey = `${state.currentBoardId}:${scope}:${targetId}`;
  if (allowedTagCache.has(cacheKey)) return allowedTagCache.get(cacheKey);
  const params = new URLSearchParams({ scope, targetId });
  const payload = await requestJson(`/api/boards/${encodeURIComponent(state.currentBoardId)}/allowed-tags?${params.toString()}`, { method: 'GET' });
  const incomingVersion = typeof payload.rulesVersion === 'string' ? payload.rulesVersion : null;
  if (incomingVersion) {
    const currentVersion = store.getState().rulesVersion;
    if (currentVersion !== incomingVersion) { allowedTagCache.clear(); store.setState(prev => ({ ...prev, rulesVersion: incomingVersion })); }
  }
  const seen = new Set();
  const values = (payload.tags ?? []).map(tag => ({ kind: tag.kind, key: tag.key, label: tag.label ?? tag.key, icon: tag.icon ?? '', color: tag.color ?? null }))
    .filter(tag => { const key = `${tag.kind}:${tag.key}`; if (seen.has(key)) return false; seen.add(key); return true; });
  values.forEach(tag => {
    if (tag.kind === 'system' && tag.key && tag.key.startsWith('category/')) {
      const meta = categoriesMeta.get(tag.key);
      if (meta) {
        tag.label = meta.label ?? labelFromKey(tag.key);
        tag.icon = meta.icon ?? tag.icon;
        tag.color = meta.themeColor ?? tag.color;
      } else {
        tag.label = labelFromKey(tag.key);
      }
    }
  });
  allowedTagCache.set(cacheKey, values);
  return values;
}

function invalidateAllowedTagCache(scope, targetId) {
  const state = store.getState(); if (!state.currentBoardId) return; allowedTagCache.delete(`${state.currentBoardId}:${scope}:${targetId}`);
}

function collectBoardTagKeys(board) {
  const seen = new Set();
  const items = [];
  if (!board?.nodes) return items;
  for (const id in board.nodes) {
    const node = board.nodes[id];
    const tags = Array.isArray(node?.tags) ? node.tags : [];
    for (const tag of tags) {
      const key = String(tag?.key ?? tag?.k ?? '');
      if (!key) continue;
      const kind = tag?.sys === true || tag?.kind === 'system' ? 'system' : 'user';
      const hash = `${kind}:${key}`;
      if (seen.has(hash)) continue;
      seen.add(hash);
      items.push({ kind, key });
    }
  }
  return items;
}

function createTagOptionButton(option, scope, targetId) {
  const button = document.createElement('button');
  button.type = 'button';
  button.className = 'tag-picker__option';
  button.dataset.kind = option.kind;
  button.dataset.key = option.key;
  if (option.color) {
    button.style.borderColor = option.color;
    button.style.background = `${option.color}1A`;
    button.style.color = option.color;
  }
  const label = option.label ?? option.key;
  if (option.icon) {
    const iconSpan = document.createElement('span');
    iconSpan.className = 'tag-picker__option-icon';
    iconSpan.textContent = option.icon;
    button.appendChild(iconSpan);
  }
  const textSpan = document.createElement('span');
  textSpan.className = 'tag-picker__option-label';
  textSpan.textContent = label;
  button.appendChild(textSpan);
  button.addEventListener('click', async () => {
    try {
      await addTagCommand(scope, targetId, { kind: option.kind, key: option.key, label: option.label, icon: option.icon, color: option.color, category: option.category });
      hideTagPicker();
    } catch (error) {
      console.error(error);
      showToast('Impossible d\'ajouter le tag', { kind: 'error' });
    }
  });
  return button;
}

function groupOptionsBySection(options) {
  const buckets = new Map();
  options.forEach(option => {
    const name = option.section ?? 'Autres';
    if (!buckets.has(name)) {
      buckets.set(name, []);
    }
    buckets.get(name).push(option);
  });
  return Array.from(buckets.entries())
    .map(([name, opts]) => ({
      name,
      options: opts.slice().sort((a, b) => a.label.localeCompare(b.label, 'fr')),
    }))
    .sort((a, b) => {
      const diff = sectionWeight(a.name) - sectionWeight(b.name);
      if (diff !== 0) return diff;
      return a.name.localeCompare(b.name, 'fr');
    });
}

function normalizeTagOptions(options, state, existingKeys = new Set()) {
  const systemMeta = new Map((state.modules?.tags?.system ?? []).map(tag => [tag.key, tag]));
  const userMeta = new Map((state.modules?.tags?.user ?? []).map(tag => [tag.key, tag]));
  const normalized = [];
  for (const option of options) {
    const key = String(option?.key ?? option?.k ?? '');
    if (!key) continue;
    const kind = option?.kind === 'system' || option?.sys === true ? 'system' : 'user';
    const id = `${kind}:${key}`;
    if (existingKeys.has(id)) {
      continue;
    }
    const meta = kind === 'system' ? systemMeta.get(key) ?? null : userMeta.get(key) ?? null;
    const category = kind === 'system' ? (meta?.category ?? option?.category ?? null) : null;
    let label = meta?.label ?? option?.label ?? key;
    let icon = meta?.icon ?? option?.icon ?? '';
    let color = meta?.color ?? option?.color ?? null;
    if (kind === 'system' && key.startsWith('category/')) {
      const datasetMeta = categoriesMeta.get(key);
      if (datasetMeta) {
        label = datasetMeta.label ?? labelFromKey(key);
        icon = datasetMeta.icon ?? icon;
        color = datasetMeta.themeColor ?? color;
      } else {
        label = labelFromKey(key);
      }
    }
    const section = kind === 'system' ? mapCategoryLabel(category) : 'Tags utilisateur';
    normalized.push({
      kind,
      key,
      label,
      icon,
      color,
      category,
      section,
      searchable: `${label} ${key}`.toLowerCase(),
    });
  }
  return normalized;
}

function buildExistingTagKeySet(currentTags) {
  const set = new Set();
  (currentTags ?? []).forEach(tag => {
    const key = String(tag?.key ?? tag?.k ?? '');
    if (!key) return;
    const kind = tag?.sys === true || tag?.kind === 'system' ? 'system' : 'user';
    set.add(`${kind}:${key}`);
  });
  return set;
}

function buildSystemTagGroups(currentTags, state, node) {
  const groups = new Map();
  const systemMeta = new Map((state.modules?.tags?.system ?? []).map(tag => [tag.key, tag]));
  const childrenCount = Array.isArray(node?.children) ? node.children.length : 0;
  (currentTags ?? []).forEach(tag => {
    const key = String(tag?.key ?? tag?.k ?? '');
    if (!key) return;
    const isSystem = tag?.sys === true || tag?.kind === 'system';
    if (!isSystem) return;
    const meta = systemMeta.get(key) ?? null;
    const category = meta?.category ?? tag?.category ?? null;
    const label = meta?.label ?? tag?.label ?? labelFromKey(key);
    const icon = meta?.icon ?? tag?.icon ?? '';
    const color = meta?.color ?? tag?.color ?? null;
    const groupName = mapCategoryLabel(category);
    if (!groups.has(groupName)) {
      groups.set(groupName, { name: groupName, category, tags: [] });
    }
    const disabled = key === 'type/list' && childrenCount > 0;
    groups.get(groupName).tags.push({
      key,
      label,
      icon,
      color,
      disabled,
      reason: disabled ? 'Impossible de retirer une liste tant qu\'elle contient des éléments.' : null,
    });
  });
  return Array.from(groups.values())
    .map(group => ({
      ...group,
      tags: group.tags.sort((a, b) => a.label.localeCompare(b.label, 'fr')),
    }))
    .sort((a, b) => {
      const diff = sectionWeight(a.name) - sectionWeight(b.name);
      if (diff !== 0) return diff;
      return a.name.localeCompare(b.name, 'fr');
    });
}

function renderSystemTagsSidebar(groups, scope, targetId, options = {}) {
  const sidebar = document.createElement('aside');
  sidebar.className = 'tag-picker__sidebar';

  const title = document.createElement('h4');
  title.className = 'tag-picker__sidebar-title';
  title.textContent = 'Tags appliqués';
  sidebar.appendChild(title);

  groups.forEach(group => {
    if (!group.tags.length) {
      return;
    }
    const section = document.createElement('section');
    section.className = 'tag-picker__sidebar-section';

    const heading = document.createElement('h5');
    heading.className = 'tag-picker__sidebar-heading';
    heading.textContent = group.name;
    section.appendChild(heading);

    const list = document.createElement('div');
    list.className = 'tag-picker__sidebar-list';

    group.tags.forEach(tag => {
      const button = document.createElement('button');
      button.type = 'button';
      button.className = 'tag-picker__system-tag';
      button.dataset.key = tag.key;
      if (tag.icon) {
        const icon = document.createElement('span');
        icon.className = 'tag-picker__system-tag-icon';
        icon.textContent = tag.icon;
        button.appendChild(icon);
      }
      const label = document.createElement('span');
      label.className = 'tag-picker__system-tag-label';
      label.textContent = tag.label;
      button.appendChild(label);
      if (tag.color) {
        button.style.borderColor = tag.color;
        button.style.background = `${tag.color}1A`;
        button.style.color = tag.color;
      }
      if (tag.disabled) {
        button.disabled = true;
        if (tag.reason) {
          button.title = tag.reason;
        }
      } else {
        button.addEventListener('click', async () => {
          try {
            await removeTag(scope, targetId, 'system', tag.key, { closePicker: false });
            options?.onRemoved?.(tag);
          } catch (error) {
            console.error('SYSTEM_TAG_REMOVE_FAILED', error);
            showToast('Impossible de retirer ce tag', { kind: 'error' });
          }
        });
      }
      list.appendChild(button);
    });

    section.appendChild(list);
    sidebar.appendChild(section);
  });

  return sidebar;
}

function mapCategoryLabel(category) {
  if (!category) {
    return 'Tags système';
  }
  switch (category) {
    case 'state':
      return 'États';
    case 'category':
      return 'Catégories';
    case 'type':
      return 'Types';
    default:
      return category ? category.charAt(0).toUpperCase() + category.slice(1) : 'Autres';
  }
}

function labelFromKey(key) {
  if (typeof key !== 'string') return '';
  if (key.includes('/')) {
    key = key.split('/').slice(1).join('/');
  }
  const cleaned = key.replace(/[-_]+/g, ' ').trim();
  if (!cleaned) return key;
  return cleaned.split(' ').map(word => word ? word.charAt(0).toUpperCase() + word.slice(1) : '').join(' ');
}

async function loadCategoriesMeta() {
  if (categoriesLoaded) return categoriesMeta;
  if (categoriesLoading) return categoriesLoading;
  categoriesLoading = requestJson('/api/packs/datasets/org%3Acategories', { method: 'GET' })
    .then(res => {
      const items = Array.isArray(res?.data?.items) ? res.data.items : [];
      const next = new Map();
      items.forEach(item => {
        if (!item || typeof item.key !== 'string') return;
        next.set(item.key, {
          label: item.label ?? null,
          icon: item.icon ?? null,
          themeColor: item.themeColor ?? null,
        });
      });
      categoriesMeta = next;
      categoriesLoaded = true;
      categoriesLoading = null;
      applyCategoryMetaToStore(next);
      return categoriesMeta;
    })
    .catch(error => {
      console.error('CATEGORY_DATASET_LOAD_FAILED', error);
      categoriesLoading = null;
      categoriesLoaded = false;
      return categoriesMeta;
    });
  return categoriesLoading;
}

function invalidateCategoryMeta() {
  categoriesLoaded = false;
  categoriesMeta = new Map();
  categoriesLoading = null;
  applyCategoryMetaToStore(categoriesMeta);
}

function applyCategoryMetaToStore(metaMap) {
  if (!store || typeof store.getState !== 'function' || typeof store.setState !== 'function') {
    return;
  }
  if (applyingCategoryMeta) {
    return;
  }
  const state = store.getState();
  const systemTags = state?.modules?.tags?.system;
  if (!Array.isArray(systemTags)) {
    return;
  }
  let changed = false;
  const updated = systemTags.map(tag => {
    const key = tag?.key;
    if (!key || !key.startsWith('category/')) {
      return tag;
    }
    const meta = metaMap.get(key) ?? null;
    const nextLabel = meta?.label ?? labelFromKey(key);
    const nextIcon = meta?.icon ?? '';
    const nextColor = meta?.themeColor ?? null;
    const currentIcon = tag.icon ?? '';
    const currentColor = tag.color ?? null;
    const currentLabel = tag.label ?? labelFromKey(key);
    if (currentLabel === nextLabel && currentIcon === nextIcon && currentColor === nextColor) {
      return tag;
    }
    changed = true;
    return {
      ...tag,
      label: nextLabel,
      icon: nextIcon,
      color: nextColor,
    };
  });
  if (!changed) {
    return;
  }
  applyingCategoryMeta = true;
  try {
    store.setState(prev => ({
      ...prev,
      modules: {
        ...(prev.modules ?? {}),
        tags: {
          ...(prev.modules?.tags ?? {}),
          system: updated,
        },
      },
    }));
  } finally {
    applyingCategoryMeta = false;
  }
}

if (typeof window !== 'undefined' && !window.__sbCategoryDatasetReload) {
  window.addEventListener('org:categories:updated', () => {
    invalidateCategoryMeta();
    loadCategoriesMeta().catch(() => {});
  });
  window.__sbCategoryDatasetReload = true;
}

function sectionWeight(name) {
  switch (name) {
    case 'États':
      return 1;
    case 'Catégories':
      return 2;
    case 'Types':
      return 3;
    case 'Autres':
    case 'Tags système':
      return 4;
    case 'Tags utilisateur':
      return 5;
    default:
      return 10;
  }
}

async function send(type, payload) {
  // Prefer injected sendCommand which handles autosave, patches, etc.
  if (typeof sendCommandRef === 'function') {
    return await sendCommandRef(type, payload);
  }
  // Fallback: direct POST to command endpoint
  const response = await requestJson('/api/commands', { method: 'POST', body: { type, boardId: store.getState().currentBoardId, revision: store.getState().boardRevision ?? store.getState().boardMeta?.revision ?? null, payload } });
  return response;
}

function slugify(str) {
  const normalized = String(str ?? '')
    .normalize('NFD')
    .replace(/[\u0300-\u036f]/g, '')
    .trim()
    .toLowerCase();
  const value = normalized.replace(/[^a-z0-9]+/g, '-').replace(/^-+|-+$/g, '');
  return value;
}
