import { renderSlot as runtimeRenderSlot } from '../../../../packages/modules/runtime.js';
import { findNode, childrenOf, isContainer } from '../domain-utils.js';
import { escapeHtml, formatDate } from './format.js';

export function renderRoot(state) {
  const board = state.board;
  if (!board?.nodes) return '';
  const root = board.nodes[board.rootId];
  if (!root) return '';
  const children = (root.children || []).map(id => board.nodes[id]).filter(Boolean);
  return `
    <div class="node-root" data-dropzone="true" data-dropzone-type="list" data-node="${root.id}">
      ${children.map(n => renderNode(n, state, 0)).join('')}
    </div>
  `;
}

export function renderNode(node, state, depth = 0) {
  const tags = Array.isArray(node.tags) ? node.tags : [];
  const stateKey = resolveStateKey(tags);
  const stateClass = stateKey ? ` card--state-${stateKey}` : '';
  const isList = isContainer(node);
  // When filtering is active, hide leaf nodes explicitly marked as not visible
  if (!isList && node._visible === false) {
    return '';
  }
  const collapsed = !!node.collapsed;
  const collapsedClass = collapsed ? ' collapsed' : '';
  const draggableAttrs = node.id === state.board.rootId
    ? ''
    : ` data-draggable="true" tabindex="0"${isList ? ` data-list="${node.id}"` : ''}`;
  // For lists, only render children that are containers or visible leaves
  const children = childrenOf(state.board, node.id)
    .filter(child => isContainer(child) || child._visible !== false);
  const visibleTags = tags.filter(tag => !shouldHideTag(tag));
  const tagsHtml = visibleTags.map(tag => renderTag(tag, state, node.id)).join('');
  const badgesSlot = `<div class="sb-badges-left" data-slot="item.badges" data-node="${node.id}" data-dnd-ignore="true"></div>`;
  const badgesInline = `<div class="item-badges" data-slot="item.badges" data-node="${node.id}" data-dnd-ignore="true"></div>`;
  const accentColor = resolveDepthAccent(depth);
  const listStyle = isList && accentColor ? ` style="--depth-accent:${accentColor};"` : '';
  const composer = isList ? renderListComposer(node.id, collapsed) : '';
  const typeMeta = resolveTypeMeta(tags, state.modules?.tags);
  const emptyHint = isList && children.length === 0 ? renderListEmptyHint() : '';

  if (isList) {
    return `
      <article class="list${collapsedClass}${stateClass}" data-type="list" data-node="${node.id}" data-id="${node.id}" data-depth="${depth}"${draggableAttrs}${listStyle}>
        <header class="list-header" data-action="open-item" data-item="${node.id}" data-node="${node.id}">
          <button class="list-header__toggle" data-action="toggle-list" data-node="${node.id}" aria-expanded="${collapsed ? 'false' : 'true'}" data-dnd-ignore="true">
            ${renderListGlyph(collapsed)}
            <span class="sr-only">${collapsed ? 'Déplier la liste' : 'Replier la liste'}</span>
          </button>
          <span class="list-header__title">${escapeHtml(node.title ?? '')}</span>
          <div class="list-header__actions" data-dnd-ignore="true">
            <button class="btn btn--icon ghost" data-dnd-ignore="true" data-action="open-list-composer" data-node="${node.id}" title="Ajouter" aria-label="Ajouter ici">＋</button>
            <button class="btn btn--icon ghost" data-dnd-ignore="true" data-action="rename-node" data-node="${node.id}" title="Renommer">✏️</button>
            <button class="btn btn--icon ghost delete-hover tag-launcher" data-dnd-ignore="true" data-action="add-tag" data-scope="item" data-node="${node.id}" title="Ajouter un tag" aria-label="Ajouter un tag">🏷️</button>
            <button class="btn btn--icon ghost delete-hover" data-dnd-ignore="true" title="Supprimer" data-action="delete-node" data-node="${node.id}">🗑️</button>
          </div>
        </header>
        <div class="list-skin">
          ${badgesSlot}
          <div class="tag-strip">
            ${tagsHtml}
          </div>
        </div>
        <div class="children" data-dropzone="true" data-dropzone-type="list" data-node="${node.id}">
          ${composer}
          ${children.map(child => renderNode(child, state, depth + 1)).join('')}
          ${emptyHint}
          <div class="list-drop-tail" data-dnd-ignore="true"></div>
        </div>
      </article>
    `;
  }

  const active = state.board.activeTab === node.id;
  const typeGlyph = renderTypeGlyph(typeMeta);
  const metaDate = `<span class="item-meta__date">${formatDate(node.updatedAt)}</span>`;
  return `
    <div class="item${stateClass}" data-type="item" data-draggable="true" data-id="${node.id}" data-node="${node.id}" data-depth="${depth}" tabindex="0">
      <div class="item__inner">
        ${typeGlyph}
        <div class="item__content">
          <header class="item-header" data-action="open-item" data-item="${node.id}" data-node="${node.id}">
            <span class="item-title">${escapeHtml(node.title ?? '')}</span>
            ${active ? '<span class="item-active-dot" aria-hidden="true">●</span>' : ''}
          </header>
          <div class="tag-strip">
            ${tagsHtml}
          </div>
          <div class="item-meta" data-dnd-ignore="true">
            <div class="item-meta__info">
              ${badgesInline}
              ${metaDate}
            </div>
            <div class="item-meta__actions" data-dnd-ignore="true">
              <button class="btn btn--icon ghost" data-action="add-tag" data-scope="item" data-node="${node.id}" title="Ajouter un tag" aria-label="Ajouter un tag">🏷️</button>
              <button class="btn btn--icon ghost" title="Supprimer" data-action="delete-node" data-node="${node.id}">🗑️</button>
            </div>
          </div>
        </div>
      </div>
    </div>
  `;
}

export function renderTag(tag, state, nodeId) {
  if (shouldHideTag(tag)) {
    return '';
  }
  const meta = resolveTagMeta(tag, state.modules?.tags);
  if (meta?.ui?.tagList?.hidden === true) {
    return '';
  }
  const color = meta?.color ?? tag.color;
  const icon = meta?.icon ?? tag.icon ?? '';
  const label = meta?.label ?? tag.label ?? (tag.key ?? tag.v);
  const style = color ? `style="border:1px solid ${color}; color:${color}; background:${color}1A"` : '';
  const key = tag.key ?? tag.k;
  return `<button type="button" class="tag" data-action="remove-tag" data-node="${nodeId}" data-key="${key}" ${style}>${icon ? `${icon} ` : ''}${escapeHtml(label)}</button>`;
}

export function mountSlotsIn(container, state) {
  const nodes = container.querySelectorAll('[data-slot]');
  const disposers = [];
  for (const holder of nodes) {
    const slotId = holder.getAttribute('data-slot');
    if (!slotId) continue;
    const ctx = { state, item: resolveItemFromHolder(holder, state) };
    try {
      const outcome = runtimeRenderSlot(slotId, ctx, holder);
      if (outcome && typeof outcome.then === 'function') {
        outcome.then(dispose => {
          if (typeof dispose === 'function') disposers.push(dispose);
        }).catch(error => console.error(`SLOT_RENDER_ASYNC_FAILED ${slotId}`, error));
      } else if (typeof outcome === 'function') {
        disposers.push(outcome);
      }
    } catch (error) {
      console.error(`SLOT_RENDER_FAILED ${slotId}`, error);
    }
  }
  return () => disposers.splice(0).forEach(fn => { try { fn(); } catch (_) {} });
}

function resolveItemFromHolder(holder, state) {
  const nodeId = holder.getAttribute('data-node');
  if (!nodeId) return null;
  try { return state.board?.nodes?.[nodeId] ?? null; } catch (_) { return null; }
}

function resolveStateKey(tags = []) {
  for (const tag of tags) {
    const key = typeof tag?.key === 'string' ? tag.key : (typeof tag?.k === 'string' ? tag.k : '');
    if (key.startsWith('state/')) {
      return key.split('/')[1] ?? null;
    }
  }
  return null;
}

function shouldHideTag(tag) {
  const key = typeof tag?.key === 'string' ? tag.key : (typeof tag?.k === 'string' ? tag.k : '');
  if (!key) {
    return false;
  }
  if (key === 'type/list') {
    return true;
  }
  if (key.startsWith('type/')) {
    return true;
  }
  return key.startsWith('state/') || key.startsWith('category/');
}

function renderListComposer(listId, collapsed = false) {
  if (!listId || collapsed) {
    return '';
  }
  const safeId = escapeHtml(listId);
  return `
    <div class="list-composer" data-composer="${safeId}" data-node="${safeId}" data-dnd-ignore="true">
      <div class="list-composer__panel">
        <input type="text" class="list-composer__input" data-composer-input data-node="${safeId}" placeholder="Titre..." aria-label="Titre du nouvel élément" />
        <div class="list-composer__actions">
          <button type="button" class="btn" data-action="composer-add-item" data-node="${safeId}">Ajouter l'item</button>
          <button type="button" class="btn ghost" data-action="composer-add-list" data-node="${safeId}">Nouvelle liste</button>
          <button type="button" class="btn ghost" data-action="composer-cancel" data-node="${safeId}">Annuler</button>
        </div>
      </div>
    </div>
  `;
}

function renderListEmptyHint() {
  return `
    <div class="list-empty-hint" data-dnd-ignore="true">
      Glissez un élément ou utilisez « Ajouter » pour commencer
    </div>
  `;
}

function renderListGlyph(collapsed) {
  if (collapsed) {
    return `
      <svg class="list-toggle__icon" viewBox="0 0 24 24" focusable="false" aria-hidden="true">
        <circle cx="12" cy="12" r="8.5" stroke="currentColor" stroke-width="1.8" fill="none"></circle>
        <rect x="11" y="7.5" width="2" height="9" rx="1"></rect>
        <rect x="7.5" y="11" width="9" height="2" rx="1"></rect>
      </svg>
    `;
  }
  return `
    <svg class="list-toggle__icon" viewBox="0 0 24 24" focusable="false" aria-hidden="true">
      <circle cx="5" cy="7" r="1.4"></circle>
      <circle cx="5" cy="12" r="1.4"></circle>
      <circle cx="5" cy="17" r="1.4"></circle>
      <rect x="8" y="6" width="11" height="2" rx="1"></rect>
      <rect x="8" y="11" width="11" height="2" rx="1"></rect>
      <rect x="8" y="16" width="11" height="2" rx="1"></rect>
    </svg>
  `;
}

const DEPTH_ACCENTS = [
  'rgba(59, 130, 246, 0.5)',
  'rgba(99, 102, 241, 0.45)',
  'rgba(236, 72, 153, 0.42)',
  'rgba(245, 158, 11, 0.44)',
  'rgba(16, 185, 129, 0.46)',
];

function resolveDepthAccent(depth = 0) {
  if (!Number.isFinite(depth) || depth < 0) {
    return DEPTH_ACCENTS[0];
  }
  const index = depth % DEPTH_ACCENTS.length;
  return DEPTH_ACCENTS[index];
}

function resolveTypeMeta(tags = [], registry) {
  for (const tag of tags) {
    const key = typeof tag?.key === 'string' ? tag.key : (typeof tag?.k === 'string' ? tag.k : '');
    if (!key) continue;
    if (key === 'type/list') continue;
    if (key.startsWith('type/')) {
      return {
        meta: resolveTagMeta(tag, registry),
        tag,
        key,
      };
    }
  }
  return null;
}

function renderTypeGlyph(meta) {
  const badge = meta?.meta?.ui?.badge ?? null;
  const fallbackIcon = typeof meta?.tag?.icon === 'string' ? meta.tag.icon : null;
  const rawIcon = badge?.icon ?? fallbackIcon;
  const icon = rawIcon && rawIcon.trim() !== '' ? rawIcon : '●';
  const color = badge?.color ?? null;
  const variant = color ? 'typed' : 'default';
  const softened = color ? softenColor(color) : null;
  const style = softened ? ` style="--type-color:${softened}"` : '';
  const label = badge?.label ?? (variant === 'typed' ? meta?.key ?? 'Type' : 'Item');
  return `<span class="item-type-glyph" data-variant="${variant}"${style} title="${escapeHtml(label)}">${escapeHtml(icon)}</span>`;
}

function softenColor(color) {
  if (!color || typeof color !== 'string') {
    return null;
  }
  const hex = color.trim();
  if (hex.startsWith('#')) {
    const value = hex.slice(1);
    const size = value.length;
    if (size === 3 || size === 6) {
      const normalized = size === 3
        ? value.split('').map(ch => ch + ch).join('')
        : value;
      const r = parseInt(normalized.slice(0, 2), 16);
      const g = parseInt(normalized.slice(2, 4), 16);
      const b = parseInt(normalized.slice(4, 6), 16);
      if (Number.isNaN(r) || Number.isNaN(g) || Number.isNaN(b)) {
        return null;
      }
      return `rgba(${r}, ${g}, ${b}, 0.28)`;
    }
    return null;
  }
  if (hex.startsWith('rgba')) {
    return hex;
  }
  if (hex.startsWith('rgb')) {
    return hex.replace('rgb', 'rgba').replace(')', ', 0.3)');
  }
  return null;
}

export function renderFilterTags(state) {
  const tags = collectTags(state.board, state.modules?.tags)
    .sort((a, b) => (a.label ?? '').localeCompare(b.label ?? '', 'fr'));
  return tags.map(tag => {
    const key = tag.key ?? tag.k;
    const active = state.filter.tags.includes(key);
    const meta = resolveTagMeta(tag, state.modules?.tags);
    const label = meta?.label ?? tag.label ?? key;
    const icon = meta?.icon ?? tag.icon ?? '';
    return `<button data-action="toggle-filter" data-tag="${key}" class="${active ? 'active' : ''}">
      ${icon ? `${icon} ` : ''}${escapeHtml(label)}
    </button>`;
  }).join('');
}

export function collectTags(board, registry) {
  if (!board?.nodes) return [];
  const tags = new Map();
  for (const id in board.nodes) {
    const node = board.nodes[id];
    (node.tags || []).forEach(tag => {
      const cat = tag.k ?? tag.key ?? '';
      const val = tag.v ?? tag.value ?? '';
      const key = (tag.key ?? (cat && val ? `${cat}/${val}` : cat));
      if (!key) return;
      const metaList = (tag.sys || tag.kind === 'system') ? (registry?.system ?? []) : (registry?.user ?? []);
      const meta = metaList.find?.(t => t.key === key) || null;
      // Respecter ui.filter.include=false pour masquer les tags non filtrables
      if (meta?.ui && meta.ui.filter && meta.ui.filter.include === false) {
        return; // non filtrable
      }
      tags.set(key, { ...tag, key });
    });
  }
  return Array.from(tags.values()).map(tag => {
    const meta = resolveTagMeta(tag, registry);
    const key = tag.key ?? tag.k;
    return {
      ...tag,
      key,
      label: meta?.label ?? tag.label ?? key,
      icon: meta?.icon ?? tag.icon ?? '',
      color: meta?.color,
    };
  });
}

export function resolveTagMeta(tag, registry) {
  if (!registry) return null;
  const key = tag.key ?? tag.k;
  const lists = (tag.sys || tag.kind === 'system') ? (registry.system ?? []) : (registry.user ?? []);
  return lists.find(entry => entry.key === key) ?? null;
}

export function renderColumn(column, state, workspaceId) {
  const nodes = state.board?.nodes || {};
  const childrenIds = Array.isArray(column.children) ? column.children : [];
  const childrenHtml = childrenIds
    .map(id => nodes[id])
    .filter(Boolean)
    // Keep containers (lists) always; keep leaves only if visible
    .filter(child => isContainer(child) || child._visible !== false)
    .map(child => renderNode(child, state, 0))
    .join('');
  const title = escapeHtml(column.title ?? '');
  const isEmpty = childrenHtml.trim() === '';
  const columnClass = `board-column${isEmpty ? ' board-column--empty' : ''}`;
  const emptyState = isEmpty ? `
        <div class="board-column__empty" data-empty="column">
          <span class="board-column__empty-icon">＋</span>
          <span class="board-column__empty-hint">Ajoutez un item, une liste ou glissez ici</span>
        </div>
      ` : '';
  return `
    <section class="${columnClass}" data-column="${column.id}" data-id="${column.id}" data-type="column" data-workspace="${workspaceId}" data-draggable="true">
      <header class="board-column__header">
        <h3>${title}</h3>
        <div class="column-actions" data-dnd-ignore="true">
          <button class="btn btn--icon ghost" data-action="create-item-root" data-column="${column.id}" data-workspace="${workspaceId}" title="Nouvel item">＋</button>
          <button class="btn btn--icon ghost" data-action="create-folder" data-column="${column.id}" data-workspace="${workspaceId}" title="Nouveau dossier">📁</button>
          <button class="btn btn--icon ghost" data-action="rename-column" data-column="${column.id}" title="Renommer">✏️</button>
          <button class="btn btn--icon ghost delete-hover" title="Supprimer" data-action="delete-column" data-column="${column.id}">🗑️</button>
        </div>
      </header>
      <div class="board-column__body" data-dropzone="true" data-dropzone-type="list" data-node="${column.id}">
        ${childrenHtml || emptyState}
      </div>
    </section>
  `;
}

// helpers delegated to ./format.js
export { escapeHtml, formatDate };

// Supprimer l'effet spécial "delete-hover" pour une apparition homogène des 4 boutons
