// Dialog rendering and helpers extracted from main.js without changes
import { createBoardExportBundle, computeImportPreview, summarizeBoardState } from '../import-export.js';
import { formatDateTime } from './format.js';
import { escapeHtml } from './board-view.js';

export function renderDialog(state) {
  const dialog = state.dialog;
  if (!dialog) { return ''; }
  if (dialog.kind === 'export') { return renderExportDialog(dialog, state); }
  if (dialog.kind === 'import') { return renderImportDialog(dialog, state); }
  return '';
}

export function renderExportDialog(dialog, state) {
  const boardTitle = state.boards.find(b => String(b.id) === state.currentBoardId)?.title ?? 'Board';
  if (dialog.status === 'loading') {
    return `
    <div class="modal-overlay" data-action="close-dialog">
      <div class="modal" data-stop-propagation>
        <header class="modal__header">
          <h3>Export en cours…</h3>
        </header>
          <div class="modal__body">
            <p>Préparation du fichier JSON pour <strong>${escapeHtml(boardTitle)}</strong>.</p>
          </div>
          <footer class="modal__footer">
            <button class="btn" data-action="close-dialog">Fermer</button>
          </footer>
        </div>
      </div>`;
  }

  if (dialog.status === 'error') {
    return `
    <div class="modal-overlay" data-action="close-dialog">
      <div class="modal" data-stop-propagation>
        <header class="modal__header">
          <h3>Export impossible</h3>
        </header>
          <div class="modal__body">
            <p class="modal__error">${escapeHtml(dialog.message ?? 'Une erreur est survenue lors de l\'export')}</p>
          </div>
          <footer class="modal__footer">
            <button class="btn" data-action="close-dialog">Fermer</button>
          </footer>
        </div>
      </div>`;
  }

  const bundle = dialog.board ? createBoardExportBundle(dialog.board) : null;
  const json = bundle ? JSON.stringify(bundle, null, 2) : '';
  const exportedAt = dialog.board?.exportedAt ? formatDateTime(dialog.board.exportedAt) : null;
  const updatedAt = dialog.board?.updatedAt ? formatDateTime(dialog.board.updatedAt) : null;
  const createdAt = dialog.board?.createdAt ? formatDateTime(dialog.board.createdAt) : null;
  const revision = dialog.board?.revision ?? null;
  const exportSummary = dialog.board?.state ? summarizeBoardState(dialog.board.state, revision) : null;
  const ruleCount = Array.isArray(bundle?.rules) ? bundle.rules.length : null;

  return `
    <div class="modal-overlay" data-action="close-dialog">
      <div class="modal" data-stop-propagation>
        <header class="modal__header">
          <h3>Exporter le board</h3>
          <p class="modal__subtitle">${escapeHtml(dialog.board?.title ?? boardTitle)}</p>
        </header>
        <div class="modal__body">
          <ul class="modal__meta">
            ${exportedAt ? `<li><strong>Exporté :</strong> ${escapeHtml(exportedAt)}</li>` : ''}
            ${updatedAt ? `<li><strong>Dernière sauvegarde :</strong> ${escapeHtml(updatedAt)}</li>` : ''}
            ${createdAt ? `<li><strong>Créé :</strong> ${escapeHtml(createdAt)}</li>` : ''}
            ${revision ? `<li><strong>Révision :</strong> ${escapeHtml(String(revision))}</li>` : ''}
            ${exportSummary && typeof exportSummary.nodeCount === 'number' ? `<li><strong>Nodes :</strong> ${exportSummary.nodeCount}</li>` : ''}
            ${exportSummary ? `<li><strong>Listes :</strong> ${exportSummary.listCount ?? 0}</li>` : ''}
            ${exportSummary ? `<li><strong>Items :</strong> ${exportSummary.itemCount ?? 0}</li>` : ''}
            ${ruleCount !== null ? `<li><strong>Règles :</strong> ${ruleCount}</li>` : ''}
          </ul>
          <label class="modal__field">
            <span>JSON du board</span>
            <textarea readonly data-ref="export-json">${escapeHtml(json)}</textarea>
          </label>
        </div>
        <footer class="modal__footer">
          <button class="btn ghost" data-action="copy-export">Copier</button>
          <button class="btn ghost" data-action="download-export">Télécharger</button>
          <button class="btn" data-action="close-dialog">Fermer</button>
        </footer>
      </div>
    </div>`;
}

export function renderImportDialog(dialog, state) {
  const mode = dialog.mode ?? 'replace';
  const form = dialog.form ?? {};
  const currentBoard = state.boards.find(b => String(b.id) === state.currentBoardId);
  const replaceTitle = form.replaceTitle ?? (currentBoard?.title ?? '');
  const newTitle = form.newTitle ?? `Import ${new Date().toLocaleDateString('fr-FR')}`;
  const workspaceTitle = form.workspaceTitle ?? 'Workspace';
  const json = form.json ?? '';
  const preview = mode !== 'reset' ? computeImportPreview(json) : { summary: null, error: null };

  return `
    <div class="modal-overlay" data-action="close-dialog">
      <div class="modal" data-stop-propagation>
        <header class="modal__header">
          <h3>Importer un board</h3>
        </header>
        <div class="modal__body">
          <fieldset class="modal__fieldset">
            <legend>Mode d'import</legend>
            <label><input type="radio" name="import-mode" value="replace" data-action="set-import-mode" ${mode === 'replace' ? 'checked' : ''}> Remplacer ce board</label>
            <label><input type="radio" name="import-mode" value="new" data-action="set-import-mode" ${mode === 'new' ? 'checked' : ''}> Créer un nouveau board</label>
            <label><input type="radio" name="import-mode" value="reset" data-action="set-import-mode" ${mode === 'reset' ? 'checked' : ''}> Réinitialiser ce board (vide)</label>
          </fieldset>
          ${mode !== 'reset' ? `
            <label class="modal__field">
              <span>JSON du board (collez le contenu exporté)</span>
              <textarea data-ref="import-json" placeholder="{ \"version\":3, \"rootId\":\"root\", \"nodes\":{...} }">${escapeHtml(json)}</textarea>
            </label>
            ${preview.error ? `<p class="modal__error">${escapeHtml(preview.error)}</p>` : ''}
            ${preview.summary ? `
              <div class="modal__hint modal__hint--import">
                <strong>Prévisualisation</strong>
                <span>Titre détecté : <em>${escapeHtml(preview.summary.title ?? 'Sans titre')}</em></span>
                ${typeof preview.summary.nodeCount === 'number' ? `<span>Nodes : ${preview.summary.nodeCount}</span>` : ''}
                <span>Listes : ${preview.summary.listCount ?? 0}</span>
                <span>Items : ${preview.summary.itemCount ?? 0}</span>
                ${typeof preview.summary.ruleCount === 'number' ? `<span>Règles : ${preview.summary.ruleCount}</span>` : ''}
                ${preview.summary.revision ? `<span>Révision : ${escapeHtml(String(preview.summary.revision))}</span>` : ''}
              </div>` : ''}
          ` : ''}
          ${mode === 'replace' ? `
            <label class="modal__field">
              <span>Nouveau titre (optionnel)</span>
              <input type="text" data-ref="import-replace-title" placeholder="${escapeHtml(currentBoard?.title ?? '')}" value="${escapeHtml(replaceTitle)}" />
            </label>
          ` : ''}
          ${mode === 'new' ? `
            <label class="modal__field">
              <span>Titre du nouveau board</span>
              <input type="text" data-ref="import-new-title" value="${escapeHtml(newTitle)}" />
            </label>
          ` : ''}
          ${mode === 'reset' ? `
            <label class="modal__field">
              <span>Nom du workspace initial</span>
              <input type="text" data-ref="import-workspace" value="${escapeHtml(workspaceTitle)}" />
            </label>
          ` : ''}
          <p class="modal__hint">Les données importées écraseront les informations selon le mode choisi. Pensez à conserver une copie du JSON courant.</p>
          ${dialog.error ? `<p class="modal__error">${escapeHtml(dialog.error)}</p>` : ''}
        </div>
        <footer class="modal__footer">
          <button class="btn ghost" data-action="close-dialog" ${dialog.busy ? 'disabled' : ''}>Annuler</button>
          <button class="btn primary" data-action="confirm-import" ${dialog.busy ? 'disabled' : ''}>${dialog.busy ? 'Import…' : 'Importer'}</button>
        </footer>
      </div>
    </div>`;
}
