import { collectItems as editorCollectItems } from '../../modules/editor.js';
import { escapeHtml } from './board-view.js';

export function renderEditorModal(state) {
  const openTabs = state.board?.openTabs ?? [];
  if (!openTabs.length) {
    return '';
  }
  const items = editorCollectItems(state.board);
  const activeId = determineActiveTab(state.board, openTabs);
  const activeItem = items.find(item => item.id === activeId) ?? null;
  const multipleTabs = openTabs.length > 1;
  const tabsMarkup = openTabs
    .map(tabId => {
      const item = items.find(i => i.id === tabId);
      if (!item) return '';
      const label = escapeHtml(item.title ?? item.id);
      const active = tabId === activeId;
      return `
        <div class="editor-tab ${active ? 'editor-tab--active' : ''}">
          <button type="button" class="editor-tab__label" data-action="switch-tab" data-item="${item.id}">
            ${label}
          </button>
          ${multipleTabs ? `<button type="button" class="editor-tab__close" data-action="close-tab" data-item="${item.id}" aria-label="Fermer l'onglet">×</button>` : ''}
        </div>
      `;
    })
    .join('');

  const closeAllButton = multipleTabs
    ? `<button type="button" class="editor-tabs__close-all" data-action="close-all-tabs" title="Fermer tous les onglets" aria-label="Fermer tous les onglets">
        <span class="editor-tabs__close-icon">✕</span>
        <span class="editor-tabs__close-label">Fermer tout</span>
      </button>`
    : '';

  const body = activeItem
    ? renderEditorContent(activeItem, state)
    : `<div class="editor-modal__empty">Sélectionnez un item à éditer.</div>`;

  const isNote = Array.isArray(activeItem?.tags)
    ? activeItem.tags.some(t => (t.key ?? t.k) === 'type/note')
    : false;
  const isText = Array.isArray(activeItem?.tags)
    ? activeItem.tags.some(t => (t.key ?? t.k) === 'type/text')
    : false;
  const hasSidePanel = isNote || isText;
  const layoutClass = hasSidePanel ? 'editor-modal editor-layout' : 'editor-modal';
  const modalWidth = hasSidePanel ? 'min(1080px, 95vw)' : 'min(700px, 90vw)';

  return `
    <sb-modal open data-editor-modal="true" style="--sb-modal-width: ${modalWidth};">
      <div class="${layoutClass}" role="document">
        <div class="editor-modal__main">
          <header class="editor-modal__header">
            <button type="button" class="btn btn--icon ghost editor-modal__close-left" data-action="close-editor-modal" data-item="${activeId ?? ''}" aria-label="Fermer" title="Fermer">✕</button>
            <h2>Édition d'item</h2>
            <div class="editor-modal__header-actions"></div>
          </header>
          <div class="editor-modal__tabs">
            ${tabsMarkup}
            ${closeAllButton}
          </div>
          <div class="editor-modal__body">
            ${body}
          </div>
        </div>
        ${hasSidePanel ? `
        <aside class="editor-modal__side">
          <div class="editor-side__content" data-slot="item.sidePanel" data-node="${activeId ?? ''}"></div>
        </aside>` : ''}
      </div>
    </sb-modal>
  `;
}

function determineActiveTab(board, openTabs) {
  const requested = board?.activeTab;
  if (requested && openTabs.includes(requested)) {
    return requested;
  }
  return openTabs[0] ?? null;
}

export function renderEditorContent(item, state) {
  const safeTitle = escapeHtml(item.title ?? '');
  const safeDescription = escapeHtml(item.description ?? '');
  return `
    <div class="item-editor" data-editor-item="${item.id}" data-node="${item.id}">
      <label class="editor-field">
        <span>Titre</span>
        <input type="text" value="${safeTitle}" data-field="title" />
      </label>
      <label class="editor-field">
        <span>Description</span>
        <textarea data-field="description">${safeDescription}</textarea>
      </label>
      <div data-slot="item.editorTabs" data-node="${item.id}"></div>
      <div class="editor-footer-slot" data-slot="item.editorFooter" data-node="${item.id}"></div>
      <footer class="editor-modal__footer">
        <button class="btn primary" data-action="save-item" data-item="${item.id}">Enregistrer</button>
      </footer>
    </div>
  `;
}
