import '../../packages/ui/modal.js';
import { confirmDialog, promptDialog } from '../../packages/ui/dialog.js';
import '../../packages/ui/menu.js';
import { showToast } from '../../packages/ui/toast.js';
import { ensureAuthenticated } from '../../packages/services/session.js';
import { fetchBoards, createBoard as createBoardRemote, renameBoard as renameBoardRemote, deleteBoard as deleteBoardRemote, setBoardThumbnail as setBoardThumbnailRemote } from '../../packages/services/boards.js';
import { applyDebugFromUrl, isDebug, toErrorMessage } from '../../packages/services/debug.js';
import { requestJson, setCsrfProvider } from '../../packages/services/http.js';
import { goToBoard, goToModules, goToAccount, goToAdmin, goToAuth } from '../../packages/services/navigation.js';
import { openFileLibrary } from '../../packages/ui/file-library.js';
import { renderGeneralBoardPanel } from '../../packages/ui/board-config/general-panel.js';
import {
  renderBrand as renderGlobalBrand,
  renderUserMenu as renderGlobalUserMenu,
  renderNotificationButton as renderGlobalNotificationButton,
  renderTopbar as renderGlobalTopbar,
  renderActionsGroup as renderTopbarActionsGroup,
} from '../../packages/ui/global-topbar.js';
import { createRichNotificationsStore, defaultRichState as notifDefaultState } from '../../packages/ui/notifications-rich.js';
import { createActivityStore, defaultActivityState } from '../../packages/ui/activity.js';

const app = document.getElementById('app');
const handleAppClick = event => {
  const actionEl = event.target instanceof Element ? event.target.closest('[data-action]') : null;
  if (!actionEl) {
    return;
  }
  const action = actionEl.dataset.action;
  switch (action) {
    case 'create-board':
      event.preventDefault();
      if (sessionState) {
        promptCreateBoard(sessionState);
      }
      return;
    case 'open-modules':
      event.preventDefault();
      goToModules();
      return;
    case 'open-account':
      event.preventDefault();
      goToAccount();
      return;
    case 'open-admin':
      event.preventDefault();
      goToAdmin();
      return;
    case 'open-notifications':
      event.preventDefault();
      toggleNotificationsOverlay();
      return;
    case 'open-activity':
      event.preventDefault();
      (async () => { try { await toggleActivityOverlay(); } catch (e) { console.error(e); } })();
      return;
    case 'open-file-library':
      event.preventDefault();
      (async () => { try { await openFileLibrary({ title: 'Mes fichiers', showSelect: false }); } catch (_) {} })();
      return;
    case 'logout':
      event.preventDefault();
      logout();
      return;
    case 'open-board-settings':
      event.preventDefault();
      openBoardSettings(actionEl.dataset.board);
      return;
    default:
      return;
  }
};
let csrfToken = '';
let sessionState = null;
let boardsState = [];
let logoutPending = false;
let richOverlay = null;
let richKeyHandler = null;
let notificationsState = notifDefaultState();
const notificationsController = createRichNotificationsStore({
  initialState: notificationsState,
  onChange: handleRichChange,
});
notificationsState = notificationsController.getState();

let activityOverlay = null;
let activityKeyHandler = null;
let activityState = defaultActivityState();
const activityController = createActivityStore({
  initialState: activityState,
  onChange: handleActivityChange,
});
activityState = activityController.getState();

init().catch(error => {
  console.error(error);
  if (app) {
    app.innerHTML = '<div class="loading">Erreur de chargement</div>';
  }
});

async function init() {
  // Allow enabling debug via ?debug=1|0
  try { applyDebugFromUrl(); } catch (_) {}
  const session = await ensureAuthenticated();
  if (!session) return;
  csrfToken = session.csrf;
  sessionState = session;
  try { setCsrfProvider(() => csrfToken); } catch (_) {}
  boardsState = await fetchBoards();
  render();
}

function render() {
  const session = sessionState;
  const boards = boardsState;
  if (!app) return;
  app.removeAttribute('data-loading');
  app.innerHTML = `
    ${renderAppTopbar(session)}
    ${boards.length ? renderGrid(boards) : renderEmpty()}
  `;
  app.querySelectorAll('[data-action="open-board"]').forEach(btn => btn.addEventListener('click', () => {
    const id = btn.dataset.board;
    goToBoard(id);
  }));
  app.querySelectorAll('[data-action="delete-board"]').forEach(btn => btn.addEventListener('click', () => handleDeleteBoard(btn.dataset.board)));
  syncNotificationButton();
}

function renderAppTopbar(session) {
  if (!session) return '';
  const brand = renderGlobalBrand({ subtitle: 'Mes boards' });
  const menuItems = [
    { id: 'open-account', label: 'Mon compte', icon: '👤' },
    { id: 'open-modules', label: 'Modules', icon: '🧩' },
    { id: 'open-file-library', label: 'Mes fichiers', icon: '📁' },
    ...(isAdmin(session.user) ? [{ id: 'open-admin', label: 'Administration', icon: '🛠️' }] : []),
  ];
  const userMenu = renderGlobalUserMenu(session.user, {
    items: menuItems,
    footer: [{ id: 'logout', label: 'Déconnexion', kind: 'danger', icon: '🚪' }],
  });
  const unreadCount = Array.isArray(notificationsState.categories)
    ? notificationsState.categories.reduce((acc, c) => acc + (Number(c?.unread ?? 0) || 0), 0)
    : 0;
  const notifications = renderGlobalNotificationButton({
    action: 'open-notifications',
    href: '/board.html',
    title: 'Ouvrir les notifications',
    count: unreadCount,
    open: !!notificationsState.open,
  });
  const primaryActions = renderTopbarActionsGroup(`
    <button class="btn" data-action="create-board">＋ Nouveau board</button>
    <button class="btn btn--icon ghost focus-ring" data-action="open-activity" aria-label="Activité">⟳</button>
  `);
  return renderGlobalTopbar({
    className: 'app-topbar--compact app-topbar--wrap',
    brand,
    actions: [primaryActions, userMenu, notifications],
  });
}

function renderGrid(boards) {
  return `
    <section class="board-grid">
      ${boards.map(board => `
        <article class="board-card">
          <div class="thumb"></div>
          <div class="title">${board.title}</div>
          ${board.thumbnail_public_id ? `<div class="board-card__image"><img src="/api/files/${encodeURIComponent(board.thumbnail_public_id)}/content" alt="" decoding="async" /></div>` : ''}
          <div class="meta" style="font-size:12px;opacity:.7;">Mis à jour le ${formatDate(board.updated_at)}</div>
          <div class="actions">
            <button class="btn" data-action="open-board" data-board="${board.id}">Ouvrir</button>
            <button class="btn ghost" data-action="open-board-settings" data-board="${board.id}">⚙️ Paramètres</button>
            <button class="btn ghost" data-action="delete-board" data-board="${board.id}">Supprimer</button>
          </div>
        </article>
      `).join('')}
    </section>
  `;
}

function renderEmpty() {
  return `
    <div class="empty-state">
      <div class="ico">☁️</div>
      <h2>Créez votre premier board</h2>
      <p>Organisez vos projets dans un espace clair et modulaire.</p>
      <button class="btn primary" data-action="create-board">＋ Nouveau board</button>
  </div>
  `;
}

function toggleNotificationsOverlay(force) {
  notificationsController.toggle(force).catch(console.error);
}

function toggleActivityOverlay(force) {
  return activityController.toggle(force);
}

function handleRichChange(next) {
  notificationsState = next;
  syncRichOverlay();
  syncNotificationButton();
}

function handleActivityChange(next) {
  activityState = next;
  syncActivityOverlay();
}

function syncRichOverlay() {
  if (!notificationsState.open) {
    if (richOverlay) {
      richOverlay.removeEventListener('click', handleRichOverlayAction);
      richOverlay.remove();
      richOverlay = null;
    }
    teardownRichKeyHandler();
    return;
  }
  if (!richOverlay) {
    richOverlay = document.createElement('div');
    richOverlay.className = 'notifications-overlay';
    richOverlay.dataset.action = 'rich-overlay';
    richOverlay.addEventListener('click', handleRichOverlayAction);
    document.body.appendChild(richOverlay);
  }
  // Render handled inside the store: we re-open to refresh content
  richOverlay.innerHTML = notificationsController.render(notificationsState, { wrap: false });
  ensureRichKeyHandler();
}

function handleRichOverlayAction(event) {
  if (!richOverlay) return;
  const actionEl = event.target instanceof Element ? event.target.closest('[data-action]') : null;
  if (!actionEl) {
    if (event.target === richOverlay) notificationsController.close();
    return;
  }
  const action = actionEl.dataset.action;
  switch (action) {
    case 'rich-overlay': {
      if (actionEl === richOverlay) notificationsController.close();
      break;
    }
    case 'rich-close':
      notificationsController.close();
      break;
    case 'rich-load-history':
      notificationsController.load('history').catch(console.error);
      break;
    case 'rich-load-unread':
      notificationsController.load('unread').catch(console.error);
      break;
    case 'rich-archive': {
      const id = Number(actionEl.dataset.id ?? 0);
      notificationsController.archiveMany([id]).catch(console.error);
      break;
    }
    case 'rich-delete': {
      const id = Number(actionEl.dataset.id ?? 0);
      notificationsController.deleteForUser(id).catch(console.error);
      break;
    }
    case 'rich-open': {
      const id = Number(actionEl.dataset.id ?? 0);
      notificationsController.select(id).catch?.(console.error);
      // Re-render to update preview
      syncRichOverlay();
      break;
    }
    case 'rich-sel-cat': {
      const categoryId = Number(actionEl.dataset.category ?? 0);
      notificationsController.selectCategory(categoryId);
      syncRichOverlay();
      break;
    }
    case 'rich-archive-cat': {
      const categoryId = Number(actionEl.dataset.category ?? 0);
      notificationsController.archiveCategory(categoryId).catch(console.error);
      break;
    }
    default:
      break;
  }
}

function ensureRichKeyHandler() {
  if (richKeyHandler) return;
  richKeyHandler = (event) => {
    if (event.key === 'Escape') {
      event.preventDefault();
      notificationsController.close();
    }
  };
  document.addEventListener('keydown', richKeyHandler);
}

function teardownRichKeyHandler() {
  if (!richKeyHandler) return;
  document.removeEventListener('keydown', richKeyHandler);
  richKeyHandler = null;
}

function syncActivityOverlay() {
  if (!activityState.open) {
    if (activityOverlay) {
      activityOverlay.removeEventListener('click', handleActivityOverlayAction);
      activityOverlay.remove();
      activityOverlay = null;
    }
    teardownActivityKeyHandler();
    return;
  }
  if (!activityOverlay) {
    activityOverlay = document.createElement('div');
    activityOverlay.className = 'notifications-overlay';
    activityOverlay.dataset.action = 'close-activity';
    activityOverlay.addEventListener('click', handleActivityOverlayAction);
    document.body.appendChild(activityOverlay);
  }
  activityOverlay.innerHTML = activityController.render(activityState, { wrap: false });
  ensureActivityKeyHandler();
}

function handleActivityOverlayAction(event) {
  if (!activityOverlay) return;
  const actionEl = event.target instanceof Element ? event.target.closest('[data-action]') : null;
  if (!actionEl) {
    if (event.target === activityOverlay) activityController.close();
    return;
  }
  const action = actionEl.dataset.action;
  if (action === 'close-activity') {
    activityController.close();
  }
}

function ensureActivityKeyHandler() {
  if (activityKeyHandler) return;
  activityKeyHandler = (event) => {
    if (event.key === 'Escape') {
      event.preventDefault();
      activityController.close();
    }
  };
  document.addEventListener('keydown', activityKeyHandler);
}

function teardownActivityKeyHandler() {
  if (!activityKeyHandler) return;
  document.removeEventListener('keydown', activityKeyHandler);
  activityKeyHandler = null;
}

function syncNotificationButton() {
  if (!app) return;
  const button = app.querySelector('.app-topbar__notifications');
  if (!button) return;
  const count = Math.min(
    (Array.isArray(notificationsState.categories)
      ? notificationsState.categories.reduce((acc, c) => acc + (Number(c?.unread ?? 0) || 0), 0)
      : 0),
    99
  );
  button.dataset.count = count > 0 ? String(count) : '';
  button.dataset.open = notificationsState.open ? 'true' : 'false';
  button.classList.toggle('is-open', !!notificationsState.open);
  let badge = button.querySelector('.app-topbar__badge');
  if (count > 0) {
    if (!badge) {
      badge = document.createElement('span');
      badge.className = 'app-topbar__badge';
      badge.setAttribute('aria-hidden', 'true');
      badge.textContent = String(count);
      button.appendChild(badge);
    } else {
      badge.textContent = String(count);
    }
  } else if (badge) {
    badge.remove();
  }
}

if (app) {
  app.addEventListener('click', handleAppClick, { capture: false });
}

async function promptCreateBoard(session) {
  const title = (await promptDialog({ title: 'Créer un board', label: 'Nom du board', defaultValue: '' }))?.trim();
  if (!title) return;
  try {
    const board = await createBoardRemote(title, csrfToken);
    goToBoard(board.id);
  } catch (error) {
    console.error(error);
    const message = isDebug() ? toErrorMessage(error, 'Impossible de créer le board') : 'Impossible de créer le board';
    showToast(message, { kind: 'error' });
  }
}

async function handleRenameBoard(boardId) {
  if (!boardId) return;
  const board = boardsState.find(item => String(item.id) === String(boardId));
  const title = (await promptDialog({ title: 'Renommer le board', label: 'Nouveau nom', defaultValue: board?.title ?? '' }))?.trim();
  if (!title || !title.length) {
    return;
  }
  try {
    await renameBoardRemote(boardId, title, csrfToken);
    boardsState = await fetchBoards();
    render();
    showToast('Board renommé', { kind: 'success' });
  } catch (error) {
    console.error(error);
    showToast('Impossible de renommer le board', { kind: 'error' });
  }
}

function openBoardSettings(boardId) {
  if (!boardId) return;
  const board = boardsState.find(item => String(item.id) === String(boardId));
  const modal = document.createElement('sb-modal');
  const container = document.createElement('div');
  container.style.display = 'grid';
  container.style.gap = '16px';
  const header = document.createElement('div');
  header.innerHTML = `<div style="display:flex;align-items:center;justify-content:space-between;gap:12px;">
    <div>
      <h3 style="margin:0;">Configuration du board</h3>
      <p style="margin:4px 0 0 0;opacity:.8;">${escapeHtml(board?.title ?? 'Board')}</p>
    </div>
    <button type="button" class="btn btn--icon ghost" aria-label="Fermer">✕</button>
  </div>`;
  const closeBtn = header.querySelector('button');
  closeBtn?.addEventListener('click', () => modal.close());
  const body = document.createElement('div');
  renderGeneralBoardPanel(body, {
    boardTitle: board?.title ?? '',
    boardId: String(boardId),
    busy: false,
    error: null,
    updatedAt: board?.updated_at ? new Date(Number(board.updated_at < 1e12 ? board.updated_at * 1000 : board.updated_at)).toLocaleString('fr-FR') : '',
    createdAt: board?.created_at ? new Date(Number(board.created_at < 1e12 ? board.created_at * 1000 : board.created_at)).toLocaleString('fr-FR') : '',
    thumbnailPublicId: typeof board?.thumbnail_public_id === 'string' ? board.thumbnail_public_id : '',
  });
  container.append(header, body);
  modal.appendChild(container);
  // Focus the title input initially
  setTimeout(() => {
    try {
      const input = body.querySelector('[data-ref="board-title"]');
      if (input && typeof input.focus === 'function') { input.focus(); input.select?.(); }
    } catch (_) {}
  }, 50);
  // Handle submit
  modal.addEventListener('submit', async (event) => {
    const form = event.target;
    if (!(form instanceof HTMLFormElement)) return;
    if (form.dataset.action !== 'board-config-save-general') return;
    event.preventDefault();
    const input = form.querySelector('[data-ref="board-title"]');
    const title = (input?.value ?? '').trim();
    const initialTitle = board?.title ?? '';
    const doRename = !!title && title !== initialTitle;
    const initialThumb = form?.dataset?.thumbnailInitial ?? '';
    const nextThumb = form?.dataset?.thumbnailCurrent ?? initialThumb;
    const doSetThumb = nextThumb !== initialThumb;
    // Disable controls
    renderGeneralBoardPanel(body, { boardTitle: title || initialTitle, boardId: String(boardId), busy: true, error: null, updatedAt: '', createdAt: '', thumbnailPublicId: nextThumb || initialThumb });
    if (closeBtn) closeBtn.disabled = true;
    try {
      if (doRename) {
        await renameBoardRemote(boardId, title, csrfToken);
      }
      if (doSetThumb) {
        await setBoardThumbnailRemote(boardId, nextThumb, csrfToken);
      }
      boardsState = await fetchBoards();
      render();
      showToast('Paramètres enregistrés', { kind: 'success' });
      modal.close();
    } catch (error) {
      console.error(error);
      showToast('Enregistrement impossible', { kind: 'error' });
      // Re-enable controls
      renderGeneralBoardPanel(body, { boardTitle: title || initialTitle, boardId: String(boardId), busy: false, error: 'Échec de la sauvegarde', updatedAt: '', createdAt: '', thumbnailPublicId: nextThumb || initialThumb });
      if (closeBtn) closeBtn.disabled = false;
    }
  });
  document.body.appendChild(modal);
  modal.show();
}

function escapeHtml(s) {
  return String(s ?? '').replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
}

async function handleDeleteBoard(boardId) {
  if (!boardId) return;
  const board = boardsState.find(item => String(item.id) === String(boardId));
  const ok = await confirmDialog({ title: 'Supprimer le board', message: `Supprimer le board « ${board?.title ?? 'sans nom'} » ?`, okLabel: 'Supprimer', cancelLabel: 'Annuler' });
  if (!ok) {
    return;
  }
  try {
    await deleteBoardRemote(boardId, csrfToken);
    boardsState = await fetchBoards();
    render();
    showToast('Board supprimé', { kind: 'success' });
  } catch (error) {
    console.error(error);
    showToast('Impossible de supprimer le board', { kind: 'error' });
  }
}

async function logout() {
  if (logoutPending) return;
  logoutPending = true;
  try {
    await requestJson('/api/commands', { method: 'POST', body: { type: 'Account.Logout', payload: {} } });
  } catch (error) {
    console.error('LOGOUT_FAILED', error);
  }
  goToAuth();
}

function isAdmin(user) {
  const role = (user?.role ?? 'standard').toLowerCase();
  return role === 'admin' || role === 'superadmin';
}

function formatDate(value) {
  const n = Number(value);
  const ms = Number.isFinite(n) ? (n < 1e12 ? n * 1000 : n) : Date.parse(String(value));
  const d = new Date(ms);
  return Number.isNaN(d.getTime()) ? '' : d.toLocaleDateString('fr-FR');
}
