import "../../packages/ui/menu.js";
import { ensureAuthenticated } from "../../packages/services/session.js";
import { requestJson, setCsrfProvider } from "../../packages/services/http.js";
import {
  renderBrand as renderGlobalBrand,
  renderUserMenu as renderGlobalUserMenu,
  renderNotificationButton as renderGlobalNotificationButton,
  renderTopbar as renderGlobalTopbar,
  renderActionsGroup as renderTopbarActionsGroup,
} from "../../packages/ui/global-topbar.js";
import { goToBoards, goToAccount, goToAdmin, goToBoardNotifications, goToModules, goToAuth } from "../../packages/services/navigation.js";
import { openFileLibrary } from "../../packages/ui/file-library.js";
import { capitalize, escapeHtml } from "../../packages/utils/format.js";

const app = document.getElementById("app");

const state = {
  loading: true,
  packs: [],
  activePackId: null,
  meta: null,
  error: null,
  pending: new Set(),
};

let sessionRef = null;
let logoutPending = false;

if (app) {
  app.addEventListener('click', handleGlobalAction);
}

init().catch(error => {
  console.error(error);
  if (app) {
    app.removeAttribute("data-loading");
    app.innerHTML = `
      <div class="modules-error surface-card surface-card--glass surface-card--bordered stack stack--gap-sm">
        <strong>Erreur de chargement des modules.</strong>
        <p>${escapeHtml(error?.message ?? "Veuillez réessayer plus tard.")}</p>
      </div>
    `;
  }
});

async function init() {
  const session = await ensureAuthenticated();
  if (!session) return;
  sessionRef = session;
  try { setCsrfProvider(() => session.csrf); } catch (_) {}
  await loadConfig();
}

async function loadConfig() {
  try {
    const config = await requestJson('/api/packs/config', { method: 'GET' });
    state.loading = false;
    state.packs = Array.isArray(config?.packs) ? config.packs : [];
    state.meta = config?.meta ?? null;
    state.error = null;
    ensureActivePackId();
    renderApp();
  } catch (error) {
    console.error('PACKS_CONFIG_LOAD_FAILED', error);
    state.loading = false;
    state.error = { message: 'Impossible de charger la configuration des packs.' };
    renderApp();
  }
}

function renderApp() {
  if (!app) return;
  const activePack = ensureActivePackId();

  app.removeAttribute("data-loading");

  if (state.loading) {
    app.innerHTML = `<div class="loading">Chargement des modules…</div>`;
    return;
  }

  app.innerHTML = `
    ${renderAppTopbar(sessionRef)}
    <section class="app-section-stack app-section-stack--page">
      ${renderErrorBanner(state.error)}
      ${renderTabs(state.packs, state.activePackId)}
      ${activePack ? renderPackView(activePack) : renderEmptyState()}
    </section>
  `;

  bindTabEvents();
  bindToggleEvents();
}

function renderAppTopbar(session) {
  if (!session) return '';
  const brand = renderGlobalBrand({ subtitle: 'Modules & packs MCC' });
  const adminQuickLink = isAdmin(session.user)
    ? `<button class="btn ghost" data-action="open-admin">Administration</button>`
    : '';
  const items = [
    { id: 'open-boards', label: 'Mes boards', icon: '📋' },
    { id: 'open-account', label: 'Mon compte', icon: '👤' },
    { id: 'open-file-library', label: 'Mes fichiers', icon: '📁' },
    ...(isAdmin(session.user) ? [{ id: 'open-admin', label: 'Administration', icon: '🛠️' }] : []),
  ];
  const menu = renderGlobalUserMenu(session.user, {
    items,
    footer: [{ id: 'logout', label: 'Déconnexion', kind: 'danger', icon: '🚪' }],
  });
  const notifications = renderGlobalNotificationButton({
    href: '/board.html',
    title: 'Ouvrir les notifications',
    count: 0,
    action: 'open-notifications',
  });
  const primaryActions = renderTopbarActionsGroup(`
    <button class="btn" data-action="open-boards">Mes boards</button>
    ${adminQuickLink}
  `);
  return renderGlobalTopbar({
    className: 'app-topbar--wide app-topbar--wrap',
    brand,
    actions: [primaryActions, menu, notifications],
  });
}

function renderErrorBanner(error) {
  if (!error || !error.message) return '';
  return `
    <div class="modules-alert surface-card surface-card--muted surface-card--bordered surface-card--compact" role="alert">
      <span>${escapeHtml(error.message)}</span>
    </div>
  `;
}

function renderTabs(packs, activeId) {
  if (!packs.length) return '';
  const tabs = packs.map(pack => {
    const isActive = pack.id === activeId;
    const pending = state.pending.has(pack.id);
    const locked = pack?.access?.locked && !pack.enabled;
    return `
      <div class="app-tab-card surface-card surface-card--glass surface-card--bordered surface-card--compact ${isActive ? 'is-active' : ''}" data-pack-tab="${escapeHtml(pack.id)}">
        <button class="app-tab-card__button" type="button" data-pack-select="${escapeHtml(pack.id)}">
          <span class="app-tab-card__title">${escapeHtml(pack.name ?? pack.id)}</span>
          ${renderStatusChip(pack)}
        </button>
        <label class="app-toggle ${pending ? 'is-pending' : ''} ${locked ? 'is-disabled' : ''}" data-pack-toggle="${escapeHtml(pack.id)}" title="${escapeHtml(toggleTooltip(pack, locked))}">
          <input type="checkbox" ${pack.enabled ? 'checked' : ''} ${pending || locked ? 'disabled' : ''} aria-label="Activer le pack ${escapeHtml(pack.name ?? pack.id)}">
          <span class="app-toggle__visual"></span>
        </label>
      </div>
    `;
  }).join('');
  return `<nav class="app-tabs">${tabs}</nav>`;
}

function renderStatusChip(pack) {
  const status = String(pack.status ?? '').toLowerCase();
  const map = {
    active: { label: 'Actif', tone: 'success' },
    available: { label: 'Disponible', tone: 'info' },
    beta: { label: 'Bêta', tone: 'info' },
    coming_soon: { label: 'À venir', tone: 'muted' },
    unavailable: { label: 'Indisponible', tone: 'muted' },
    unknown: { label: 'Inconnu', tone: 'muted' },
  };
  const info = map[status] ?? { label: capitalize(status || 'État'), tone: 'info' };
  return `<span class="badge badge--${info.tone} badge--caps">${escapeHtml(info.label)}</span>`;
}

function renderPackView(pack) {
  const summary = renderSummary(pack);
  const highlights = renderHighlights(pack.highlights ?? []);
  const descriptionHtml = pack.descriptionHtml || '';
  return `
    <div class="modules-content app-grid app-grid--split">
      <div class="modules-main surface-card surface-card--glass surface-card--bordered stack stack--gap-lg">
        <header class="modules-main__header">
          <h1>${escapeHtml(pack.name ?? pack.id)}</h1>
          ${pack.tagline ? `<p class="modules-main__tagline">${escapeHtml(pack.tagline)}</p>` : ''}
        </header>
        <div class="modules-main__body">
          <div class="modules-description" data-pack-description>${descriptionHtml}</div>
          ${highlights}
        </div>
      </div>
      <aside class="modules-aside">
        ${summary}
        ${renderDetailSection('Tags système', pack.technical?.tags ?? [], renderTagItem)}
        ${renderDetailSection('Règles déposées', pack.technical?.rules ?? [], renderRuleItem)}
        ${renderDetailSection('Datasets', pack.technical?.datasets ?? [], renderDatasetItem)}
        ${renderDetailSection('Slots UI', pack.technical?.slots ?? [], renderSlotItem)}
      </aside>
    </div>
  `;
}

function renderEmptyState() {
  return `
    <div class="modules-empty surface-card surface-card--muted surface-card--bordered stack stack--gap-sm">
      <h2>Aucun pack disponible</h2>
      <p>La configuration courante ne référence aucun pack. Ajoutez un manifest MCC pour commencer.</p>
    </div>
  `;
}

function renderHighlights(items) {
  if (!items || !items.length) return '';
  const cards = items.map(item => `
    <article class="modules-highlight surface-card surface-card--muted surface-card--bordered surface-card--compact">
      ${item.icon ? `<div class="modules-highlight__icon">${escapeHtml(item.icon)}</div>` : ''}
      <div>
        ${item.title ? `<h3>${escapeHtml(item.title)}</h3>` : ''}
        ${item.description ? `<p>${escapeHtml(item.description)}</p>` : ''}
      </div>
    </article>
  `).join('');
  return `<div class="modules-highlights">${cards}</div>`;
}

function renderSummary(pack) {
  const technical = pack.technical?.pack ?? {};
  const access = pack.access ?? {};
  const lines = [];
  lines.push({ label: 'Statut', value: formatPackStatus(pack) });
  if (technical.version) {
    lines.push({ label: 'Version', value: escapeHtml(String(technical.version)) });
  }
  if (technical.engine) {
    lines.push({ label: 'Compatibilité moteur', value: escapeHtml(String(technical.engine)) });
  }
  lines.push({ label: 'Installé', value: pack.installed ? 'Oui' : 'Non' });
  lines.push({ label: 'Activé', value: pack.enabled ? 'Oui' : 'Non' });
  if (access.requiresLicense) {
    lines.push({
      label: 'Licence requise',
      value: escapeHtml(access.licenseTier ? `Niveau ${access.licenseTier}` : 'Oui'),
    });
  }
  if (access.licenseNotes) {
    lines.push({ label: 'Notes licence', value: escapeHtml(access.licenseNotes) });
  }
  const items = lines.map(line => `
    <div class="app-kv">
      <span class="app-kv__label">${escapeHtml(line.label)}</span>
      <span>${line.value}</span>
    </div>
  `).join('');
  return `
    <section class="surface-card surface-card--glass surface-card--bordered app-card stack stack--gap-sm">
      <h2>Détails</h2>
      <div class="app-kv-list">${items}</div>
    </section>
  `;
}

function formatPackStatus(pack) {
  if (pack.enabled) return 'Actif';
  const status = String(pack.status ?? '').toLowerCase();
  switch (status) {
    case 'available': return 'Disponible';
    case 'beta': return 'Bêta';
    case 'coming_soon': return 'À venir';
    case 'unavailable': return 'Indisponible';
    case 'unknown': return 'Inconnu';
    default: return capitalize(status || 'État inconnu');
  }
}

function renderDetailSection(title, items, renderItem) {
  const content = (!items || !items.length)
    ? '<p class="app-card__empty">Aucune donnée déclarée.</p>'
    : `<ul class="app-detail-list">${items.map(renderItem).join('')}</ul>`;
  return `
    <section class="surface-card surface-card--glass surface-card--bordered app-card stack stack--gap-sm">
      <h2>${escapeHtml(title)}</h2>
      ${content}
    </section>
  `;
}

function renderTagItem(tag) {
  const badge = [];
  if (tag.icon) badge.push(`<span class="app-detail__icon">${escapeHtml(tag.icon)}</span>`);
  if (tag.label) badge.push(`<strong>${escapeHtml(tag.label)}</strong>`);
  return `
    <li class="app-detail">
      <div class="app-detail__main">
        ${badge.join(' ')}
        <code>${escapeHtml(tag.key)}</code>
      </div>
      <div class="app-detail__meta">
        ${tag.category ? `<span>${escapeHtml(tag.category)}</span>` : ''}
        ${tag.hiddenInPicker ? `<span>masqué picker</span>` : ''}
      </div>
    </li>
  `;
}

function renderRuleItem(rule) {
  return `
    <li class="app-detail">
      <div class="app-detail__main">
        <strong>${escapeHtml(rule.id ?? '')}</strong>
        ${rule.trigger ? `<span>· ${escapeHtml(rule.trigger)}</span>` : ''}
      </div>
      <div class="app-detail__meta">
        ${rule.scope ? `<span>${escapeHtml(rule.scope)}</span>` : ''}
        ${Number.isFinite(rule.priority) ? `<span>prio ${escapeHtml(String(rule.priority))}</span>` : ''}
        <span>${rule.actionsCount ?? 0} action(s)</span>
      </div>
    </li>
  `;
}

function renderDatasetItem(dataset) {
  return `
    <li class="app-detail">
      <div class="app-detail__main">
        <code>${escapeHtml(dataset.id ?? '')}</code>
      </div>
      <div class="app-detail__meta">
        ${dataset.kind ? `<span>${escapeHtml(dataset.kind)}</span>` : ''}
        ${dataset.path ? `<span>${escapeHtml(dataset.path)}</span>` : ''}
      </div>
    </li>
  `;
}

function renderSlotItem(slot) {
  return `
    <li class="app-detail">
      <div class="app-detail__main">
        <code>${escapeHtml(slot.id ?? '')}</code>
      </div>
      <div class="app-detail__meta">
        ${slot.component ? `<span>${escapeHtml(slot.component)}</span>` : ''}
        ${slot.type ? `<span>${escapeHtml(slot.type)}</span>` : ''}
      </div>
    </li>
  `;
}

function bindTabEvents() {
  if (!app) return;
  app.querySelectorAll('[data-pack-select]').forEach(button => {
    button.addEventListener('click', event => {
      const id = event.currentTarget?.getAttribute('data-pack-select');
      if (id) {
        state.activePackId = id;
        renderApp();
      }
    });
  });
}

function bindToggleEvents() {
  if (!app) return;
  app.querySelectorAll('[data-pack-toggle]').forEach(wrapper => {
    const packId = wrapper.getAttribute('data-pack-toggle');
    const input = wrapper.querySelector('input[type="checkbox"]');
    if (!packId || !input) return;
    input.addEventListener('click', event => event.stopPropagation());
    input.addEventListener('change', event => {
      event.stopPropagation();
      const target = event.currentTarget;
      const nextValue = !!(target && typeof target.checked === 'boolean' ? target.checked : input.checked);
      updatePackToggle(packId, nextValue);
    });
  });
}

function handleGlobalAction(event) {
  const target = event.target instanceof Element ? event.target.closest('[data-action]') : null;
  if (!target || !app || !app.contains(target)) return;
  const action = target.dataset.action;
  switch (action) {
    case 'open-boards':
      event.preventDefault();
      goToBoards();
      break;
    case 'open-account':
      event.preventDefault();
      goToAccount();
      break;
    case 'open-admin':
      event.preventDefault();
      goToAdmin();
      break;
    case 'open-notifications':
      event.preventDefault();
      goToBoardNotifications();
      break;
    case 'open-file-library':
      event.preventDefault();
      (async () => { try { await openFileLibrary({ title: 'Mes fichiers', showSelect: false }); } catch (_) {} })();
      break;
    case 'open-modules':
      event.preventDefault();
      goToModules();
      break;
    case 'logout':
      event.preventDefault();
      logout();
      break;
    default:
      break;
  }
}

async function logout() {
  if (logoutPending) return;
  logoutPending = true;
  try {
    await requestJson('/api/commands', { method: 'POST', body: { type: 'Account.Logout', payload: {} } });
  } catch (error) {
    console.error('LOGOUT_FAILED', error);
  }
  goToAuth();
}

function isAdmin(user) {
  const role = (user?.role ?? 'standard').toString().toLowerCase();
  return role === 'admin' || role === 'superadmin';
}

async function updatePackToggle(packId, nextValue) {
  if (!packId) return;
  state.pending.add(packId);
  state.error = null;
  renderApp();
  try {
    const result = await requestJson('/api/packs/config', {
      method: 'PUT',
      body: { packs: { [packId]: nextValue } },
    });
    state.pending.delete(packId);
    state.packs = Array.isArray(result?.packs) ? result.packs : state.packs;
    state.meta = result?.meta ?? state.meta;
    ensureActivePackId();
    renderApp();
  } catch (error) {
    console.error('PACK_TOGGLE_FAILED', error);
    state.pending.delete(packId);
    const message = error?.payload?.message || 'Impossible de modifier ce pack.';
    state.error = { message };
    renderApp();
  }
}

function ensureActivePackId() {
  if (!Array.isArray(state.packs) || state.packs.length === 0) {
    state.activePackId = null;
    return null;
  }
  if (state.activePackId) {
    const current = state.packs.find(pack => pack.id === state.activePackId);
    if (current) return current;
  }
  const fallback = state.packs.find(pack => pack.enabled) ?? state.packs[0];
  state.activePackId = fallback?.id ?? null;
  return fallback ?? null;
}

function toggleTooltip(pack, locked) {
  if (pack.enabled) return 'Pack activé';
  if (locked) {
    if (pack?.access?.reason === 'coming_soon') return 'Ce pack arrive bientôt.';
    if (pack?.access?.reason === 'not-installed') return 'Pack non installé.';
    return 'Activation indisponible.';
  }
  return 'Activer / désactiver ce pack';
}
