import { registerDocAdapter } from '../adapter-registry.js';
import { renderMarkdown } from '../../format/markdown.js';
import { showToast } from '../../ui/toast.js';

const adapter = {
  id: 'core:markdown-textarea',
  label: 'Markdown (textarea)',
  priority: 50,
  matches({ slot, attachment }) {
    if (slot && slot !== 'note') {
      return false;
    }
    if (!attachment) {
      return true;
    }
    const mime = typeof attachment.mimeType === 'string' ? attachment.mimeType.toLowerCase() : '';
    return !mime || mime.startsWith('text/') || mime.includes('markdown');
  },
  create({ container, session } = {}) {
    if (!container) {
      return () => {};
    }

    container.innerHTML = '';

    const wrapper = document.createElement('div');
    wrapper.className = 'sb-note-editor';

    const toolbar = document.createElement('div');
    toolbar.className = 'sb-note-panel__toolbar sb-note-editor__toolbar';

    const previewButton = document.createElement('button');
    previewButton.type = 'button';
    previewButton.className = 'btn ghost';
    previewButton.textContent = 'Aperçu';
    previewButton.setAttribute('aria-pressed', 'false');
    toolbar.appendChild(previewButton);

    const editor = document.createElement('textarea');
    editor.className = 'sb-note-panel__editor sb-note-editor__textarea';
    editor.rows = 12;
    editor.spellcheck = true;
    editor.placeholder = 'Écrivez en Markdown…';

    const preview = document.createElement('div');
    preview.className = 'sb-note-panel__preview sb-note-editor__preview';
    preview.style.display = 'none';

    const footer = document.createElement('div');
    footer.className = 'sb-note-panel__footer';

    const saveState = document.createElement('span');
    saveState.className = 'sb-note-panel__save-state';

    const saveButton = document.createElement('button');
    saveButton.type = 'button';
    saveButton.className = 'btn primary';
    saveButton.textContent = 'Enregistrer';

    footer.append(saveState, saveButton);

    wrapper.append(toolbar, editor, preview, footer);
    container.appendChild(wrapper);

    const localState = {
      preview: false,
      timer: null,
      lastRendered: '',
    };

    const updatePreview = (content) => {
      const value = typeof content === 'string' ? content : '';
      if (!localState.preview) {
        preview.innerHTML = '';
        return;
      }
      if (localState.lastRendered === value && !localState.timer) {
        return;
      }
      localState.lastRendered = value;
      if (typeof window === 'undefined') {
        preview.innerHTML = renderMarkdown(value);
        return;
      }
      if (localState.timer) {
        window.clearTimeout(localState.timer);
      }
      localState.timer = window.setTimeout(() => {
        localState.timer = null;
        preview.innerHTML = renderMarkdown(localState.lastRendered);
      }, 150);
    };

    const cancelPreview = () => {
      if (localState.timer && typeof window !== 'undefined') {
        window.clearTimeout(localState.timer);
      }
      localState.timer = null;
    };

    const update = (snapshot) => {
      const hasAttachment = !!(snapshot && snapshot.attachment);
      const busy = !!(snapshot && (snapshot.loading || snapshot.saving));
      const value = snapshot && typeof snapshot.content === 'string' ? snapshot.content : '';
      if (editor.value !== value) {
        editor.value = value;
      }
      editor.disabled = !hasAttachment || busy;
      editor.placeholder = hasAttachment ? 'Écrivez en Markdown…' : 'Associez un fichier pour éditer.';

      saveButton.disabled = !hasAttachment || busy || !(snapshot && snapshot.dirty);
      const saveTitle = !hasAttachment
        ? 'Aucun fichier sélectionné'
        : busy
          ? (snapshot.loading ? 'Chargement en cours' : 'Enregistrement en cours')
          : (!snapshot.dirty ? 'Aucune modification à enregistrer' : 'Enregistrer la note');
      if (saveTitle) {
        saveButton.title = saveTitle;
      } else {
        saveButton.removeAttribute('title');
      }

      if (!hasAttachment) {
        localState.preview = false;
        preview.style.display = 'none';
        preview.innerHTML = '';
        previewButton.disabled = true;
        previewButton.textContent = 'Aperçu';
        previewButton.setAttribute('aria-pressed', 'false');
      } else {
        previewButton.disabled = busy;
        previewButton.textContent = localState.preview ? 'Éditer' : 'Aperçu';
        previewButton.setAttribute('aria-pressed', localState.preview ? 'true' : 'false');
        preview.style.display = localState.preview ? '' : 'none';
      }

      if (!hasAttachment) {
        saveState.textContent = '';
      } else if (snapshot.loading) {
        saveState.textContent = 'Chargement…';
      } else if (snapshot.saving) {
        saveState.textContent = 'Enregistrement…';
      } else if (snapshot.dirty) {
        saveState.textContent = 'Modifications non enregistrées';
      } else {
        saveState.textContent = 'Enregistré';
      }

      if (localState.preview && hasAttachment) {
        updatePreview(value);
      } else {
        cancelPreview();
      }
    };

    const handleInput = () => {
      if (!session || typeof session.setContent !== 'function') {
        return;
      }
      session.setContent(editor.value);
    };

    const handleKeydown = (event) => {
      if ((event.metaKey || event.ctrlKey) && (event.key === 's' || event.key === 'S')) {
        event.preventDefault();
        saveButton.click();
      }
    };

    const handlePreview = () => {
      if (!session || typeof session.getState !== 'function') {
        return;
      }
      const snapshot = session.getState();
      if (!snapshot.attachment || snapshot.loading || snapshot.saving) {
        return;
      }
      localState.preview = !localState.preview;
      if (!localState.preview) {
        cancelPreview();
      }
      update(snapshot);
      if (localState.preview) {
        updatePreview(snapshot.content);
      }
    };

    const handleSave = async () => {
      if (!session || typeof session.save !== 'function' || typeof session.getState !== 'function') {
        return;
      }
      const snapshot = session.getState();
      if (!snapshot.attachment) {
        showToast('Aucun fichier Markdown sélectionné.');
        return;
      }
      try {
        const result = await session.save();
        if (result && result.ok) {
          showToast('Note enregistrée.');
        } else if (result && result.conflict) {
          showToast('Le fichier a été modifié ailleurs. Rechargement…');
          const reload = await session.load();
          if (!reload || !reload.ok) {
            showToast('Impossible de recharger le fichier Markdown.');
          }
        } else if (!result || !result.cancelled) {
          showToast('Impossible d’enregistrer la note.');
        }
      } catch (error) {
        console.error('DOC_MARKDOWN_SAVE_FAILED', error);
        showToast('Impossible d’enregistrer la note.');
      }
    };

    editor.addEventListener('input', handleInput);
    editor.addEventListener('keydown', handleKeydown);
    previewButton.addEventListener('click', handlePreview);
    saveButton.addEventListener('click', handleSave);

    const unsubscribe = session && typeof session.onChange === 'function'
      ? session.onChange((snapshot) => update(snapshot))
      : () => {};

    if (session && typeof session.getState === 'function') {
      update(session.getState());
    } else {
      update(null);
    }

    return () => {
      cancelPreview();
      unsubscribe?.();
      editor.removeEventListener('input', handleInput);
      editor.removeEventListener('keydown', handleKeydown);
      previewButton.removeEventListener('click', handlePreview);
      saveButton.removeEventListener('click', handleSave);
    };
  },
};

registerDocAdapter(adapter);

export default adapter;

