function cloneAttachment(attachment) {
  if (!attachment || typeof attachment !== 'object') {
    return null;
  }
  const copy = { ...attachment };
  if (attachment.meta && typeof attachment.meta === 'object') {
    copy.meta = { ...attachment.meta };
  }
  return copy;
}

function attachmentsEqual(a, b) {
  if (!a && !b) {
    return true;
  }
  if (!a || !b) {
    return false;
  }
  const leftId = typeof a.fileId === 'string' && a.fileId ? a.fileId : null;
  const rightId = typeof b.fileId === 'string' && b.fileId ? b.fileId : null;
  if (leftId !== rightId) {
    return false;
  }
  const leftChecksum = typeof a.checksum === 'string' ? a.checksum : null;
  const rightChecksum = typeof b.checksum === 'string' ? b.checksum : null;
  return leftChecksum === rightChecksum;
}

function resolveChecksum(attachment) {
  return attachment && typeof attachment.checksum === 'string' ? attachment.checksum : null;
}

function resolveMessage(message, fallback) {
  if (typeof message === 'string' && message.trim().length) {
    return message.trim();
  }
  if (message && typeof message.message === 'string' && message.message.trim().length) {
    return message.message.trim();
  }
  return fallback;
}

const noop = () => {};

export function createDocSession(options = {}) {
  const listeners = new Set();
  let destroyed = false;
  let loadRequestId = 0;
  let saveRequestId = 0;

  const initialAttachment = options.attachment ? cloneAttachment(options.attachment) : null;
  const initialContent = typeof options.content === 'string' ? options.content : '';
  const initialLoaded = typeof options.loadedContent === 'string'
    ? options.loadedContent
    : initialContent;

  const state = {
    slot: typeof options.slot === 'string' ? options.slot : null,
    attachment: initialAttachment,
    content: initialContent,
    loadedContent: initialLoaded,
    checksum: resolveChecksum(initialAttachment),
    loading: false,
    saving: false,
    dirty: initialContent !== initialLoaded,
    error: null,
    errorMessage: '',
  };

  const io = {
    read: typeof options.io?.read === 'function' ? options.io.read : async () => ({ ok: false, message: 'Lecture indisponible.' }),
    write: typeof options.io?.write === 'function' ? options.io.write : async () => ({ ok: false, message: 'Écriture indisponible.' }),
  };

  const attachmentListener = typeof options.onAttachmentChange === 'function'
    ? options.onAttachmentChange
    : null;

  function getState() {
    return {
      slot: state.slot,
      attachment: cloneAttachment(state.attachment),
      content: state.content,
      loadedContent: state.loadedContent,
      checksum: state.checksum,
      loading: state.loading,
      saving: state.saving,
      dirty: state.dirty,
      error: state.error,
      errorMessage: state.errorMessage,
    };
  }

  function notify() {
    if (destroyed) {
      return;
    }
    const snapshot = getState();
    listeners.forEach((listener) => {
      try {
        listener(snapshot);
      } catch (error) {
        console.error('DOC_SESSION_LISTENER_FAILED', error);
      }
    });
  }

  function onChange(listener) {
    if (typeof listener !== 'function') {
      return noop;
    }
    listeners.add(listener);
    return () => {
      listeners.delete(listener);
    };
  }

  function applyContent(content, loadedContent) {
    const value = typeof content === 'string' ? content : '';
    state.content = value;
    if (loadedContent !== undefined) {
      state.loadedContent = typeof loadedContent === 'string' ? loadedContent : '';
    }
    state.dirty = state.content !== state.loadedContent;
  }

  function applyAttachment(nextAttachment, options = {}) {
    const previousAttachment = cloneAttachment(state.attachment);
    const previousContent = state.content;
    const previousLoadedContent = state.loadedContent;
    const normalized = nextAttachment ? cloneAttachment(nextAttachment) : null;
    const reason = typeof options.reason === 'string' ? options.reason : 'update';
    const changed = !attachmentsEqual(previousAttachment, normalized);

    if (changed && reason !== 'load' && reason !== 'save') {
      loadRequestId += 1;
    }

    state.attachment = normalized;
    state.checksum = options.checksum !== undefined
      ? (typeof options.checksum === 'string' ? options.checksum : null)
      : resolveChecksum(normalized);
    if (options.content !== undefined) {
      const loaded = options.loadedContent !== undefined ? options.loadedContent : options.content;
      applyContent(options.content, loaded);
    } else if (!normalized) {
      applyContent('', '');
    } else if (changed) {
      applyContent('', '');
    } else {
      state.dirty = state.content !== state.loadedContent;
    }
    state.loading = false;
    state.saving = false;
    state.error = null;
    state.errorMessage = '';
    notify();

    if (!attachmentListener || options.silent) {
      return Promise.resolve(state.attachment);
    }

    const meta = {
      reason,
      silent: !!options.silent,
      triggerPersist: !!options.triggerPersist,
      previousAttachment,
      previousContent,
      previousLoadedContent,
    };

    try {
      const result = attachmentListener(state.attachment, meta);
      if (result && typeof result.then === 'function') {
        return result.then(() => state.attachment);
      }
      return Promise.resolve(state.attachment);
    } catch (error) {
      return Promise.reject(error);
    }
  }

  async function setAttachment(attachment, options = {}) {
    if (destroyed) {
      return Promise.resolve(null);
    }
    const resolved = attachment ? cloneAttachment(attachment) : null;
    return applyAttachment(resolved, options);
  }

  async function load() {
    if (destroyed) {
      return { ok: false, cancelled: true };
    }
    const attachment = state.attachment;
    if (!attachment || !attachment.fileId) {
      applyContent('', '');
      state.error = null;
      state.errorMessage = '';
      notify();
      return { ok: false, skipped: true };
    }
    const requestId = ++loadRequestId;
    state.loading = true;
    state.error = null;
    state.errorMessage = '';
    notify();

    let result;
    try {
      result = await io.read(cloneAttachment(attachment), { attachment: cloneAttachment(attachment) });
    } catch (error) {
      result = { ok: false, error, message: 'Impossible de lire le document.' };
    }

    if (requestId !== loadRequestId || destroyed) {
      return { ok: false, cancelled: true };
    }

    state.loading = false;

    if (!result || !result.ok) {
      state.error = result?.error || new Error('DOC_READ_FAILED');
      state.errorMessage = resolveMessage(result?.message, '⚠️ Impossible de lire le fichier Markdown.');
      notify();
      return { ok: false, error: state.error, message: state.errorMessage };
    }

    const nextAttachment = result.attachment ? cloneAttachment(result.attachment) : cloneAttachment(attachment);
    const nextContent = typeof result.content === 'string' ? result.content : '';

    try {
      await applyAttachment(nextAttachment, {
        reason: 'load',
        silent: !!result.silent,
        triggerPersist: !!result.triggerPersist,
        content: nextContent,
        loadedContent: nextContent,
        checksum: result.checksum ?? resolveChecksum(nextAttachment),
      });
      state.error = null;
      state.errorMessage = '';
      notify();
      return { ok: true, attachment: cloneAttachment(state.attachment), content: nextContent };
    } catch (error) {
      state.error = error;
      state.errorMessage = resolveMessage(error?.message, '⚠️ Impossible de traiter le document.');
      notify();
      return { ok: false, error: state.error, message: state.errorMessage };
    }
  }

  async function save() {
    if (destroyed) {
      return { ok: false, cancelled: true };
    }
    const attachment = state.attachment;
    if (!attachment || !attachment.fileId) {
      state.error = new Error('DOC_NO_ATTACHMENT');
      state.errorMessage = '⚠️ Aucun fichier Markdown sélectionné.';
      notify();
      return { ok: false, error: state.error, message: state.errorMessage };
    }
    const requestId = ++saveRequestId;
    state.saving = true;
    state.error = null;
    state.errorMessage = '';
    notify();

    let result;
    try {
      result = await io.write(cloneAttachment(attachment), state.content, { attachment: cloneAttachment(attachment) });
    } catch (error) {
      result = { ok: false, error, message: 'Impossible d\'enregistrer le document.' };
    }

    if (requestId !== saveRequestId || destroyed) {
      return { ok: false, cancelled: true };
    }

    state.saving = false;

    if (!result || !result.ok) {
      state.error = result?.error || new Error('DOC_WRITE_FAILED');
      state.errorMessage = resolveMessage(result?.message, '⚠️ Impossible d\'enregistrer la note.');
      notify();
      return { ok: false, error: state.error, message: state.errorMessage, conflict: !!result?.conflict };
    }

    const nextAttachment = result.attachment ? cloneAttachment(result.attachment) : cloneAttachment(attachment);
    const nextContent = typeof result.content === 'string' ? result.content : state.content;
    const persist = result.triggerPersist !== undefined ? !!result.triggerPersist : true;

    try {
      await applyAttachment(nextAttachment, {
        reason: 'save',
        silent: !!result.silent,
        triggerPersist: persist,
        content: nextContent,
        loadedContent: nextContent,
        checksum: result.checksum ?? resolveChecksum(nextAttachment),
      });
      state.error = null;
      state.errorMessage = '';
      notify();
      return { ok: true, attachment: cloneAttachment(state.attachment), content: nextContent };
    } catch (error) {
      state.error = error;
      state.errorMessage = resolveMessage(error?.message, '⚠️ Impossible d\'enregistrer la note.');
      notify();
      return { ok: false, error: state.error, message: state.errorMessage };
    }
  }

  function setContent(value) {
    if (destroyed) {
      return;
    }
    const text = typeof value === 'string' ? value : '';
    if (text === state.content && state.dirty === (text !== state.loadedContent)) {
      return;
    }
    state.content = text;
    state.dirty = state.content !== state.loadedContent;
    state.error = null;
    state.errorMessage = '';
    notify();
  }

  function destroy() {
    destroyed = true;
    listeners.clear();
    loadRequestId += 1;
    saveRequestId += 1;
  }

  return {
    getState,
    onChange,
    setAttachment,
    load,
    save,
    setContent,
    destroy,
  };
}

