import { listUserFiles, listUserFolders, uploadUserFiles, deleteUserFile as apiDeleteUserFile, renameUserFile as apiRenameUserFile, createUserFolder as apiCreateUserFolder, renameUserFolder as apiRenameUserFolder, deleteUserFolder as apiDeleteUserFolder, moveUserFile as apiMoveUserFile } from '../services/files.js';

let cachedFiles = [];
let loaded = false;
let loadingPromise = null;
let cachedFolders = [];
let foldersLoaded = false;
let foldersLoadingPromise = null;

export function resetUserFilesCache() {
  cachedFiles = [];
  loaded = false;
  loadingPromise = null;
}

export function getCachedUserFiles() {
  return cachedFiles.slice();
}

export function getCachedUserFolders() {
  return cachedFolders.slice();
}

export async function ensureUserFiles() {
  if (loaded) {
    return cachedFiles.slice();
  }
  if (loadingPromise) {
    await loadingPromise;
    return cachedFiles.slice();
  }
  loadingPromise = listUserFiles()
    .then((res) => {
      if (res.ok) {
        cachedFiles = Array.isArray(res.files) ? res.files : [];
        loaded = true;
      }
    })
    .finally(() => {
      loadingPromise = null;
    });
  await loadingPromise;
  return cachedFiles.slice();
}

export async function refreshUserFiles() {
  const res = await listUserFiles();
  if (res.ok) {
    cachedFiles = Array.isArray(res.files) ? res.files : [];
    loaded = true;
    return cachedFiles.slice();
  }
  return null;
}

export async function uploadFiles(files, folderId = '') {
  const res = await uploadUserFiles(files, folderId);
  if (res.ok) {
    const uploaded = Array.isArray(res.files) ? res.files : [];
    if (uploaded.length) {
      cachedFiles = [...uploaded, ...cachedFiles];
      loaded = true;
    }
    return { ok: true, files: uploaded };
  }
  return { ok: false, error: res.error };
}

export async function deleteUserFile(id) {
  const res = await apiDeleteUserFile(id);
  if (res.ok) {
    cachedFiles = cachedFiles.filter((file) => file.id !== id);
    return { ok: true };
  }
  return { ok: false, error: res.error };
}

export async function renameUserFile(id, name) {
  const res = await apiRenameUserFile(id, name);
  if (res.ok && res.file) {
    cachedFiles = cachedFiles.map((file) => (file.id === id ? res.file : file));
    return { ok: true, file: res.file };
  }
  return { ok: false, error: res.error };
}

export function upsertUserFile(file) {
  if (!file || typeof file !== 'object') {
    return;
  }
  const id = file.id;
  if (!id) {
    return;
  }
  const index = cachedFiles.findIndex((entry) => entry.id === id);
  if (index >= 0) {
    cachedFiles[index] = file;
  } else {
    cachedFiles = [file, ...cachedFiles];
    loaded = true;
  }
}

export async function ensureUserFolders() {
  if (foldersLoaded) {
    return cachedFolders.slice();
  }
  if (foldersLoadingPromise) {
    await foldersLoadingPromise;
    return cachedFolders.slice();
  }
  foldersLoadingPromise = listUserFolders()
    .then((res) => {
      if (res.ok) {
        cachedFolders = Array.isArray(res.folders) ? res.folders : [];
        foldersLoaded = true;
      }
    })
    .finally(() => { foldersLoadingPromise = null; });
  await foldersLoadingPromise;
  return cachedFolders.slice();
}

export async function refreshUserFolders() {
  const res = await listUserFolders();
  if (res.ok) {
    cachedFolders = Array.isArray(res.folders) ? res.folders : [];
    foldersLoaded = true;
    return cachedFolders.slice();
  }
  return null;
}

export async function createFolder(name, parentFolderId = null) {
  const res = await apiCreateUserFolder(name, parentFolderId);
  if (res.ok && res.folder) {
    cachedFolders = [res.folder, ...cachedFolders];
    foldersLoaded = true;
    return { ok: true, folder: res.folder };
  }
  return { ok: false, error: res.error };
}

export async function moveFile(id, folderId = null) {
  const res = await apiMoveUserFile(id, folderId);
  if (res.ok && res.file) {
    // Update local file record
    const idx = cachedFiles.findIndex(f => f.id === id || f.fileId === id);
    if (idx >= 0) {
      cachedFiles[idx] = { ...cachedFiles[idx], ...res.file };
    }
    return { ok: true, file: res.file };
  }
  return { ok: false, error: res.error };
}

export async function renameFolder(id, name) {
  const res = await apiRenameUserFolder(id, name);
  if (res.ok && res.folder) {
    // Préserver parentId si manquant dans la réponse (backend ne renvoie pas toujours parentId)
    cachedFolders = cachedFolders.map((f) => {
      if (f.id !== id) return f;
      const next = { ...f, ...res.folder };
      if (!next.parentId && f.parentId) next.parentId = f.parentId;
      return next;
    });
    return { ok: true, folder: res.folder };
  }
  return { ok: false, error: res.error };
}

export async function deleteFolder(id) {
  const res = await apiDeleteUserFolder(id);
  if (res.ok) {
    cachedFolders = cachedFolders.filter(f => f.id !== id);
    return { ok: true };
  }
  return { ok: false, error: res.error };
}
