import { requestJson } from './http.js';

export async function listUserFiles() {
  try {
    const res = await requestJson('/api/files', { method: 'GET' });
    const files = Array.isArray(res?.files) ? res.files : [];
    return { ok: true, files };
  } catch (error) {
    console.error('USER_FILES_LIST_FAILED', error);
    return { ok: false, error };
  }
}

export async function listUserFolders() {
  try {
    const res = await requestJson('/api/files/folders', { method: 'GET' });
    const folders = Array.isArray(res?.folders) ? res.folders : [];
    return { ok: true, folders };
  } catch (error) {
    console.error('USER_FOLDERS_LIST_FAILED', error);
    return { ok: false, error };
  }
}

export async function uploadUserFiles(files, folderId = '') {
  const form = new FormData();
  const array = normalizeFilesArray(files);
  array.forEach((file) => {
    form.append('file', file);
  });
  // Envoyer via /api/commands (File.Upload), en multipart
  form.append('type', 'File.Upload');
  const payload = {};
  if (typeof folderId === 'string' && folderId) {
    payload.folderId = folderId;
  }
  form.append('payload', JSON.stringify(payload));
  try {
    const res = await requestJson('/api/commands', { method: 'POST', body: form });
    if (Array.isArray(res?.files)) {
      return { ok: true, files: res.files };
    }
    if (res?.file) {
      return { ok: true, files: [res.file] };
    }
    return { ok: false, error: new Error('UPLOAD_RESPONSE_INVALID') };
  } catch (error) {
    console.error('USER_FILES_UPLOAD_FAILED', error);
    return { ok: false, error };
  }
}

export async function deleteUserFile(id) {
  try {
    const res = await requestJson('/api/commands', { method: 'POST', body: { type: 'File.Delete', payload: { fileId: id } } });
    if (res?.ok) {
      return { ok: true };
    }
    return { ok: false, error: new Error(res?.error || 'DELETE_FAILED') };
  } catch (error) {
    console.error('USER_FILES_DELETE_FAILED', error);
    return { ok: false, error };
  }
}

export async function renameUserFile(id, name) {
  try {
    const res = await requestJson('/api/commands', { method: 'POST', body: { type: 'File.Rename', payload: { fileId: id, name } } });
    if (res?.ok && res?.file) {
      return { ok: true, file: res.file };
    }
    return { ok: false, error: new Error(res?.error || 'RENAME_FAILED') };
  } catch (error) {
    console.error('USER_FILES_RENAME_FAILED', error);
    return { ok: false, error };
  }
}

export async function createUserFolder(name, parentFolderId = null) {
  try {
    const body = { type: 'File.CreateFolder', payload: { name } };
    if (typeof parentFolderId === 'string' && parentFolderId) {
      body.payload.parentFolderId = parentFolderId;
    }
    const res = await requestJson('/api/commands', { method: 'POST', body });
    if (res?.ok && res?.folder) {
      return { ok: true, folder: res.folder };
    }
    return { ok: false, error: new Error(res?.error || 'CREATE_FOLDER_FAILED') };
  } catch (error) {
    console.error('USER_FOLDER_CREATE_FAILED', error);
    return { ok: false, error };
  }
}

export async function renameUserFolder(id, name) {
  try {
    const res = await requestJson('/api/commands', { method: 'POST', body: { type: 'File.RenameFolder', payload: { folderId: id, name } } });
    if (res?.ok && res?.folder) {
      return { ok: true, folder: res.folder };
    }
    return { ok: false, error: new Error(res?.error || 'RENAME_FOLDER_FAILED') };
  } catch (error) {
    console.error('USER_FOLDER_RENAME_FAILED', error);
    return { ok: false, error };
  }
}

export async function deleteUserFolder(id) {
  try {
    const res = await requestJson('/api/commands', { method: 'POST', body: { type: 'File.DeleteFolder', payload: { folderId: id } } });
    if (res?.ok) {
      return { ok: true };
    }
    return { ok: false, error: new Error(res?.error || 'DELETE_FOLDER_FAILED') };
  } catch (error) {
    console.error('USER_FOLDER_DELETE_FAILED', error);
    return { ok: false, error };
  }
}

export async function moveUserFile(id, folderId = null) {
  try {
    const payload = { fileId: id };
    if (typeof folderId === 'string') {
      payload.folderId = folderId || '';
    }
    const res = await requestJson('/api/commands', { method: 'POST', body: { type: 'File.Move', payload } });
    if (res?.ok && res?.file) {
      return { ok: true, file: res.file };
    }
    return { ok: false, error: new Error(res?.error || 'MOVE_FAILED') };
  } catch (error) {
    console.error('USER_FILE_MOVE_FAILED', error);
    return { ok: false, error };
  }
}

export async function createUserTextFile({ name, content } = {}) {
  try {
    const body = {
      name,
      content,
      kind: 'text',
      format: 'markdown',
    };
    const res = await requestJson('/api/commands', { method: 'POST', body: { type: 'File.Compose', payload: body } });
    if (res?.ok && res?.file) {
      return { ok: true, file: res.file, content: typeof res.content === 'string' ? res.content : '' };
    }
    return { ok: false, error: new Error(res?.error || 'CREATE_TEXT_FAILED') };
  } catch (error) {
    console.error('USER_FILES_CREATE_TEXT_FAILED', error);
    return { ok: false, error };
  }
}

export async function createUserHtmlFile({ name, content } = {}) {
  try {
    const body = {
      name,
      content,
      kind: 'text',
      format: 'text',
      mimeType: 'text/html',
    };
    const res = await requestJson('/api/commands', { method: 'POST', body: { type: 'File.Compose', payload: body } });
    if (res?.ok && res?.file) {
      return { ok: true, file: res.file, content: typeof res.content === 'string' ? res.content : '' };
    }
    return { ok: false, error: new Error(res?.error || 'CREATE_HTML_FAILED') };
  } catch (error) {
    console.error('USER_FILES_CREATE_HTML_FAILED', error);
    return { ok: false, error };
  }
}

export async function readUserFileText(id) {
  try {
    const res = await requestJson(`/api/files/${encodeURIComponent(id)}/content`, { method: 'GET' });
    if (res?.ok && typeof res.content === 'string') {
      return { ok: true, file: res.file || null, content: res.content };
    }
    return { ok: false, error: new Error(res?.error || 'READ_TEXT_FAILED') };
  } catch (error) {
    console.error('USER_FILES_READ_TEXT_FAILED', error);
    return { ok: false, error };
  }
}

export async function writeUserFileText(id, content, checksum) {
  try {
    const body = { content, kind: 'text', format: 'markdown' };
    if (typeof checksum === 'string' && checksum) {
      body.checksum = checksum;
    }
    const res = await requestJson('/api/commands', { method: 'POST', body: { type: 'File.UpdateContent', payload: { fileId: id, ...body } } });
    if (res?.ok && res?.file) {
      return { ok: true, file: res.file, content: typeof res.content === 'string' ? res.content : (typeof content === 'string' ? content : '') };
    }
    return { ok: false, error: new Error(res?.error || 'WRITE_TEXT_FAILED') };
  } catch (error) {
    console.error('USER_FILES_WRITE_TEXT_FAILED', error);
    return { ok: false, error };
  }
}

export async function writeUserFileHtml(id, content, checksum) {
  try {
    const body = { content, kind: 'text', format: 'text', mimeType: 'text/html' };
    if (typeof checksum === 'string' && checksum) {
      body.checksum = checksum;
    }
    const res = await requestJson('/api/commands', { method: 'POST', body: { type: 'File.UpdateContent', payload: { fileId: id, ...body } } });
    if (res?.ok && res?.file) {
      return { ok: true, file: res.file, content: typeof res.content === 'string' ? res.content : (typeof content === 'string' ? content : '') };
    }
    return { ok: false, error: new Error(res?.error || 'WRITE_HTML_FAILED') };
  } catch (error) {
    console.error('USER_FILES_WRITE_HTML_FAILED', error);
    return { ok: false, error };
  }
}

function normalizeFilesArray(files) {
  if (!files) return [];
  if (files instanceof FileList) {
    return Array.from(files);
  }
  if (Array.isArray(files)) {
    return files.filter((file) => file instanceof File);
  }
  if (files instanceof File) {
    return [files];
  }
  return [];
}
