import { fetchSession } from './session.js';
import { debugLog } from './debug.js';

let csrfProvider = null;

export function setCsrfProvider(fn) {
  csrfProvider = typeof fn === 'function' ? fn : null;
}

function needsCsrf(method = 'GET') {
  const m = String(method || 'GET').toUpperCase();
  return m !== 'GET' && m !== 'HEAD' && m !== 'OPTIONS';
}

async function getTokenOrRefresh() {
  try {
    if (csrfProvider) {
      const t = await csrfProvider();
      if (t) return t;
    }
  } catch (_) { /* ignore */ }
  const session = await fetchSession();
  return session?.csrf ?? '';
}

export async function requestJson(url, options = {}) {
  const method = options.method || 'GET';
  const body = options.body;
  const headers = { ...(options.headers || {}), 'Accept': 'application/json' };
  const isForm = typeof FormData !== 'undefined' && body instanceof FormData;
  const isBlob = typeof Blob !== 'undefined' && body instanceof Blob;
  const isArrayBuffer = typeof ArrayBuffer !== 'undefined' && body instanceof ArrayBuffer;
  const isArrayBufferView = typeof ArrayBuffer !== 'undefined' && typeof ArrayBuffer.isView === 'function' ? ArrayBuffer.isView(body) : false;
  const isUrlSearchParams = typeof URLSearchParams !== 'undefined' && body instanceof URLSearchParams;
  const isBinaryBody = isBlob || isArrayBuffer || isArrayBufferView;
  const shouldStringify = body !== undefined
    && !isForm
    && !isBinaryBody
    && !isUrlSearchParams
    && typeof body !== 'string';
  let token = null;
  if (needsCsrf(method)) {
    token = await getTokenOrRefresh();
    const hasHeaderCsrf = typeof headers['X-CSRF-Token'] === 'string' || typeof headers['x-csrf-token'] === 'string';
    if (!hasHeaderCsrf) {
      headers['X-CSRF-Token'] = token;
    }
    if (shouldStringify && !headers['Content-Type']) {
      headers['Content-Type'] = 'application/json';
    }
    try { debugLog('HTTP send', { url, method, hasCsrf: hasHeaderCsrf ? true : !!token, csrfLen: (hasHeaderCsrf ? String(headers['X-CSRF-Token'] ?? headers['x-csrf-token']).length : (token || '').length) }); } catch (_) {}
  }

  const init = {
    method,
    credentials: 'include',
    headers,
  };
  if (body !== undefined) {
    if (shouldStringify) {
      init.body = JSON.stringify(body);
    } else {
      init.body = body;
    }
  }

  // First request
  let res = await fetch(url, init);
  let { payload, raw } = await parseJsonStrict(url, method, res);
  if (res.ok) return payload;

  try { debugLog('HTTP error', { url, method, status: res.status, body: payload, rawExcerpt: typeof raw === 'string' ? raw.slice(0, 200) : undefined }); } catch (_) {}

  // Retry on CSRF 403 with fresh token once
  if (res.status === 403 && needsCsrf(method)) {
    try {
      const session = await fetchSession();
      const fresh = session?.csrf ?? null;
      if (fresh) {
        debugLog('CSRF retry', { url, method });
        const retryHeaders = { ...headers, 'X-CSRF-Token': fresh };
        const retryRes = await fetch(url, { ...init, headers: retryHeaders });
        const retryParsed = await parseJsonStrict(url, method, retryRes);
        if (retryRes.ok) return retryParsed.payload;
        try { debugLog('CSRF retry failed', { url, method, status: retryRes.status, body: retryParsed.payload, rawExcerpt: (retryParsed.raw || '').slice(0, 200) }); } catch (_) {}
        const retryErr = new Error(retryParsed.payload?.message || retryParsed.payload?.error || 'REQUEST_FAILED');
        retryErr.payload = retryParsed.payload;
        retryErr.status = retryRes.status;
        throw retryErr;
      }
    } catch (e) {
      try { debugLog('CSRF retry failed', e); } catch (_) {}
    }
  }

  const err = new Error(payload?.message || payload?.error || 'REQUEST_FAILED');
  err.payload = payload;
  err.status = res.status;
  throw err;
}

async function parseJsonStrict(url, method, res) {
  const ct = (res.headers.get('Content-Type') || res.headers.get('content-type') || '').toLowerCase();
  const raw = await res.text();
  if (!ct.includes('application/json')) {
    const excerpt = raw.slice(0, 200);
    try { debugLog('COMMAND_RESPONSE_INVALID', { url, method, status: res.status, contentType: ct || '(none)', rawExcerpt: excerpt }); } catch (_) {}
    const err = new Error('COMMAND_RESPONSE_INVALID');
    err.status = res.status;
    err.raw = raw;
    throw err;
  }
  let payload = null;
  try {
    payload = raw === '' ? null : JSON.parse(raw);
  } catch (e) {
    try { debugLog('COMMAND_RESPONSE_INVALID_PARSE', e); } catch (_) {}
    const err = new Error('COMMAND_RESPONSE_INVALID');
    err.status = res.status;
    err.raw = raw;
    throw err;
  }
  return { payload, raw };
}
