import { openFileLibrary } from '../../ui/file-library.js';

function ensureStructure(root) {
  root.innerHTML = '';
  const form = document.createElement('form');
  form.className = 'board-config__form';
  form.setAttribute('data-action', 'board-config-save-general');

  const field = document.createElement('label');
  field.className = 'board-config__field';

  const title = document.createElement('span');
  title.textContent = 'Nom du board';

  const input = document.createElement('input');
  input.type = 'text';
  input.setAttribute('data-ref', 'board-title');
  input.setAttribute('data-config-autofocus', '');

  field.append(title, input);

  // Thumbnail block
  const thumbField = document.createElement('div');
  thumbField.className = 'board-config__field';
  const thumbLabel = document.createElement('span');
  thumbLabel.textContent = 'Vignette';
  const thumbContainer = document.createElement('div');
  thumbContainer.setAttribute('data-config-thumb', '');
  thumbContainer.style.display = 'flex';
  thumbContainer.style.alignItems = 'center';
  thumbContainer.style.gap = '12px';
  const thumbPreview = document.createElement('div');
  thumbPreview.setAttribute('data-config-thumb-preview', '');
  thumbPreview.style.minWidth = '120px';
  thumbPreview.style.minHeight = '72px';
  thumbPreview.style.borderRadius = '10px';
  thumbPreview.style.overflow = 'hidden';
  thumbPreview.style.background = 'rgba(0,0,0,0.04)';
  thumbPreview.style.display = 'flex';
  thumbPreview.style.alignItems = 'center';
  thumbPreview.style.justifyContent = 'center';
  const thumbActions = document.createElement('div');
  thumbActions.setAttribute('data-config-thumb-actions', '');
  const chooseBtn = document.createElement('button');
  chooseBtn.type = 'button';
  chooseBtn.className = 'btn ghost';
  chooseBtn.textContent = 'Choisir une image…';
  chooseBtn.setAttribute('data-action', 'choose-thumbnail');
  const clearBtn = document.createElement('button');
  clearBtn.type = 'button';
  clearBtn.className = 'btn ghost';
  clearBtn.textContent = 'Effacer';
  clearBtn.setAttribute('data-action', 'clear-thumbnail');
  thumbActions.append(chooseBtn, clearBtn);
  thumbContainer.append(thumbPreview, thumbActions);
  thumbField.append(thumbLabel, thumbContainer);

  const meta = document.createElement('dl');
  meta.className = 'board-config__meta';
  meta.setAttribute('data-config-meta', '');

  const footer = document.createElement('div');
  footer.className = 'board-config__footer';
  footer.setAttribute('data-config-footer', '');

  const closeBtn = document.createElement('button');
  closeBtn.type = 'button';
  closeBtn.className = 'btn ghost';
  closeBtn.setAttribute('data-action', 'close-board-config');
  closeBtn.textContent = 'Fermer';

  const submitBtn = document.createElement('button');
  submitBtn.type = 'submit';
  submitBtn.className = 'btn primary';
  submitBtn.textContent = 'Enregistrer';

  footer.append(closeBtn, submitBtn);

  form.append(field, thumbField, meta, footer);
  root.append(form);
}

function updateMeta(metaNode, { boardId, updatedAt, createdAt }) {
  if (!metaNode) return;
  metaNode.innerHTML = '';
  const entries = [
    { label: 'Identifiant', value: boardId },
    { label: 'Dernière sauvegarde', value: updatedAt },
    { label: 'Créé le', value: createdAt },
  ];
  entries.forEach(({ label, value }) => {
    if (!value) return;
    const wrapper = document.createElement('div');
    const dt = document.createElement('dt');
    dt.textContent = label;
    const dd = document.createElement('dd');
    dd.textContent = value;
    wrapper.append(dt, dd);
    metaNode.append(wrapper);
  });
}

function updateError(form, errorMessage) {
  const footer = form.querySelector('[data-config-footer]');
  if (!footer) return;
  let errorNode = form.querySelector('[data-config-error]');
  if (errorMessage) {
    if (!errorNode) {
      errorNode = document.createElement('p');
      errorNode.className = 'board-config__error';
      errorNode.setAttribute('data-config-error', '');
      footer.before(errorNode);
    }
    errorNode.textContent = errorMessage;
  } else if (errorNode) {
    errorNode.remove();
  }
}

export function renderGeneralBoardPanel(root, context) {
  if (!root) return;
  if (!root.firstElementChild) {
    ensureStructure(root);
  }

  const form = root.querySelector('form[data-action="board-config-save-general"]');
  if (!form) return;

  const ctx = context ?? {};

  const input = form.querySelector('[data-ref="board-title"]');
  if (input) {
    if (document.activeElement !== input) {
      input.value = ctx.boardTitle ?? '';
    }
    input.disabled = !!ctx.busy;
  }

  // Initialize thumb dataset
  if (typeof ctx.thumbnailPublicId === 'string') {
    form.dataset.thumbnailInitial = ctx.thumbnailPublicId || '';
  } else if (!('thumbnailInitial' in form.dataset)) {
    form.dataset.thumbnailInitial = '';
  }
  if (!('thumbnailCurrent' in form.dataset)) {
    form.dataset.thumbnailCurrent = form.dataset.thumbnailInitial || '';
  }

  const thumbPreview = form.querySelector('[data-config-thumb-preview]');
  if (thumbPreview) {
    const publicId = form.dataset.thumbnailCurrent || form.dataset.thumbnailInitial || '';
    updateThumbPreview(thumbPreview, publicId);
  }

  const chooseBtn = form.querySelector('button[data-action="choose-thumbnail"]');
  const clearBtn = form.querySelector('button[data-action="clear-thumbnail"]');
  if (chooseBtn) chooseBtn.disabled = !!ctx.busy;
  if (clearBtn) clearBtn.disabled = !!ctx.busy;

  const submitBtn = form.querySelector('button[type="submit"]');
  if (submitBtn) {
    submitBtn.disabled = !!ctx.busy;
  }

  updateMeta(form.querySelector('[data-config-meta]'), {
    boardId: ctx.boardId ?? '',
    updatedAt: ctx.updatedAt ?? '',
    createdAt: ctx.createdAt ?? '',
  });

  updateError(form, ctx.error ?? '');

  // Attach once: handlers for choose/clear thumbnail
  if (!form.__thumbHandlersInstalled) {
    form.addEventListener('click', async (event) => {
      const el = event.target instanceof Element ? event.target.closest('[data-action]') : null;
      if (!el) return;
      const action = el.getAttribute('data-action');
      if (action === 'choose-thumbnail') {
        event.preventDefault();
        try {
          const result = await openFileLibrary({ title: 'Bibliothèque', accept: 'image/*', showSelect: true, allowDelete: true, filter: (f) => typeof f?.mimeType === 'string' && f.mimeType.startsWith('image/') });
          const file = result?.file || null;
          if (file && typeof file.id === 'string') {
            form.dataset.thumbnailCurrent = file.id;
            const preview = form.querySelector('[data-config-thumb-preview]');
            if (preview) updateThumbPreview(preview, file.id);
          }
        } catch (_) {}
      }
      if (action === 'clear-thumbnail') {
        event.preventDefault();
        form.dataset.thumbnailCurrent = '';
        const preview = form.querySelector('[data-config-thumb-preview]');
        if (preview) updateThumbPreview(preview, '');
      }
    });
    form.__thumbHandlersInstalled = true;
  }
}

function updateThumbPreview(container, publicId) {
  if (!(container instanceof HTMLElement)) return;
  container.innerHTML = '';
  const pid = typeof publicId === 'string' ? publicId : '';
  if (!pid) {
    const empty = document.createElement('div');
    empty.textContent = 'Aucune image';
    empty.style.fontSize = '12px';
    empty.style.opacity = '.7';
    container.appendChild(empty);
    return;
  }
  const img = document.createElement('img');
  img.src = `/api/files/${encodeURIComponent(pid)}/content`;
  img.alt = '';
  img.decoding = 'async';
  img.style.display = 'block';
  img.style.width = '120px';
  img.style.height = '72px';
  img.style.objectFit = 'cover';
  container.appendChild(img);
window.toastui && window.toastui.Editor }
