function ensureStructure(root) {
  root.innerHTML = '';

  const wrapper = document.createElement('div');
  wrapper.className = 'board-config__logic';

  const intro = document.createElement('div');
  intro.className = 'board-config__placeholder';

  const title = document.createElement('strong');
  title.textContent = 'Automatisations du board';

  const description = document.createElement('p');
  description.textContent = "Configurez les règles et automatismes propres au board dès que le moteur de logique est activé. Vous pouvez déjà importer ou exporter un bundle complet via les boutons ci-dessous.";

  intro.append(title, description);

  const summary = document.createElement('dl');
  summary.className = 'board-config__meta';
  summary.setAttribute('data-logic-summary', '');

  const rulesContainer = document.createElement('div');
  rulesContainer.className = 'board-config__rules';
  rulesContainer.setAttribute('data-logic-rules', '');

  const footer = document.createElement('div');
  footer.className = 'board-config__footer';
  footer.setAttribute('data-logic-footer', '');

  const importButton = document.createElement('button');
  importButton.type = 'button';
  importButton.className = 'btn ghost';
  importButton.setAttribute('data-action', 'import-board');
  importButton.textContent = 'Importer un bundle JSON';

  const exportButton = document.createElement('button');
  exportButton.type = 'button';
  exportButton.className = 'btn primary';
  exportButton.setAttribute('data-action', 'export-board');
  exportButton.textContent = 'Exporter ce board';

  footer.append(importButton, exportButton);

  wrapper.append(intro, summary, rulesContainer, footer);
  root.append(wrapper);
}

function renderSummary(summaryNode, context) {
  if (!summaryNode) return;
  summaryNode.innerHTML = '';

  const entries = [
    { label: 'Règles actives', value: context.ruleCount > 0 ? String(context.ruleCount) : 'Aucune' },
    { label: 'Dernière modification', value: context.updatedAt || '–' },
    { label: 'Version moteur', value: context.rulesVersion || 'Non initialisée' },
  ];

  entries.forEach(({ label, value }) => {
    const container = document.createElement('div');
    const dt = document.createElement('dt');
    dt.textContent = label;
    const dd = document.createElement('dd');
    dd.textContent = value;
    container.append(dt, dd);
    summaryNode.append(container);
  });
}

export function renderLogicBoardPanel(root, context) {
  if (!root) return;
  if (!root.firstElementChild) {
    ensureStructure(root);
  }

  const summaryNode = root.querySelector('[data-logic-summary]');
  const footer = root.querySelector('[data-logic-footer]');
  const rulesRoot = root.querySelector('[data-logic-rules]');

  const ctx = {
    ruleCount: typeof context?.ruleCount === 'number' ? context.ruleCount : 0,
    updatedAt: context?.updatedAt ?? '',
    rulesVersion: context?.rulesVersion ?? '',
    busy: !!context?.busy,
    packs: Array.isArray(context?.packs) ? context.packs : [],
  };

  renderSummary(summaryNode, ctx);
  renderRules(rulesRoot, ctx);

  if (footer) {
    footer.querySelectorAll('button').forEach((button) => {
      button.disabled = ctx.busy;
    });
  }
}

function renderRules(root, context) {
  if (!root) return;
  root.innerHTML = '';

  const packs = context.packs;
  if (!packs.length) {
    const empty = document.createElement('p');
    empty.className = 'board-config__rules-empty';
    empty.textContent = 'Aucune règle déposée par les packs.';
    root.append(empty);
    return;
  }

  packs.forEach((pack) => {
    const details = document.createElement('details');
    details.className = 'board-config__rules-pack';
    details.open = packs.length === 1;

    const summary = document.createElement('summary');
    summary.textContent = formatPackLabel(pack.id, pack.name);
    details.append(summary);

    const list = document.createElement('ul');
    list.className = 'board-config__rules-list';

    pack.rules.forEach((rule) => {
      const item = document.createElement('li');
      item.className = 'board-config__rules-item';

      const label = document.createElement('label');
      label.className = 'board-config__rules-entry';

      const checkbox = document.createElement('input');
      checkbox.type = 'checkbox';
      checkbox.checked = rule.enabled;
      checkbox.disabled = true;
      checkbox.dataset.ruleId = rule.id;
      checkbox.setAttribute('aria-label', ruleLabel(rule));

      const span = document.createElement('span');
      span.className = 'board-config__rules-label';
      span.textContent = ruleLabel(rule);

      const meta = document.createElement('span');
      meta.className = 'board-config__rules-meta';
      meta.textContent = rule.description ?? '';
      meta.hidden = !rule.description;

      label.append(checkbox, span);
      item.append(label);
      if (rule.description) {
        item.append(meta);
      }
      list.append(item);
    });

    details.append(list);
    root.append(details);
  });
}

function formatPackLabel(id, name) {
  if (typeof name === 'string' && name.trim()) {
    return name.trim();
  }
  return `Pack ${id}`;
}

function ruleLabel(rule) {
  if (typeof rule.id === 'string' && rule.id.trim()) {
    return rule.id.trim();
  }
  return 'Règle sans identifiant';
}
