export function createDragManager(root, options = {}) {
  const state = {
    pointerId: null,
    sourceEl: null,
    dragData: null,
    ghost: null,
    overEl: null,
    startX: 0,
    startY: 0,
    lastClientX: null,
    lastClientY: null,
    active: false,
    longPressTimer: null,
  };

  const getData = options.getData ?? (el => ({ id: el.dataset.id, type: el.dataset.type }));
  const onDrop = options.onDrop ?? (() => {});
  const onPreview = options.onPreview ?? (() => {});
  const getScrollContainers = options.getScrollContainers ?? (() => []);
  const autoScrollMargin = options.autoScrollMargin ?? 56;
  const autoScrollSpeed = options.autoScrollSpeed ?? 14;
  const ignoredSelectors = options.ignoredSelectors ?? 'button, a, input, textarea, select, [contenteditable="true"], [data-dnd-ignore="true"]';
  const activationThreshold = Number.isFinite(options.activationThreshold) ? options.activationThreshold : 6;
  const longPressDelay = Number.isFinite(options.longPressDelay) ? options.longPressDelay : 250;

  const autoScroll = createAutoScroll();

  function pointerDown(event) {
    const target = event.target.closest('[data-draggable="true"]');
    if (!target) return;
    if (event.pointerType === 'mouse' && event.button !== 0) return;
    if (event.target.closest(ignoredSelectors)) {
      return;
    }
    state.pointerId = event.pointerId;
    state.sourceEl = target;
    state.dragData = getData(target);
    state.startX = event.clientX;
    state.startY = event.clientY;
    state.active = false;
    state.longPressTimer = window.setTimeout(() => {
      if (!state.sourceEl || state.active) return;
      activateDrag(null, 'long-press');
      createGhost();
      positionGhost(state.startX, state.startY);
      updateHoverPosition(state.startX, state.startY);
    }, longPressDelay);
  }

  function pointerMove(event) {
    if (state.pointerId !== event.pointerId || !state.sourceEl) return;
    if (!state.active) {
      const dx = Math.abs(event.clientX - state.startX);
      const dy = Math.abs(event.clientY - state.startY);
      if (Math.max(dx, dy) < activationThreshold) {
        return;
      }
      cancelLongPress();
      activateDrag(event, 'pointer-move');
      createGhost();
    }
    event.preventDefault();
    state.lastClientX = event.clientX;
    state.lastClientY = event.clientY;
    if (!state.ghost) {
      createGhost();
    }
    positionGhost(event.clientX, event.clientY);
    updateHoverPosition(event.clientX, event.clientY);
    scheduleAutoScroll(event);
  }

  function pointerUp(event) {
    if (state.pointerId !== event.pointerId) return;
    cancelLongPress();
    releasePointerCapture();
    if (state.ghost) {
      cleanupGhost();
    }
    if (state.active && state.sourceEl && state.overEl) {
      onDrop({ data: state.dragData, dropzone: state.overEl });
    }
    if (state.active && state.dragData) {
      onPreview({ type: 'end', data: state.dragData });
    }
    autoScroll.stop();
    reset();
  }

  function pointerCancel(event) {
    if (state.pointerId !== event.pointerId) return;
    cancelLongPress();
    releasePointerCapture();
    cleanupGhost();
    if (state.active && state.dragData) {
      onPreview({ type: 'end', data: state.dragData });
    }
    autoScroll.stop();
    reset();
  }

  function updateHoverPosition(clientX, clientY, cause = 'pointer') {
    if (!state.dragData) {
      return;
    }
    const over = document.elementFromPoint(clientX, clientY);
    const nextDropzone = over?.closest('[data-dropzone="true"]') ?? null;

    if (nextDropzone !== state.overEl) {
      if (state.overEl) {
        onPreview({ type: 'leave', data: state.dragData, dropzone: state.overEl });
      }
      state.overEl = nextDropzone;
      if (nextDropzone) {
        onPreview({
          type: 'enter',
          data: state.dragData,
          dropzone: nextDropzone,
          clientX,
          clientY,
          cause,
        });
      }
    }

    if (!state.overEl) {
      return;
    }

    onPreview({
      type: 'update',
      data: state.dragData,
      dropzone: state.overEl,
      clientX,
      clientY,
      cause,
    });
  }

  function scheduleAutoScroll(event) {
    const containers = getScrollContainers(event, {
      dropzone: state.overEl,
      source: state.sourceEl,
    }) || [];
    autoScroll.update({
      clientX: event.clientX,
      clientY: event.clientY,
      margin: autoScrollMargin,
      speed: autoScrollSpeed,
      containers,
      onScroll: () => {
        if (state.lastClientX !== null && state.lastClientY !== null) {
          updateHoverPosition(state.lastClientX, state.lastClientY, 'auto-scroll');
        }
      },
    });
  }

  function createGhost() {
    const rect = state.sourceEl.getBoundingClientRect();
    const ghost = state.sourceEl.cloneNode(true);
    ghost.style.position = 'fixed';
    ghost.style.pointerEvents = 'none';
    ghost.style.width = `${rect.width}px`;
    ghost.style.opacity = '0.85';
    ghost.style.zIndex = '70';
    ghost.classList.add('drag-ghost');
    document.body.appendChild(ghost);
    state.ghost = ghost;
  }

  function positionGhost(x, y) {
    if (!state.ghost) return;
    state.ghost.style.transform = `translate(${x + 6}px, ${y + 6}px)`;
  }

  function cleanupGhost() {
    if (state.ghost) {
      state.ghost.remove();
      state.ghost = null;
    }
  }

  function reset() {
    if (state.sourceEl) {
      state.sourceEl.classList.remove('is-drag-source');
    }
    document.body.classList.remove('is-dragging');
    state.pointerId = null;
    state.sourceEl = null;
    state.dragData = null;
    state.overEl = null;
    state.startX = 0;
    state.startY = 0;
    state.lastClientX = null;
    state.lastClientY = null;
    state.active = false;
    cancelLongPress();
  }

  function activateDrag(event, cause) {
    if (!state.sourceEl || state.active) return;
    state.active = true;
    try {
      root.setPointerCapture(state.pointerId);
    } catch (_) { /* noop */ }
    state.sourceEl.classList.add('is-drag-source');
    document.body.classList.add('is-dragging');
    if (event?.preventDefault) {
      event.preventDefault();
    }
    onPreview({ type: 'start', data: state.dragData, cause });
  }

  function cancelLongPress() {
    if (state.longPressTimer !== null) {
      clearTimeout(state.longPressTimer);
      state.longPressTimer = null;
    }
  }

  function releasePointerCapture() {
    try {
      if (state.pointerId !== null) {
        root.releasePointerCapture(state.pointerId);
      }
    } catch (_) {
      /* noop */
    }
  }

  root.addEventListener('pointerdown', pointerDown);
  root.addEventListener('pointermove', pointerMove);
  root.addEventListener('pointerup', pointerUp);
  root.addEventListener('pointercancel', pointerCancel);

  return {
    destroy() {
      root.removeEventListener('pointerdown', pointerDown);
      root.removeEventListener('pointermove', pointerMove);
      root.removeEventListener('pointerup', pointerUp);
      root.removeEventListener('pointercancel', pointerCancel);
      cleanupGhost();
      autoScroll.stop();
      reset();
    }
  };
}

function createAutoScroll() {
  const active = new Map();
  let frame = null;
  let onScrollCallback = null;
  let lastPayload = null;

  function step() {
    frame = null;
    let scrolled = false;
    active.forEach((vector, container) => {
      const el = normalizeScrollContainer(container);
      if (!el) {
        return;
      }
      const { vx, vy } = vector;
      if (vx) {
        el.scrollLeft += vx;
      }
      if (vy) {
        el.scrollTop += vy;
      }
      if (vx || vy) {
        scrolled = true;
      }
    });

    if (scrolled && typeof onScrollCallback === 'function') {
      onScrollCallback();
    }

    if (lastPayload) {
      const vectors = computeVectors(lastPayload);
      active.clear();
      vectors.forEach((value, key) => active.set(key, value));
    } else {
      active.clear();
    }

    if (active.size) {
      frame = requestAnimationFrame(step);
    }
  }

  function update(payload) {
    lastPayload = {
      clientX: payload.clientX,
      clientY: payload.clientY,
      margin: payload.margin,
      speed: payload.speed,
      containers: payload.containers ?? [],
      onScroll: payload.onScroll,
    };
    onScrollCallback = payload.onScroll;
    const vectors = computeVectors(lastPayload);
    active.clear();
    vectors.forEach((value, key) => active.set(key, value));

    if (active.size && frame === null) {
      frame = requestAnimationFrame(step);
    }

    if (!active.size && frame !== null) {
      cancelAnimationFrame(frame);
      frame = null;
    }
  }

  function stop() {
    active.clear();
    lastPayload = null;
    if (frame !== null) {
      cancelAnimationFrame(frame);
      frame = null;
    }
  }

  return { update, stop };
}

function computeVectors(payload) {
  const vectors = new Map();
  const { clientX, clientY, margin, speed } = payload;
  const containers = Array.isArray(payload.containers) ? payload.containers : [];

  containers.forEach((entry) => {
    const el = normalizeScrollContainer(entry);
    const isDocumentScroller = el === (document.scrollingElement ?? document.documentElement);
    if (!el || (!isDocumentScroller && !el.isConnected)) {
      return;
    }
    if (el.scrollHeight <= el.clientHeight && el.scrollWidth <= el.clientWidth) {
      return;
    }
    const rect = el === document.scrollingElement
      ? {
          top: 0,
          left: 0,
          right: window.innerWidth,
          bottom: window.innerHeight,
        }
      : el.getBoundingClientRect();

    let vx = 0;
    let vy = 0;

    if (el.scrollWidth > el.clientWidth) {
      if (clientX - rect.left < margin) {
        const intensity = 1 - Math.max(0, (clientX - rect.left) / margin);
        vx = -Math.ceil(speed * intensity);
      } else if (rect.right - clientX < margin) {
        const intensity = 1 - Math.max(0, (rect.right - clientX) / margin);
        vx = Math.ceil(speed * intensity);
      }
    }

    if (el.scrollHeight > el.clientHeight) {
      if (clientY - rect.top < margin) {
        const intensity = 1 - Math.max(0, (clientY - rect.top) / margin);
        vy = -Math.ceil(speed * intensity);
      } else if (rect.bottom - clientY < margin) {
        const intensity = 1 - Math.max(0, (rect.bottom - clientY) / margin);
        vy = Math.ceil(speed * intensity);
      }
    }

    if (vx || vy) {
      vectors.set(el, { vx, vy });
    }
  });

  return vectors;
}

function normalizeScrollContainer(container) {
  if (!container) return null;
  if (container === window || container === document || container === document.body) {
    return document.scrollingElement ?? document.documentElement;
  }
  if (container instanceof Element) {
    return container;
  }
  return null;
}
