import {
  ensureUserFiles,
  refreshUserFiles,
  ensureUserFolders,
  refreshUserFolders,
  createFolder as storeCreateFolder,
  moveFile as storeMoveFile,
  uploadFiles,
  deleteUserFile,
  renameFolder,
  deleteFolder,
} from '../files/store.js';
import { showToast } from './toast.js';
import './modal.js';
import { confirmDialog, promptDialog } from './dialog.js';
import { requestJson } from '../services/http.js';
import { escapeHtml } from '../utils/format.js';

const DEFAULT_OPTIONS = {
  title: 'Bibliothèque de fichiers',
  description: '',
  accept: '',
  filter: null,
  allowUpload: true,
  allowDelete: true,
  showSelect: true,
  selectLabel: 'Sélectionner',
  emptyLabel: 'Aucun fichier disponible pour le moment.',
  autoSelectUploaded: false,
};

let modal = null;
let refs = null;
let currentRequest = null;

export async function openFileLibrary(options = {}) {
  const finalOptions = { ...DEFAULT_OPTIONS, ...options };
  ensureModal();

  if (currentRequest) {
    const previousResolve = currentRequest.resolve;
    currentRequest = null;
    try { previousResolve(null); } catch (_) { /* noop */ }
  }

  return new Promise((resolve) => {
    currentRequest = {
      resolve,
      options: finalOptions,
      state: {
        files: [],
        folders: [],
        currentFolderId: '',
        loading: false,
        uploading: false,
        deleting: new Set(),
      },
    };

    // Bring modal to front in DOM to ensure top stacking when multiple sb-modal coexist
    try { if (modal.parentNode === document.body) { document.body.appendChild(modal); } } catch (_) {}
    prepareModal(finalOptions);
    modal.show();
    // Always refresh on open to avoid stale lists
    loadFolders(true).then(() => loadFiles(true));
  });
}

function ensureModal() {
  if (modal) {
    return;
  }
  modal = document.createElement('sb-modal');
  modal.className = 'file-library-modal';
  try { modal.style.zIndex = '2000'; } catch (_) {}
  modal.innerHTML = `
    <div class="file-library" role="document">
      <header class="file-library__header">
        <div class="file-library__heading">
          <h2 class="file-library__title"></h2>
          <p class="file-library__subtitle"></p>
        </div>
        <button type="button" class="btn ghost btn--icon file-library__close" title="Fermer" aria-label="Fermer">✕</button>
      </header>
      <div class="file-library__toolbar">
        <label class="btn ghost file-library__upload">
          📤 Importer un fichier
          <input type="file" hidden />
        </label>
        <button type="button" class="btn ghost file-library__new-folder">📁 Nouveau dossier</button>
        <button type="button" class="btn ghost file-library__refresh">🔄 Actualiser</button>
        <div class="file-library__folders">
          <label for="file-library-folder-select" class="sr-only">Filtrer par dossier</label>
          <select id="file-library-folder-select" class="file-library__folder-select"></select>
        </div>
      </div>
      <div class="file-library__nav">
        <nav class="file-library__breadcrumb" aria-label="Fil d'Ariane"></nav>
      </div>
      <div class="file-library__status" aria-live="polite"></div>
      <div class="file-library__body">
        <div class="file-library__folders-list" role="list" aria-label="Dossiers"></div>
        <div class="file-library__empty">Aucun fichier disponible pour le moment.</div>
        <div class="file-library__list" role="list"></div>
      </div>
    </div>
  `;
  document.body.appendChild(modal);

  refs = {
    title: modal.querySelector('.file-library__title'),
    subtitle: modal.querySelector('.file-library__subtitle'),
    close: modal.querySelector('.file-library__close'),
    uploadLabel: modal.querySelector('.file-library__upload'),
    uploadInput: modal.querySelector('.file-library__upload input'),
    refresh: modal.querySelector('.file-library__refresh'),
    folderSelect: modal.querySelector('.file-library__folder-select'),
    newFolder: modal.querySelector('.file-library__new-folder'),
    breadcrumb: modal.querySelector('.file-library__breadcrumb'),
    status: modal.querySelector('.file-library__status'),
    foldersList: modal.querySelector('.file-library__folders-list'),
    empty: modal.querySelector('.file-library__empty'),
    list: modal.querySelector('.file-library__list'),
  };

  if (refs.close) {
    refs.close.addEventListener('click', () => modal.close());
  }
  modal.addEventListener('sb-modal-close', handleModalClose);

  if (refs.refresh) {
    refs.refresh.addEventListener('click', () => loadFiles(true));
  }

  if (refs.uploadInput) {
    refs.uploadInput.addEventListener('change', handleUpload);
  }
  if (refs.folderSelect) {
    refs.folderSelect.addEventListener('change', handleFolderFilter);
  }
  if (refs.newFolder) {
    refs.newFolder.addEventListener('click', handleCreateFolder);
  }
}

function prepareModal(options) {
  if (!refs) return;
  refs.title.textContent = options.title || DEFAULT_OPTIONS.title;
  if (options.description) {
    refs.subtitle.textContent = options.description;
    refs.subtitle.style.display = '';
  } else {
    refs.subtitle.textContent = '';
    refs.subtitle.style.display = 'none';
  }
  if (refs.uploadLabel) {
    refs.uploadLabel.style.display = options.allowUpload ? '' : 'none';
  }
  if (refs.uploadInput) {
    refs.uploadInput.value = '';
    if (options.accept) {
      refs.uploadInput.setAttribute('accept', options.accept);
    } else {
      refs.uploadInput.removeAttribute('accept');
    }
  }
  if (refs.refresh) {
    refs.refresh.disabled = false;
  }
  if (refs.empty) {
    refs.empty.textContent = options.emptyLabel || DEFAULT_OPTIONS.emptyLabel;
  }
  setStatus('');
  updateToolbar();
  renderList();
}

async function loadFolders(forceRefresh) {
  if (!currentRequest) return;
  const { state } = currentRequest;
  if (state.loading) return;
  state.loading = true;
  setStatus(forceRefresh ? 'Actualisation…' : 'Chargement…');
  updateToolbar();
  try {
    const folders = forceRefresh ? await refreshUserFolders() : await ensureUserFolders();
    if (Array.isArray(folders)) {
      state.folders = folders;
    } else if (!Array.isArray(state.folders)) {
      state.folders = [];
    }
    renderFolderSelect();
    renderBreadcrumb();
    renderFoldersSection();
  } catch (error) {
    console.error('USER_FOLDERS_LIBRARY_LOAD_FAILED', error);
  } finally {
    state.loading = false;
    setStatus('');
    updateToolbar();
  }
}

async function loadFiles(forceRefresh) {
  if (!currentRequest) return;
  const { state } = currentRequest;
  if (state.loading) return;
  state.loading = true;
  setStatus(forceRefresh ? 'Actualisation…' : 'Chargement…');
  updateToolbar();
  try {
    const files = forceRefresh ? await refreshUserFiles() : await ensureUserFiles();
    if (Array.isArray(files)) {
      state.files = files;
    } else if (!Array.isArray(state.files)) {
      state.files = [];
    }
    renderList();
    renderFoldersSection();
  } catch (error) {
    console.error('USER_FILES_LIBRARY_LOAD_FAILED', error);
    showToast('Impossible de charger les fichiers.');
  } finally {
    state.loading = false;
    setStatus('');
    updateToolbar();
  }
}

function handleModalClose() {
  if (!currentRequest) {
    return;
  }
  const resolver = currentRequest.resolve;
  currentRequest = null;
  try { resolver(null); } catch (_) { /* noop */ }
}

async function handleUpload(event) {
  if (!currentRequest || !refs || !event?.target) {
    return;
  }
  const input = event.target;
  const files = input.files;
  input.value = '';
  if (!files || !files.length) {
    return;
  }
  const { state, options } = currentRequest;
  if (!options.allowUpload) {
    return;
  }
  state.uploading = true;
  setStatus('Téléversement en cours…');
  updateToolbar();
  try {
    const folderId = currentRequest?.state?.currentFolderId || '';
    const result = await uploadFiles(files, folderId);
    if (!result.ok) {
      showToast('Échec du téléversement.');
      return;
    }
    const uploaded = Array.isArray(result.files) ? result.files : [];
    if (uploaded.length) {
      state.files = [...uploaded, ...state.files];
      renderList();
      showToast(uploaded.length > 1 ? 'Fichiers importés.' : 'Fichier importé.');
      if (options.autoSelectUploaded) {
        completeSelection(uploaded[0]);
        return;
      }
    }
  } catch (error) {
    console.error('USER_FILES_LIBRARY_UPLOAD_FAILED', error);
    showToast('Impossible de téléverser le fichier.');
  } finally {
    state.uploading = false;
    setStatus('');
    updateToolbar();
  }
}

function renderList() {
  if (!currentRequest || !refs) {
    return;
  }
  const { state, options } = currentRequest;
  const files = Array.isArray(state.files) ? state.files : [];
  const folderId = state.currentFolderId || '';
  const filteredByFolder = folderId
    ? files.filter((f) => (typeof f.folderId === 'string' ? f.folderId : '') === folderId)
    : files.filter((f) => !f.folderId);
  const filtered = typeof options.filter === 'function' ? filteredByFolder.filter((file) => {
    try {
      return !!options.filter(file);
    } catch (_) {
      return false;
    }
  }) : filteredByFolder;

  refs.list.innerHTML = '';
  if (!filtered.length) {
    if (refs.empty) {
      refs.empty.style.display = '';
    }
    return;
  }
  if (refs.empty) {
    refs.empty.style.display = 'none';
  }
  filtered.forEach((file) => {
    const row = document.createElement('div');
    row.className = 'file-library__item';
    row.setAttribute('role', 'listitem');
    const preview = buildFilePreview(file);
    // Ouvrir au clic sur l’aperçu
    preview.style.cursor = 'pointer';
    preview.title = 'Ouvrir';
    preview.addEventListener('click', () => openFile(file));
    const body = document.createElement('div');
    body.className = 'file-library__item-body';
    const name = document.createElement('div');
    name.className = 'file-library__item-name';
    name.textContent = String(file.name || file.id || 'Fichier');
    name.title = 'Ouvrir';
    name.style.cursor = 'pointer';
    name.addEventListener('click', () => openFile(file));
    const meta = document.createElement('div');
    meta.className = 'file-library__item-meta';
    meta.textContent = buildFileMeta(file);
    body.append(name, meta);

    const actions = document.createElement('div');
    actions.className = 'file-library__item-actions';
    // Ouvrir
    const openBtn = document.createElement('button');
    openBtn.type = 'button';
    openBtn.className = 'btn btn--icon ghost';
    openBtn.title = 'Ouvrir';
    openBtn.setAttribute('aria-label', 'Ouvrir');
    openBtn.textContent = '↗';
    openBtn.addEventListener('click', () => openFile(file));
    openBtn.disabled = state.loading || state.uploading || state.deleting.has(file.id);
    actions.appendChild(openBtn);
    if (options.showSelect !== false) {
      const selectBtn = document.createElement('button');
      selectBtn.type = 'button';
      selectBtn.className = 'btn primary';
      selectBtn.textContent = options.selectLabel || DEFAULT_OPTIONS.selectLabel;
      selectBtn.addEventListener('click', () => completeSelection(file));
      selectBtn.disabled = state.loading || state.uploading || state.deleting.has(file.id);
      actions.appendChild(selectBtn);
    }

    if (options.allowDelete) {
      const deleteBtn = document.createElement('button');
      deleteBtn.type = 'button';
      deleteBtn.className = 'btn btn--icon ghost file-library__delete';
      deleteBtn.title = 'Supprimer';
      deleteBtn.setAttribute('aria-label', 'Supprimer');
      deleteBtn.textContent = '🗑️';
      deleteBtn.disabled = state.loading || state.uploading || state.deleting.has(file.id);
      deleteBtn.addEventListener('click', () => handleDelete(file));
      actions.appendChild(deleteBtn);
    }

    const moveBtn = document.createElement('button');
    moveBtn.type = 'button';
    moveBtn.className = 'btn btn--icon ghost';
    moveBtn.title = 'Déplacer';
    moveBtn.setAttribute('aria-label', 'Déplacer');
    moveBtn.textContent = '➡️';
    moveBtn.disabled = state.loading || state.uploading || state.deleting.has(file.id);
    moveBtn.addEventListener('click', () => handleMove(file, moveBtn));
    actions.appendChild(moveBtn);

    row.append(preview, body, actions);
    refs.list.appendChild(row);
  });
}

function buildFilePreview(file) {
  const wrap = document.createElement('div');
  wrap.className = 'file-library__item-preview';
  const mime = typeof file.mimeType === 'string' ? file.mimeType : '';
  if (mime.startsWith('image/') && typeof file.url === 'string' && file.url) {
    const img = document.createElement('img');
    img.src = file.url;
    img.alt = '';
    img.decoding = 'async';
    img.loading = 'lazy';
    wrap.appendChild(img);
  } else {
    const fallback = document.createElement('div');
    fallback.className = 'file-library__item-fallback';
    fallback.textContent = extractFallbackLabel(file);
    wrap.appendChild(fallback);
  }
  return wrap;
}

function renderFolderSelect() {
  if (!refs || !refs.folderSelect || !currentRequest) return;
  const { state } = currentRequest;
  const select = refs.folderSelect;
  select.innerHTML = '';
  const optAll = document.createElement('option');
  optAll.value = '';
  optAll.textContent = 'Tous les fichiers';
  select.appendChild(optAll);
  const folders = Array.isArray(state.folders) ? state.folders : [];
  folders.forEach((f) => {
    const opt = document.createElement('option');
    opt.value = String(f.id || '');
    opt.textContent = String(f.name || 'Dossier');
    select.appendChild(opt);
  });
  select.value = state.currentFolderId || '';
}

function handleFolderFilter(event) {
  if (!currentRequest || !refs) return;
  const select = event?.target;
  if (!(select instanceof HTMLSelectElement)) return;
  currentRequest.state.currentFolderId = select.value || '';
  renderBreadcrumb();
  renderFoldersSection();
  renderList();
}

async function handleCreateFolder() {
  if (!currentRequest) return;
  const name = (await promptDialog({ title: 'Nouveau dossier', label: 'Nom du dossier', defaultValue: '' }))?.trim();
  if (!name) return;
  setStatus('Création du dossier…');
  try {
    const parentId = currentRequest.state.currentFolderId || null;
    const result = await storeCreateFolder(name, parentId);
    if (!result.ok) {
      showToast('Impossible de créer le dossier.');
      return;
    }
    currentRequest.state.folders = [result.folder, ...currentRequest.state.folders];
    currentRequest.state.currentFolderId = String(result.folder.id || '');
    renderFolderSelect();
    renderBreadcrumb();
    renderFoldersSection();
    renderList();
    showToast('Dossier créé.');
  } catch (error) {
    console.error('USER_FOLDER_CREATE_FAILED', error);
    showToast('Impossible de créer le dossier.');
  } finally {
    setStatus('');
  }
}

async function handleMove(file, anchorEl) {
  if (!currentRequest) return;
  const folders = currentRequest.state.folders || [];
  const close = createMovePopover(anchorEl, folders, async (destId) => {
    setStatus('Déplacement…');
    try {
      const res = await storeMoveFile(file.id || file.fileId, destId || '');
      if (!res.ok) {
        showToast('Impossible de déplacer le fichier.');
        return;
      }
      currentRequest.state.files = currentRequest.state.files.map((f) => (f.id === res.file.id ? { ...f, ...res.file } : f));
      renderList();
      showToast('Fichier déplacé.');
    } catch (error) {
      console.error('USER_FILE_MOVE_FAILED', error);
      showToast('Impossible de déplacer le fichier.');
    } finally {
      setStatus('');
      try { close(); } catch (_) {}
    }
  });
}

function extractFallbackLabel(file) {
  const name = String(file.name || file.downloadName || file.id || '?');
  const match = name.trim()[0];
  return match ? match.toUpperCase() : 'F';
}

function buildFileMeta(file) {
  const parts = [];
  if (typeof file.sizeLabel === 'string' && file.sizeLabel) {
    parts.push(file.sizeLabel);
  } else if (Number.isFinite(file.byteSize)) {
    parts.push(formatByteSize(file.byteSize));
  }
  if (typeof file.mimeType === 'string' && file.mimeType) {
    parts.push(file.mimeType);
  }
  if (Number.isFinite(file.updatedAt)) {
    const date = new Date(Number(file.updatedAt) * 1000);
    if (!Number.isNaN(date.getTime())) {
      parts.push(date.toLocaleString());
    }
  }
  return parts.join(' · ');
}

async function handleDelete(file) {
  if (!currentRequest || !refs) {
    return;
  }
  const ok = await confirmDialog({ title: 'Supprimer le fichier', message: 'Supprimer ce fichier de votre bibliothèque ?', okLabel: 'Supprimer', cancelLabel: 'Annuler' });
  if (!ok) {
    return;
  }
  const { state } = currentRequest;
  state.deleting.add(file.id);
  updateToolbar();
  renderList();
  try {
    const res = await deleteUserFile(file.id);
    if (!res.ok) {
      showToast('Impossible de supprimer le fichier.');
      return;
    }
    state.files = state.files.filter((entry) => entry.id !== file.id);
    renderList();
    showToast('Fichier supprimé.');
  } catch (error) {
    console.error('USER_FILES_LIBRARY_DELETE_FAILED', error);
    showToast('Impossible de supprimer le fichier.');
  } finally {
    state.deleting.delete(file.id);
    updateToolbar();
  }
}

function completeSelection(file) {
  if (!currentRequest) {
    return;
  }
  const resolver = currentRequest.resolve;
  currentRequest = null;
  modal.close();
  try { resolver({ file }); } catch (error) {
    console.error('USER_FILES_LIBRARY_RESOLVE_FAILED', error);
  }
}

function setStatus(text) {
  if (!refs) return;
  refs.status.textContent = text || '';
}

function updateToolbar() {
  if (!currentRequest || !refs) {
    return;
  }
  const { state, options } = currentRequest;
  const busy = state.loading || state.uploading || state.deleting.size > 0;
  if (refs.uploadLabel) {
    refs.uploadLabel.classList.toggle('is-busy', state.uploading);
  }
  if (refs.uploadInput) {
    refs.uploadInput.disabled = busy || !options.allowUpload;
  }
  if (refs.refresh) {
    refs.refresh.disabled = busy;
  }
  if (refs.list) {
    refs.list.querySelectorAll('button').forEach((btn) => {
      btn.disabled = busy;
    });
  }
}

function formatByteSize(bytes) {
  const units = ['octets', 'Ko', 'Mo', 'Go', 'To'];
  let value = Math.max(0, Number(bytes) || 0);
  let index = 0;
  while (value >= 1024 && index < units.length - 1) {
    value /= 1024;
    index += 1;
  }
  return index === 0 ? `${Math.round(value)} ${units[index]}` : `${value.toFixed(1)} ${units[index]}`;
}

// === New UI bits: breadcrumb, folders section, move popover ===

function buildFoldersIndex(folders) {
  const map = new Map();
  folders.forEach((f) => { if (f && f.id) map.set(String(f.id), f); });
  return map;
}

function renderBreadcrumb() {
  if (!refs || !refs.breadcrumb || !currentRequest) return;
  const { state } = currentRequest;
  const folders = Array.isArray(state.folders) ? state.folders : [];
  const index = buildFoldersIndex(folders);
  const current = state.currentFolderId || '';
  // Build path from current to root
  let cursor = current ? index.get(String(current)) : null;
  const chain = [];
  while (cursor) {
    chain.push(cursor);
    const parentId = cursor.parentId || '';
    cursor = parentId ? index.get(String(parentId)) : null;
  }
  chain.reverse();
  // Root
  const nav = document.createElement('div');
  nav.className = 'file-library__breadcrumb-inner';
  const root = document.createElement('button');
  root.type = 'button';
  root.className = 'link';
  root.textContent = 'Tous les fichiers';
  root.addEventListener('click', () => {
    state.currentFolderId = '';
    renderFolderSelect();
    renderBreadcrumb();
    renderFoldersSection();
    renderList();
  });
  nav.appendChild(root);
  chain.forEach((f) => {
    const sep = document.createElement('span');
    sep.textContent = ' / ';
    sep.setAttribute('aria-hidden', 'true');
    nav.appendChild(sep);
    const btn = document.createElement('button');
    btn.type = 'button';
    btn.className = 'link';
    btn.textContent = String(f.name || 'Dossier');
    btn.addEventListener('click', () => {
      state.currentFolderId = String(f.id || '');
      renderFolderSelect();
      renderBreadcrumb();
      renderFoldersSection();
      renderList();
    });
    nav.appendChild(btn);
  });
  refs.breadcrumb.innerHTML = '';
  refs.breadcrumb.appendChild(nav);
}

function renderFoldersSection() {
  if (!refs || !refs.foldersList || !currentRequest) return;
  const { state } = currentRequest;
  const all = Array.isArray(state.folders) ? state.folders : [];
  const current = state.currentFolderId || '';
  const entries = all.filter((f) => (String(f.parentId || '') === (current || '')));
  const container = refs.foldersList;
  container.innerHTML = '';
  if (!entries.length) {
    container.style.display = 'none';
    return;
  }
  container.style.display = '';
  entries.forEach((folder) => {
    const row = document.createElement('div');
    row.className = 'file-library__folder-item';
    row.setAttribute('role', 'listitem');
    const icon = document.createElement('span');
    icon.className = 'file-library__folder-icon';
    icon.textContent = '📁';
    const name = document.createElement('button');
    name.type = 'button';
    name.className = 'link file-library__folder-name';
    name.textContent = String(folder.name || 'Dossier');
    name.addEventListener('click', () => {
      state.currentFolderId = String(folder.id || '');
      renderFolderSelect();
      renderBreadcrumb();
      renderFoldersSection();
      renderList();
    });
    const actions = document.createElement('div');
    actions.className = 'file-library__folder-actions';
    const more = document.createElement('button');
    more.type = 'button';
    more.className = 'btn ghost btn--icon';
    more.textContent = '…';
    more.setAttribute('aria-label', 'Actions dossier');
    more.addEventListener('click', (e) => openFolderMenu(e.currentTarget, folder));
    actions.appendChild(more);
    row.append(icon, name, actions);
    container.appendChild(row);
  });
}

function openFolderMenu(anchor, folder) {
  closeAnyPopover();
  const menu = document.createElement('div');
  menu.className = 'popover-menu';
  menu.setAttribute('role', 'menu');
  const btnRename = document.createElement('button');
  btnRename.type = 'button';
  btnRename.className = 'btn ghost popover-menu__item';
  btnRename.textContent = 'Renommer';
  btnRename.addEventListener('click', async () => {
    closeAnyPopover();
    const newName = (await promptDialog({ title: 'Renommer le dossier', label: 'Nouveau nom', defaultValue: String(folder.name || '') }))?.trim();
    if (!newName) return;
    setStatus('Renommage…');
    try {
      const res = await renameFolder(folder.id, newName);
      if (!res.ok) {
        showToast('Impossible de renommer le dossier.');
        return;
      }
      currentRequest.state.folders = currentRequest.state.folders.map((f) => (f.id === folder.id ? { ...f, name: newName } : f));
      renderFolderSelect();
      renderBreadcrumb();
      renderFoldersSection();
    } catch (e) {
      console.error('USER_FOLDER_RENAME_FAILED', e);
      showToast('Impossible de renommer le dossier.');
    } finally { setStatus(''); }
  });
  const btnDelete = document.createElement('button');
  btnDelete.type = 'button';
  btnDelete.className = 'btn ghost popover-menu__item file-library__delete';
  btnDelete.textContent = 'Supprimer';
  btnDelete.addEventListener('click', async () => {
    closeAnyPopover();
    const { files, folders } = getImmediateFolderChildren(folder.id);
    if ((files.length + folders.length) > 0) {
      showFolderNotEmptyDialog(folder, files, folders, { allowCascade: true });
      return;
    }
    const ok = await confirmDialog({ title: 'Supprimer le dossier', message: 'Supprimer ce dossier vide ?', okLabel: 'Supprimer', cancelLabel: 'Annuler' });
    if (!ok) return;
    setStatus('Suppression…');
    try {
      const res = await deleteFolder(folder.id);
      if (!res.ok) {
        showToast('Impossible de supprimer le dossier.');
        return;
      }
      currentRequest.state.folders = currentRequest.state.folders.filter((f) => f.id !== folder.id);
      if ((currentRequest.state.currentFolderId || '') === String(folder.id)) {
        currentRequest.state.currentFolderId = folder.parentId || '';
      }
      renderFolderSelect();
      renderBreadcrumb();
      renderFoldersSection();
      renderList();
      showToast('Dossier supprimé.');
    } catch (e) {
      console.error('USER_FOLDER_DELETE_FAILED', e);
      showToast('Impossible de supprimer le dossier.');
    } finally { setStatus(''); }
  });
  menu.append(btnRename, btnDelete);
  document.body.appendChild(menu);
  positionPopover(anchor, menu);
  installGlobalPopoverCloser(menu);
}

function createMovePopover(anchor, folders, onSelect) {
  closeAnyPopover();
  const pop = document.createElement('div');
  pop.className = 'popover-menu';
  pop.setAttribute('role', 'menu');
  const root = document.createElement('button');
  root.type = 'button';
  root.className = 'btn ghost popover-menu__item';
  root.textContent = 'Racine';
  root.addEventListener('click', () => onSelect(''));
  pop.appendChild(root);
  folders.forEach((f) => {
    const btn = document.createElement('button');
    btn.type = 'button';
    btn.className = 'btn ghost popover-menu__item';
    btn.textContent = String(f.name || 'Dossier');
    btn.addEventListener('click', () => onSelect(String(f.id || '')));
    pop.appendChild(btn);
  });
  document.body.appendChild(pop);
  positionPopover(anchor, pop);
  installGlobalPopoverCloser(pop);
  return () => { try { pop.remove(); } catch (_) {} };
}

function positionPopover(anchor, pop) {
  try {
    const rect = anchor.getBoundingClientRect();
    pop.style.position = 'absolute';
    pop.style.minWidth = Math.max(rect.width, 160) + 'px';
    pop.style.top = `${window.scrollY + rect.bottom + 4}px`;
    pop.style.left = `${window.scrollX + rect.left}px`;
    pop.style.zIndex = '9999';
  } catch (_) {}
}

function installGlobalPopoverCloser(pop) {
  const close = () => { try { pop.remove(); } catch (_) {} detach(); };
  const onDocClick = (e) => {
    if (!pop.contains(e.target)) close();
  };
  const onKey = (e) => { if (e.key === 'Escape') close(); };
  const detach = () => {
    document.removeEventListener('mousedown', onDocClick, true);
    document.removeEventListener('keydown', onKey, true);
  };
  document.addEventListener('mousedown', onDocClick, true);
  document.addEventListener('keydown', onKey, true);
  pop.__detach = detach;
}

function closeAnyPopover() {
  document.querySelectorAll('.popover-menu').forEach((el) => {
    try { el.__detach?.(); } catch (_) {}
    try { el.remove(); } catch (_) {}
  });
}

function openFile(file) {
  const url = typeof file.url === 'string' && file.url ? file.url : '';
  if (url) {
    try { window.open(url, '_blank', 'noopener'); } catch (_) {}
  }
}

function getImmediateFolderChildren(folderId) {
  const state = currentRequest?.state || { files: [], folders: [] };
  const files = (state.files || []).filter((f) => String(f.folderId || '') === String(folderId || ''));
  const folders = (state.folders || []).filter((d) => String(d.parentId || '') === String(folderId || ''));
  return { files, folders };
}

function showFolderNotEmptyDialog(folder, files, folders, options = {}) {
  const host = modal?.querySelector('.file-library');
  if (!host) return;
  const overlay = document.createElement('div');
  overlay.className = 'file-library__overlay';
  const dialog = document.createElement('div');
  dialog.className = 'file-library__dialog';
  const header = document.createElement('header');
  header.textContent = 'Dossier non vide';
  const body = document.createElement('div');
  body.className = 'dialog-body';
  const countTxt = `${files.length} fichier${files.length>1?'s':''} · ${folders.length} sous-dossier${folders.length>1?'s':''}`;
  const p = document.createElement('p');
  p.innerHTML = `Le dossier « <strong>${escapeHtml(String(folder.name||'Dossier'))}</strong> » contient des éléments.<br/><span class="muted">${countTxt}</span>`;
  body.appendChild(p);
  const maxShow = 8;
  if (folders.length) {
    const h = document.createElement('div');
    h.textContent = 'Sous-dossiers :';
    h.style.marginTop = '8px';
    body.appendChild(h);
    const ul = document.createElement('ul');
    folders.slice(0, maxShow).forEach((d) => {
      const li = document.createElement('li');
      li.textContent = String(d.name || 'Dossier');
      ul.appendChild(li);
    });
    if (folders.length > maxShow) {
      const li = document.createElement('li');
      li.className = 'muted';
      li.textContent = `… et ${folders.length - maxShow} autres`;
      ul.appendChild(li);
    }
    body.appendChild(ul);
  }
  if (files.length) {
    const h = document.createElement('div');
    h.textContent = 'Fichiers :';
    h.style.marginTop = '8px';
    body.appendChild(h);
    const ul = document.createElement('ul');
    files.slice(0, maxShow).forEach((f) => {
      const li = document.createElement('li');
      li.textContent = String(f.name || f.id || 'Fichier');
      ul.appendChild(li);
    });
    if (files.length > maxShow) {
      const li = document.createElement('li');
      li.className = 'muted';
      li.textContent = `… et ${files.length - maxShow} autres`;
      ul.appendChild(li);
    }
    body.appendChild(ul);
  }
  const footer = document.createElement('div');
  footer.className = 'dialog-footer';
  const closeBtn = document.createElement('button');
  closeBtn.type = 'button';
  closeBtn.className = 'btn';
  closeBtn.textContent = 'Fermer';
  closeBtn.addEventListener('click', () => {
    try { overlay.remove(); } catch (_) {}
  });
  if (options && options.allowCascade) {
    const cascadeBtn = document.createElement('button');
    cascadeBtn.type = 'button';
    cascadeBtn.className = 'btn danger';
    cascadeBtn.textContent = 'Supprimer tout';
    cascadeBtn.addEventListener('click', async () => {
      try {
        closeAnyPopover();
        setStatus('Suppression…');
        await requestJson('/api/commands', { method: 'POST', body: { type: 'File.DeleteFolderCascade', payload: { folderId: folder.id } } });
        // Refresh pour refléter la suppression récursive complète
        const folders = await refreshUserFolders();
        const files = await refreshUserFiles();
        if (Array.isArray(folders)) currentRequest.state.folders = folders;
        if (Array.isArray(files)) currentRequest.state.files = files;
        if ((currentRequest.state.currentFolderId || '') === String(folder.id)) {
          currentRequest.state.currentFolderId = folder.parentId || '';
        }
        renderFolderSelect();
        renderBreadcrumb();
        renderFoldersSection();
        renderList();
        showToast('Dossier et contenu supprimés.');
      } catch (e) {
        console.error('USER_FOLDER_DELETE_CASCADE_FAILED', e);
        showToast('Suppression impossible.');
      } finally {
        setStatus('');
        try { overlay.remove(); } catch (_) {}
      }
    });
    footer.appendChild(cascadeBtn);
  }
  footer.appendChild(closeBtn);
  dialog.append(header, body, footer);
  overlay.appendChild(dialog);
  host.appendChild(overlay);
  const onKey = (e) => { if (e.key === 'Escape') { try { overlay.remove(); } catch (_) {} document.removeEventListener('keydown', onKey, true);} };
  document.addEventListener('keydown', onKey, true);
}
