import { openFileLibrary } from '../../../packages/ui/file-library.js';
import '../../../packages/ui/modal.js';
import { promptDialog } from '../../../packages/ui/dialog.js';
import { showToast } from '../../../packages/ui/toast.js';
import { createUserTextFile, createUserHtmlFile, readUserFileText, writeUserFileText, writeUserFileHtml } from '../../../packages/services/files.js';
import { upsertUserFile } from '../../../packages/files/store.js';
import { mapFileToAttachment } from '../../../packages/files/attachments-helpers.js';
import {
  cloneFileAttachments,
  getFileAttachment,
  withFileAttachment,
} from '../../../packages/files/attachments.js';
import { createDocSession } from '../../../packages/docs/doc-session.js';
import { resolveDocAdapter } from '../../../packages/docs/adapter-registry.js';
import '../../../packages/docs/adapters/markdown-textarea.js';
import '../../../packages/docs/adapters/html-quill.js';
import '../../../packages/docs/adapters/markdown-toastui.js';
import { escapeHtml } from '../../../packages/utils/format.js';

const MARKDOWN_ACCEPT = 'text/markdown,text/plain,.md,.markdown,.mdown,.mkd,.txt';

function nowSeconds() {
  return Math.floor(Date.now() / 1000);
}

function noteTab(el, { item, commands }) {
  const tags = Array.isArray(item?.tags) ? item.tags : [];
  const isNote = tags.some((t) => (t.key ?? t.k) === 'type/note');
  if (!isNote) {
    try { if (el && el.tagName && el.tagName.toLowerCase() === 'sb-tab') el.remove(); } catch (_) {}
    return;
  }

  const baseProps = { ...(item?.props || {}) };
  if (baseProps.files && typeof baseProps.files === 'object') {
    baseProps.files = cloneFileAttachments(baseProps.files);
  }

  const initialAttachment = getFileAttachment(baseProps, 'note');

  const state = {
    itemId: item?.id,
    commands: commands && typeof commands === 'object' ? commands : null,
    props: baseProps,
    currentView: initialAttachment ? buildNoteViewFromAttachment(initialAttachment) : null,
    pending: 0,
  };

  const wrap = document.createElement('div');
  wrap.className = 'sb-note-side';
  const panel = document.createElement('div');
  panel.className = 'sb-note-panel';
  wrap.appendChild(panel);
  el.appendChild(wrap);

  const refs = buildNotePanel(panel);

  const session = createDocSession({
    slot: 'note',
    attachment: initialAttachment,
    onAttachmentChange: (attachment, meta) => handleAttachmentChange(state, session, refs, attachment, meta),
    io: createNoteIO(),
  });

  const adapterEntry = resolveDocAdapter({
    slot: 'note',
    attachment: session.getState().attachment,
    item,
  });
  let adapterCleanup = null;
  if (adapterEntry && typeof adapterEntry.create === 'function') {
    adapterCleanup = adapterEntry.create({
      container: refs.editorHost,
      session,
      context: { slot: 'note', item },
    }) || null;
  }

  const unsubscribe = session.onChange((snapshot) => {
    updateNotePanelState(refs, state, snapshot);
  });

  updateNotePanelState(refs, state, session.getState());

  const handlers = {
    choose: async () => {
      if (!ensureCommands(state)) {
        return;
      }
      await runWithPending(state, refs, session, async () => {
        try {
          const result = await openFileLibrary({
            title: 'Fichiers Markdown',
            description: 'Sélectionnez ou importez un fichier Markdown à associer à cette note.',
            accept: MARKDOWN_ACCEPT,
            filter: isMarkdownFile,
            selectLabel: 'Utiliser ce fichier',
            emptyLabel: 'Aucun fichier texte disponible pour le moment.',
          });
          if (result && result.file) {
            const success = await attachFile(state, session, refs, result.file);
            if (success) {
              showToast('Fichier Markdown sélectionné.');
            } else {
              showToast('Impossible d’utiliser ce fichier.');
            }
          }
        } catch (error) {
          console.error('USER_NOTE_SELECT_FAILED', error);
          showToast('Impossible de sélectionner un fichier.');
        }
      });
    },
    create: async () => {
      if (!ensureCommands(state)) {
        return;
      }
      const defaultName = state.currentView?.name || 'nouvelle-note.md';
      let value = await promptDialog({ title: 'Nouveau fichier Markdown', label: 'Nom du fichier', defaultValue: defaultName });
      if (value === null) {
        return;
      }
      value = value.trim();
      if (!value) {
        showToast('Veuillez saisir un nom de fichier.');
        return;
      }
      await runWithPending(state, refs, session, async () => {
        try {
          const result = await createUserTextFile({ name: value, content: '' });
          if (!result.ok || !result.file) {
            throw result.error || new Error('CREATE_TEXT_FAILED');
          }
          const initialContent = typeof result.content === 'string' ? result.content : '';
          const success = await attachFile(state, session, refs, result.file, { content: initialContent });
          if (success) {
            showToast('Fichier Markdown créé.');
          } else {
            showToast('Impossible de créer le fichier Markdown.');
          }
        } catch (error) {
          console.error('USER_NOTE_CREATE_FAILED', error);
          showToast('Impossible de créer le fichier Markdown.');
        }
      });
    },
    open: () => {
      const url = state.currentView?.url;
      if (!url) {
        showToast('Aucun fichier à ouvrir.');
        return;
      }
      window.open(url, '_blank');
    },
    togglePreview: () => {
      // Delegate to the internal adapter preview button if present
      const btn = refs.editorHost?.querySelector('.sb-note-editor__toolbar .btn, .sb-note-editor__toolbar button');
      if (btn && !btn.disabled) {
        btn.click();
      }
    },
    detach: async () => {
      if (!ensureCommands(state)) {
        return;
      }
      if (!session.getState().attachment) {
        showToast('Aucun fichier sélectionné.');
        return;
      }
      await runWithPending(state, refs, session, async () => {
        try {
          await session.setAttachment(null, {
            reason: 'detach',
            triggerPersist: true,
            content: '',
            loadedContent: '',
          });
          showToast('Note détachée.');
        } catch (error) {
          console.error('USER_NOTE_DETACH_FAILED', error);
        }
      });
    },
  };

  refs.chooseButton?.addEventListener('click', handlers.choose);
  refs.createButton?.addEventListener('click', handlers.create);
  refs.openButton?.addEventListener('click', handlers.open);
  refs.previewToggle?.addEventListener('click', handlers.togglePreview);
  refs.detachButton?.addEventListener('click', handlers.detach);

  if (initialAttachment && initialAttachment.fileId) {
    session.load()
      .then((result) => {
        if (!result || !result.ok) {
          showToast('Impossible de charger le fichier Markdown.');
        }
      })
      .catch((error) => {
        console.error('USER_NOTE_INITIAL_LOAD_FAILED', error);
        showToast('Impossible de charger le fichier Markdown.');
      });
  }

  return () => {
    unsubscribe?.();
    adapterCleanup?.();
    session.destroy();
    refs.chooseButton?.removeEventListener('click', handlers.choose);
    refs.createButton?.removeEventListener('click', handlers.create);
    refs.openButton?.removeEventListener('click', handlers.open);
    refs.detachButton?.removeEventListener('click', handlers.detach);
  };
}

function textTab(el, { item, commands }) {
  const tags = Array.isArray(item?.tags) ? item.tags : [];
  const isText = tags.some((t) => (t.key ?? t.k) === 'type/text');
  if (!isText) {
    try { if (el && el.tagName && el.tagName.toLowerCase() === 'sb-tab') el.remove(); } catch (_) {}
    return;
  }

  const baseProps = { ...(item?.props || {}) };
  if (baseProps.files && typeof baseProps.files === 'object') {
    baseProps.files = cloneFileAttachments(baseProps.files);
  }

  const initialAttachment = getFileAttachment(baseProps, 'text');

  const state = {
    itemId: item?.id,
    commands: commands && typeof commands === 'object' ? commands : null,
    props: baseProps,
    currentView: initialAttachment ? buildNoteViewFromAttachment(initialAttachment) : null,
    pending: 0,
  };

  const wrap = document.createElement('div');
  wrap.className = 'sb-note-side';
  const panel = document.createElement('div');
  panel.className = 'sb-note-panel';
  wrap.appendChild(panel);
  el.appendChild(wrap);

  const refs = buildNotePanel(panel);
  // Override header for text editor
  try {
    const titleNode = panel.querySelector('.sb-note-panel__title');
    if (titleNode) titleNode.textContent = '✒️ Texte';
  } catch (_) {}
  // No preview toggle for HTML editor
  try { if (refs.previewToggle) refs.previewToggle.style.display = 'none'; } catch (_) {}

  const session = createDocSession({
    slot: 'text',
    attachment: initialAttachment,
    onAttachmentChange: (attachment, meta) => handleTextAttachmentChange(state, session, refs, attachment, meta),
    io: createTextIO(),
  });

  const adapterEntry = resolveDocAdapter({ slot: 'text', attachment: session.getState().attachment, item });
  let adapterCleanup = null;
  if (adapterEntry && typeof adapterEntry.create === 'function') {
    adapterCleanup = adapterEntry.create({ container: refs.editorHost, session, context: { slot: 'text', item } }) || null;
  }

  const unsubscribe = session.onChange((snapshot) => { updateNotePanelState(refs, state, snapshot); });
  updateNotePanelState(refs, state, session.getState());

  const handlers = {
    async choose() {
      return runWithPending(state, refs, session, async () => {
        const result = await openFileLibrary({ title: 'Bibliothèque de documents', accept: 'text/*,.md,.markdown,.txt,.html', emptyLabel: 'Aucun document pour le moment.', selectLabel: 'Utiliser ce document' });
        const file = result?.file || null;
        if (!file) return;
        const attachment = mapFileToAttachment(file, { role: 'text' });
        await session.setAttachment(attachment, { reason: 'select', triggerPersist: true });
      });
    },
    async create() {
      return runWithPending(state, refs, session, async () => {
        const name = await promptDialog({ title: 'Nouveau document', message: 'Nom du fichier (.md recommandé)', defaultValue: 'nouveau-document.md' });
        const value = String(name || '').trim();
        if (!value) return;
        const result = await createUserHtmlFile({ name: value, content: '' });
        if (!result?.ok || !result.file) { showToast('Impossible de créer le fichier.'); return; }
        upsertUserFile(result.file);
        const attachment = mapFileToAttachment(result.file, { role: 'text' });
        await session.setAttachment(attachment, { reason: 'create', triggerPersist: true, content: '', loadedContent: '' });
      });
    },
    open() {
      const att = session.getState().attachment;
      if (!att?.url) { showToast('Aucun fichier sélectionné.'); return; }
      try { window.open(att.url, '_blank', 'noopener'); } catch (_) {}
    },
    download() {
      const snap = session.getState();
      const html = ensureStandaloneHtml(String(snap?.content || ''));
      const blob = new Blob([html], { type: 'text/html;charset=utf-8' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      const baseName = (state.currentView?.name || 'document').trim();
      const name = suggestHtmlName(baseName);
      a.href = url; a.download = name;
      document.body.appendChild(a);
      a.click();
      setTimeout(() => { try { URL.revokeObjectURL(url); } catch (_) {} try { a.remove(); } catch (_) {} }, 0);
    },
    async detach() {
      return runWithPending(state, refs, session, async () => {
        await session.setAttachment(null, { reason: 'detach', triggerPersist: true, silent: false });
      });
    },
  };

  refs.chooseButton?.addEventListener('click', handlers.choose);
  refs.createButton?.addEventListener('click', handlers.create);
  refs.openButton?.addEventListener('click', handlers.open);
  // Inject a Download (.html) button next to actions
  try {
    const actionsNode = panel.querySelector('.sb-note-panel__actions');
    if (actionsNode) {
      const downloadButton = document.createElement('button');
      downloadButton.type = 'button';
      downloadButton.className = 'btn ghost btn--icon';
      downloadButton.textContent = '⬇';
      downloadButton.title = 'Télécharger (.html)';
      downloadButton.setAttribute('aria-label', 'Télécharger en HTML');
      downloadButton.addEventListener('click', handlers.download);
      actionsNode.appendChild(downloadButton);
    }
  } catch (_) {}
  refs.detachButton?.addEventListener('click', handlers.detach);

  if (initialAttachment && initialAttachment.fileId) {
    session.load().catch((error) => { console.error('USER_TEXT_INITIAL_LOAD_FAILED', error); showToast('Impossible de charger le document HTML.'); });
  }

  return () => {
    unsubscribe?.();
    if (typeof adapterCleanup === 'function') {
      try { adapterCleanup(); } catch (_) {}
    }
    session.destroy();
    refs.chooseButton?.removeEventListener('click', handlers.choose);
    refs.createButton?.removeEventListener('click', handlers.create);
    refs.openButton?.removeEventListener('click', handlers.open);
    try {
      const actionsNode = panel.querySelector('.sb-note-panel__actions');
      const downloadBtn = actionsNode?.querySelector('button[aria-label="Télécharger en HTML"]');
      if (downloadBtn) downloadBtn.removeEventListener('click', handlers.download);
    } catch (_) {}
    refs.detachButton?.removeEventListener('click', handlers.detach);
  };
}

function createTextIO() {
  return {
    async read(attachment) {
      if (!attachment || !attachment.fileId) {
        return { ok: false, error: new Error('NO_FILE'), message: '⚠️ Aucun fichier HTML sélectionné.' };
      }
      try {
        const result = await readUserFileText(attachment.fileId);
        if (!result.ok) throw result.error || new Error('READ_TEXT_FAILED');
        if (result.file) upsertUserFile(result.file);
        const updatedAttachment = result.file ? mapFileToAttachment(result.file, { role: 'text' }) : attachment;
        return { ok: true, attachment: updatedAttachment, content: String(result.content || ''), checksum: updatedAttachment?.checksum ?? null };
      } catch (error) {
        return { ok: false, error, message: '⚠️ Impossible de lire le document HTML.' };
      }
    },
    async write(attachment, content) {
      if (!attachment || !attachment.fileId) {
        return { ok: false, error: new Error('NO_FILE'), message: '⚠️ Aucun fichier HTML sélectionné.' };
      }
      try {
        const html = ensureStandaloneHtml(String(content || ''));
        const result = await writeUserFileHtml(attachment.fileId, html, attachment.checksum || null);
        if (!result.ok || !result.file) throw result.error || new Error('WRITE_HTML_FAILED');
        upsertUserFile(result.file);
        const updatedAttachment = mapFileToAttachment(result.file, { role: 'text' });
        const normalizedContent = typeof result.content === 'string' ? result.content : html;
        return { ok: true, attachment: updatedAttachment, content: normalizedContent, checksum: updatedAttachment?.checksum ?? null, triggerPersist: true };
      } catch (error) {
        if (isFileConflictError(error)) {
          return { ok: false, error, conflict: true, message: '⚠️ Le fichier a été modifié ailleurs. Rechargement…' };
        }
        return { ok: false, error, message: '⚠️ Impossible d’enregistrer le document.' };
      }
    },
  };
}

async function handleTextAttachmentChange(state, session, refs, attachment, meta = {}) {
  const previousProps = state.props;
  const previousView = state.currentView;
  const nextProps = withFileAttachment(state.props, 'text', attachment);
  delete nextProps.textUrl;
  state.props = nextProps;
  state.currentView = attachment ? buildNoteViewFromAttachment(attachment) : null;
  updateNotePanelState(refs, state, session.getState());

  if (!meta.triggerPersist || !ensureCommands(state, { silent: true }) || !state.itemId) {
    return Promise.resolve();
  }

  return state.commands.updateNode(state.itemId, { props: state.props }).catch((error) => {
    console.error('ITEM_TEXT_UPDATE_FAILED', error);
    state.props = previousProps; state.currentView = previousView;
    updateNotePanelState(refs, state, session.getState());
    if (meta.previousAttachment) {
      session.setAttachment(meta.previousAttachment, { silent: true, reason: 'rollback', content: meta.previousContent, loadedContent: meta.previousLoadedContent, triggerPersist: false });
    }
    if (!meta.silent) { showToast('Impossible de mettre à jour le document.'); }
    throw error;
  });
}

function ensureStandaloneHtml(innerHtml) {
  const html = String(innerHtml || '').trim();
  // If already a full document, keep as-is
  if (/<!DOCTYPE\s+html/i.test(html) || /<html[\s>]/i.test(html)) {
    return html;
  }
  const css = minimalStandaloneCss();
  const head = `<!DOCTYPE html><html lang="fr"><head><meta charset="utf-8"/><meta name="viewport" content="width=device-width, initial-scale=1"/><title>Document</title><style>${css}</style></head>`;
  const body = `<body><article class="ql-container ql-snow"><div class="ql-editor">${html}</div></article></body></html>`;
  return head + body;
}

function minimalStandaloneCss() {
  return `
  html,body{margin:0;padding:0;color:#0f172a;background:#fff;font:16px/1.6 system-ui,-apple-system,Segoe UI,Roboto,Ubuntu,Cantarell,Noto Sans,sans-serif}
  .ql-container{box-sizing:border-box}
  .ql-editor{padding:16px}
  .ql-editor h1{font-size:2em;margin:0.67em 0}
  .ql-editor h2{font-size:1.5em;margin:0.75em 0}
  .ql-editor h3{font-size:1.25em;margin:0.85em 0}
  .ql-editor p{margin:0.6em 0}
  .ql-editor blockquote{border-left:4px solid #e2e8f0;margin:1em 0;padding:0.5em 1em;color:#475569;background:#f8fafc}
  .ql-editor pre{background:#f8fafc;border-radius:6px;padding:12px;overflow:auto}
  .ql-editor code{background:rgba(15,23,42,.06);padding:2px 4px;border-radius:4px}
  .ql-editor ul{list-style:disc;margin:0.6em 0 0.6em 1.6em}
  .ql-editor ol{list-style:decimal;margin:0.6em 0 0.6em 1.6em}
  .ql-editor a{color:#2563eb;text-decoration:underline}
  .ql-align-center{text-align:center}
  .ql-align-right{text-align:right}
  .ql-align-justify{text-align:justify}
  .ql-size-small{font-size:0.875em}
  .ql-size-large{font-size:1.5em}
  .ql-size-huge{font-size:2.5em}
  img{max-width:100%;display:block;margin:8px 0;border-radius:6px}
  table{border-collapse:collapse;margin:1em 0;width:100%}
  td,th{border:1px solid #e2e8f0;padding:6px 8px}
  `;
}

function suggestHtmlName(base) {
  const raw = String(base || 'document');
  const cleaned = raw.replace(/\s+/g, ' ').trim();
  const noBad = cleaned.replace(/[\\/:*?"<>|]/g, '_');
  if (/\.html?$/i.test(noBad)) return noBad;
  if (/\.md$/i.test(noBad)) return noBad.replace(/\.md$/i, '.html');
  return noBad + '.html';
}

function buildNotePanel(container) {
  // Header row with title and compact actions, aligned with editor header
  const header = document.createElement('div');
  header.className = 'sb-note-panel__header';
  const headerTitle = document.createElement('div');
  headerTitle.className = 'sb-note-panel__title';
  headerTitle.textContent = '📝 Note';
  const actions = document.createElement('div');
  actions.className = 'sb-note-panel__actions';

  const chooseButton = document.createElement('button');
  chooseButton.type = 'button';
  chooseButton.className = 'btn ghost btn--icon';
  chooseButton.textContent = '📂';
  chooseButton.title = 'Choisir dans la bibliothèque';
  chooseButton.setAttribute('aria-label', 'Choisir dans la bibliothèque');

  const createButton = document.createElement('button');
  createButton.type = 'button';
  createButton.className = 'btn ghost btn--icon';
  createButton.textContent = '➕';
  createButton.title = 'Créer un fichier Markdown';
  createButton.setAttribute('aria-label', 'Créer un fichier Markdown');

  const openButton = document.createElement('button');
  openButton.type = 'button';
  openButton.className = 'btn ghost btn--icon';
  openButton.textContent = '↗';
  openButton.title = 'Ouvrir dans un nouvel onglet';
  openButton.setAttribute('aria-label', 'Ouvrir dans un nouvel onglet');

  const detachButton = document.createElement('button');
  detachButton.type = 'button';
  detachButton.className = 'btn ghost btn--icon';
  detachButton.textContent = '⏏';
  detachButton.title = 'Détacher ce fichier de la note';
  detachButton.setAttribute('aria-label', 'Détacher ce fichier de la note');

  const previewToggle = document.createElement('button');
  previewToggle.type = 'button';
  previewToggle.className = 'btn ghost btn--icon';
  previewToggle.textContent = '👁';
  previewToggle.title = 'Afficher/masquer l\'aperçu';
  previewToggle.setAttribute('aria-label', 'Afficher ou masquer l\'aperçu');

  actions.append(chooseButton, createButton, openButton, previewToggle, detachButton);
  header.append(headerTitle, actions);

  const fileInfo = document.createElement('div');
  fileInfo.className = 'sb-note-panel__file-info';
  fileInfo.textContent = 'Aucun fichier sélectionné.';

  const empty = document.createElement('div');
  empty.className = 'sb-note-panel__empty';
  empty.textContent = 'Associez un fichier Markdown pour éditer cette note.';

  const status = document.createElement('div');
  status.className = 'sb-note-panel__status';
  status.setAttribute('aria-live', 'polite');

  const editorWrapper = document.createElement('div');
  editorWrapper.className = 'sb-note-panel__editor-wrapper';

  const editorHost = document.createElement('div');
  editorHost.className = 'sb-note-panel__editor-host';
  editorWrapper.appendChild(editorHost);

  container.append(header, fileInfo, empty, status, editorWrapper);

  return {
    fileInfo,
    chooseButton,
    createButton,
    openButton,
    previewToggle,
    detachButton,
    empty,
    status,
    editorWrapper,
    editorHost,
  };
}



function handleAttachmentChange(state, session, refs, attachment, meta = {}) {
  const previousProps = state.props;
  const previousView = state.currentView;
  const nextProps = withFileAttachment(state.props, 'note', attachment);
  // Strip legacy fields to keep a single source of truth
  delete nextProps.note;
  state.props = nextProps;
  state.currentView = attachment ? buildNoteViewFromAttachment(attachment) : null;
  updateNotePanelState(refs, state, session.getState());

  if (!meta.triggerPersist || !ensureCommands(state, { silent: true }) || !state.itemId) {
    return Promise.resolve();
  }

  return state.commands.updateNode(state.itemId, { props: state.props }).catch((error) => {
    console.error('ITEM_NOTE_UPDATE_FAILED', error);
    state.props = previousProps;
    state.currentView = previousView;
    updateNotePanelState(refs, state, session.getState());
    if (meta.previousAttachment) {
      session.setAttachment(meta.previousAttachment, {
        silent: true,
        reason: 'rollback',
        content: meta.previousContent,
        loadedContent: meta.previousLoadedContent,
        triggerPersist: false,
      });
    }
    if (!meta.silent) {
      showToast('Impossible de mettre à jour la note.');
    }
    throw error;
  });
}

function ensureCommands(state, options = {}) {
  const silent = !!options.silent;
  if (!state.itemId) {
    if (!silent) {
      showToast('Item introuvable.');
    }
    return false;
  }
  if (!state.commands || typeof state.commands.updateNode !== 'function') {
    if (!silent) {
      showToast('La note ne peut pas être mise à jour pour le moment.');
    }
    return false;
  }
  return true;
}

async function runWithPending(state, refs, session, fn) {
  state.pending += 1;
  updateNotePanelState(refs, state, session.getState());
  try {
    return await fn();
  } finally {
    state.pending = Math.max(0, state.pending - 1);
    updateNotePanelState(refs, state, session.getState());
  }
}

function createNoteIO() {
  return {
    async read(attachment) {
      if (!attachment || !attachment.fileId) {
        return { ok: false, error: new Error('NO_FILE'), message: '⚠️ Aucun fichier Markdown sélectionné.' };
      }
      try {
        const result = await readUserFileText(attachment.fileId);
        if (!result.ok) {
          throw result.error || new Error('READ_TEXT_FAILED');
        }
        if (result.file) {
          upsertUserFile(result.file);
        }
        const updatedAttachment = result.file ? mapFileToAttachment(result.file, { role: 'note' }) : attachment;
        return {
          ok: true,
          attachment: updatedAttachment,
          content: typeof result.content === 'string' ? result.content : '',
          checksum: updatedAttachment?.checksum ?? null,
        };
      } catch (error) {
        return { ok: false, error, message: '⚠️ Impossible de lire le fichier Markdown.' };
      }
    },
    async write(attachment, content) {
      if (!attachment || !attachment.fileId) {
        return { ok: false, error: new Error('NO_FILE'), message: '⚠️ Aucun fichier Markdown sélectionné.' };
      }
      try {
        const result = await writeUserFileText(attachment.fileId, content, attachment.checksum || null);
        if (!result.ok || !result.file) {
          throw result.error || new Error('WRITE_TEXT_FAILED');
        }
        upsertUserFile(result.file);
        const updatedAttachment = mapFileToAttachment(result.file, { role: 'note' });
        const normalizedContent = typeof result.content === 'string' ? result.content : content;
        return {
          ok: true,
          attachment: updatedAttachment,
          content: normalizedContent,
          checksum: updatedAttachment?.checksum ?? null,
          triggerPersist: true,
        };
      } catch (error) {
        if (isFileConflictError(error)) {
          return {
            ok: false,
            error,
            conflict: true,
            message: '⚠️ Le fichier a été modifié ailleurs. Rechargement…',
          };
        }
        return { ok: false, error, message: '⚠️ Impossible d’enregistrer la note.' };
      }
    },
  };
}

function updateNotePanelState(refs, state, snapshot) {
  const sessionState = snapshot || {};
  const hasAttachment = !!sessionState.attachment;
  if (refs.empty) {
    refs.empty.style.display = hasAttachment ? 'none' : '';
  }
  if (refs.editorWrapper) {
    refs.editorWrapper.style.display = hasAttachment ? '' : 'none';
  }
  if (refs.fileInfo) {
    refs.fileInfo.textContent = hasAttachment ? buildNoteFileInfo(state.currentView) : 'Aucun fichier sélectionné.';
  }

  const busy = state.pending > 0;
  const loading = !!sessionState.loading;
  const saving = !!sessionState.saving;
  const disableActions = busy || loading || saving;

  if (refs.chooseButton) {
    refs.chooseButton.disabled = disableActions;
  }
  if (refs.createButton) {
    refs.createButton.disabled = disableActions;
  }
  if (refs.detachButton) {
    refs.detachButton.disabled = disableActions || !hasAttachment;
  }
  if (refs.openButton) {
    const hasUrl = !!state.currentView?.url;
    const openDisabled = disableActions || !hasAttachment || !hasUrl;
    refs.openButton.disabled = openDisabled;
    let title = '';
    if (!hasAttachment) {
      title = 'Aucun fichier sélectionné';
    } else if (busy) {
      title = 'Mise à jour en cours';
    } else if (loading) {
      title = 'Chargement en cours';
    } else if (saving) {
      title = 'Enregistrement en cours';
    } else if (!hasUrl) {
      title = 'Aucun fichier à ouvrir';
    } else {
      title = 'Ouvrir dans un nouvel onglet';
    }
    if (title) {
      refs.openButton.title = title;
    } else {
      refs.openButton.removeAttribute('title');
    }
  }

  if (refs.status) {
    let statusText = '';
    if (state.pending > 0) {
      statusText = 'Mise à jour…';
    } else if (loading) {
      statusText = 'Chargement…';
    } else if (saving) {
      statusText = 'Enregistrement…';
    } else if (sessionState.errorMessage) {
      statusText = String(sessionState.errorMessage);
    }
    refs.status.textContent = statusText;
    refs.status.style.display = statusText ? '' : 'none';
  }
}

async function attachFile(state, session, refs, file, options = {}) {
  const noteData = buildNoteDataFromFile(file);
  if (!noteData) {
    return false;
  }
  const initialContent = options.content;
  try {
    await session.setAttachment(noteData.attachment, {
      reason: 'attach',
      triggerPersist: true,
      content: initialContent !== undefined ? initialContent : undefined,
      loadedContent: initialContent !== undefined ? initialContent : undefined,
    });
    upsertUserFile(file);
    if (initialContent === undefined) {
      const result = await session.load();
      if (!result || !result.ok) {
        showToast('Impossible de charger le fichier Markdown.');
      }
    }
    return true;
  } catch (error) {
    console.error('USER_NOTE_APPLY_FILE_FAILED', error);
    return false;
  }
}

function buildNoteDataFromFile(file) {
  const attachment = mapFileToAttachment(file, { role: 'note' });
  if (!attachment) {
    return null;
  }
  const view = buildNoteViewFromAttachment(attachment);
  if (!view) {
    return null;
  }
  return { attachment, view };
}

function buildNoteViewFromAttachment(attachment) {
  if (!attachment || typeof attachment !== 'object') {
    return null;
  }
  const mimeType = typeof attachment.mimeType === 'string' && attachment.mimeType ? attachment.mimeType : 'text/markdown';
  return {
    mode: 'file',
    fileId: attachment.fileId || '',
    name: typeof attachment.name === 'string' ? attachment.name : '',
    mimeType,
    byteSize: Number.isFinite(attachment.byteSize) ? Number(attachment.byteSize) : null,
    sizeLabel: typeof attachment.sizeLabel === 'string' ? attachment.sizeLabel : null,
    checksum: typeof attachment.checksum === 'string' ? attachment.checksum : null,
    url: typeof attachment.url === 'string' ? attachment.url : '',
    updatedAt: Number.isFinite(attachment.updatedAt) ? Number(attachment.updatedAt) : null,
    source: 'file',
  };
}

function buildNoteFileInfo(view) {
  if (!view) {
    return 'Aucun fichier sélectionné.';
  }
  const parts = [];
  if (view.name) {
    parts.push(view.name);
  }
  if (view.sizeLabel) {
    parts.push(view.sizeLabel);
  } else if (Number.isFinite(view.byteSize)) {
    parts.push(formatByteSize(view.byteSize));
  }
  if (view.mimeType) {
    parts.push(view.mimeType);
  }
  return parts.length ? parts.join(' · ') : 'Fichier Markdown';
}

function isMarkdownFile(file) {
  if (!file || typeof file !== 'object') {
    return false;
  }
  const mime = typeof file.mimeType === 'string' ? file.mimeType.toLowerCase() : '';
  if (mime.startsWith('text/')) {
    return true;
  }
  const name = typeof file.name === 'string' ? file.name.toLowerCase() : '';
  return /\.(md|markdown|mdown|mkd|txt)$/.test(name);
}

function isFileConflictError(error) {
  if (!error || typeof error !== 'object') {
    return false;
  }
  if (Number(error.status) === 409) {
    return true;
  }
  const payload = error.payload && typeof error.payload === 'object' ? error.payload : null;
  const code = typeof payload?.code === 'string'
    ? payload.code
    : (typeof error.code === 'string' ? error.code : '');
  return code === 'FILE_CONFLICT' || code === 'FILE_CHECKSUM_MISMATCH';
}

function imageTab(el, { item, commands }) {
  const tags = Array.isArray(item?.tags) ? item.tags : [];
  const isImage = tags.some(t => (t.key ?? t.k) === 'type/image');
  if (!isImage) return;
  const baseProps = { ...(item?.props || {}) };
  if (baseProps.files && typeof baseProps.files === 'object') {
    baseProps.files = cloneFileAttachments(baseProps.files);
  }
  const state = {
    itemId: item?.id,
    props: baseProps,
    current: extractCurrentImage(baseProps),
    saving: false,
    selecting: false,
    commands: commands && typeof commands === 'object' ? commands : null,
  };
  const wrap = document.createElement('div');
  wrap.className = 'sb-tab';
  const title = document.createElement('div');
  title.className = 'sb-tab-title';
  title.textContent = '🖼️ Image';
  wrap.appendChild(title);
  const panel = document.createElement('div');
  panel.className = 'sb-image-panel';
  wrap.appendChild(panel);
  el.appendChild(wrap);

  const refs = buildPanel(panel);
  bindPanelEvents(refs, state);
  updatePreview(refs, state);
  updateControls(refs, state);
}

export default [
  { id: 'org:tab-note',  slot: 'item.sidePanel',    order: 20, label: 'Note',  render: noteTab  },
  { id: 'org:tab-text',  slot: 'item.sidePanel',    order: 21, label: 'Texte', render: textTab  },
  { id: 'org:state-footer', slot: 'item.editorFooter', order: 5,  label: 'État', render: organisationStateFooter },
  { id: 'org:tab-image', slot: 'item.editorTabs',   order: 30, label: 'Image', render: imageTab }
];

function buildPanel(container) {
  const preview = document.createElement('div');
  preview.className = 'sb-image-panel__preview';
  const previewImage = document.createElement('img');
  previewImage.className = 'sb-image-panel__preview-image';
  previewImage.alt = '';
  previewImage.decoding = 'async';
  previewImage.loading = 'lazy';
  const previewPlaceholder = document.createElement('div');
  previewPlaceholder.className = 'sb-image-panel__preview-placeholder';
  previewPlaceholder.textContent = 'Aucune image sélectionnée.';
  const previewInfo = document.createElement('div');
  previewInfo.className = 'sb-image-panel__info';
  preview.append(previewImage, previewPlaceholder, previewInfo);

  const controls = document.createElement('div');
  controls.className = 'sb-image-panel__controls';
  const chooseButton = document.createElement('button');
  chooseButton.type = 'button';
  chooseButton.className = 'btn primary';
  chooseButton.textContent = 'Choisir dans la bibliothèque';
  const removeButton = document.createElement('button');
  removeButton.type = 'button';
  removeButton.className = 'btn ghost';
  removeButton.textContent = 'Retirer l’image';
  controls.append(chooseButton, removeButton);

  const hint = document.createElement('p');
  hint.className = 'sb-image-panel__hint';
  hint.textContent = 'Accédez à votre bibliothèque pour téléverser un fichier ou en sélectionner un existant.';

  const urlBlock = document.createElement('div');
  urlBlock.className = 'sb-image-panel__url';
  const urlLabel = document.createElement('label');
  urlLabel.textContent = 'Ou utiliser une URL';
  const urlInput = document.createElement('input');
  urlInput.type = 'url';
  urlInput.placeholder = 'https://…';
  const applyUrlButton = document.createElement('button');
  applyUrlButton.type = 'button';
  applyUrlButton.className = 'btn ghost';
  applyUrlButton.textContent = 'Appliquer';
  urlBlock.append(urlLabel, urlInput, applyUrlButton);

  const status = document.createElement('div');
  status.className = 'sb-image-panel__status';

  container.append(preview, controls, hint, status, urlBlock);

  return {
    preview,
    previewImage,
    previewPlaceholder,
    previewInfo,
    chooseButton,
    removeButton,
    urlInput,
    applyUrlButton,
    status,
  };
}

function bindPanelEvents(refs, state) {
  if (refs.chooseButton) {
    refs.chooseButton.addEventListener('click', async () => {
      if (state.saving || state.selecting) {
        return;
      }
      state.selecting = true;
      setStatus(refs, 'Ouverture de la bibliothèque…');
      updateControls(refs, state);
      try {
        const result = await openFileLibrary({
          title: 'Bibliothèque d’images',
          description: 'Choisissez ou importez une image pour cet item.',
          accept: 'image/*',
          filter: isImageFile,
          emptyLabel: 'Aucune image téléversée pour le moment.',
          selectLabel: 'Utiliser cette image',
          autoSelectUploaded: true,
        });
        if (result && result.file) {
          const next = buildImageDataFromFile(result.file);
          if (next) {
            await applyNewImage(state, refs, next);
            showToast('Image mise à jour.');
          } else {
            showToast('Impossible d’utiliser ce fichier.');
          }
        }
      } catch (error) {
        console.error('USER_FILES_LIBRARY_SELECT_FAILED', error);
        showToast('Impossible de sélectionner un fichier.');
      } finally {
        state.selecting = false;
        setStatus(refs, '');
        updateControls(refs, state);
      }
    });
  }

  refs.removeButton.addEventListener('click', async () => {
    if (!state.current) {
      return;
    }
    await applyNewImage(state, refs, null);
    showToast('Image retirée.');
  });

  refs.applyUrlButton.addEventListener('click', async () => {
    const value = refs.urlInput.value.trim();
    if (!value) {
      showToast('Veuillez saisir une URL.');
      return;
    }
    if (!isValidUrl(value)) {
      showToast('URL invalide.');
      return;
    }
    const next = {
      mode: 'url',
      url: value,
      source: 'url',
      updatedAt: nowSeconds(),
    };
    await applyNewImage(state, refs, next);
    showToast('Image mise à jour.');
  });

  refs.urlInput.addEventListener('keydown', async (event) => {
    if (event.key === 'Enter') {
      event.preventDefault();
      refs.applyUrlButton.click();
    }
  });
}

async function applyNewImage(state, refs, nextImage) {
  const commands = state.commands;
  if (!commands || typeof commands.updateNode !== 'function' || !state.itemId) {
    return;
  }
  if (state.saving) {
    return;
  }

  const previousProps = state.props;
  const previousCurrent = state.current;

  let nextProps;
  let nextView = null;
  let libraryUpdate = null;

  if (nextImage && nextImage.mode === 'file' && nextImage.attachment && nextImage.view) {
    nextProps = withFileAttachment(state.props, 'image', nextImage.attachment);
    const viewUpdatedAt = Number.isFinite(nextImage.view.updatedAt)
      ? Number(nextImage.view.updatedAt)
      : nowSeconds();
    nextProps.image = {
      mode: 'file',
      fileKey: 'image',
      source: nextImage.view.source || 'upload',
      updatedAt: viewUpdatedAt,
    };
    nextView = { ...nextImage.view };
    if (nextImage.file) {
      libraryUpdate = nextImage.file;
    }
  } else if (nextImage && nextImage.mode === 'url' && typeof nextImage.url === 'string') {
    nextProps = withFileAttachment(state.props, 'image', null);
    const urlUpdatedAt = Number.isFinite(nextImage.updatedAt)
      ? Number(nextImage.updatedAt)
      : nowSeconds();
    nextProps.image = {
      mode: 'url',
      url: nextImage.url,
      source: typeof nextImage.source === 'string' ? nextImage.source : 'url',
      updatedAt: urlUpdatedAt,
    };
    nextView = {
      mode: 'url',
      url: nextImage.url,
      source: typeof nextImage.source === 'string' ? nextImage.source : 'url',
    };
  } else {
    nextProps = withFileAttachment(state.props, 'image', null);
    delete nextProps.image;
  }

  delete nextProps.imageUrl;

  state.props = nextProps;
  state.current = nextView;
  state.saving = true;
  updatePreview(refs, state);
  updateControls(refs, state);
  setStatus(refs, 'Enregistrement…');

  try {
    await commands.updateNode(state.itemId, { props: state.props });
    if (libraryUpdate) {
      upsertUserFile(libraryUpdate);
    }
  } catch (error) {
    console.error('ITEM_IMAGE_UPDATE_FAILED', error);
    state.props = previousProps;
    state.current = previousCurrent;
    updatePreview(refs, state);
    showToast('Impossible de mettre à jour l’image.');
  } finally {
    state.saving = false;
    setStatus(refs, '');
    updateControls(refs, state);
  }
}

function updatePreview(refs, state) {
  const current = state.current;
  if (current && typeof current.url === 'string' && current.url) {
    refs.preview.classList.add('sb-image-panel__preview--with-image');
    refs.previewImage.src = current.url;
    refs.previewPlaceholder.style.display = 'none';
    refs.previewInfo.textContent = buildPreviewInfo(current);
  } else {
    refs.preview.classList.remove('sb-image-panel__preview--with-image');
    refs.previewImage.src = '';
    refs.previewPlaceholder.style.display = '';
    refs.previewInfo.textContent = '';
  }
  if (refs.urlInput) {
    refs.urlInput.value = current?.mode === 'url' && typeof current.url === 'string' ? current.url : '';
  }
}

function updateControls(refs, state) {
  const busy = state.saving || state.selecting;
  if (refs.chooseButton) {
    refs.chooseButton.disabled = busy;
  }
  if (refs.removeButton) {
    refs.removeButton.disabled = busy || !state.current;
  }
  if (refs.applyUrlButton) {
    refs.applyUrlButton.disabled = state.saving;
  }
  if (refs.urlInput) {
    refs.urlInput.disabled = state.saving;
  }
}

function setStatus(refs, text) {
  if (!refs.status) return;
  refs.status.textContent = text || '';
}

function isImageFile(file) {
  if (!file || typeof file !== 'object') {
    return false;
  }
  const mime = typeof file.mimeType === 'string' ? file.mimeType.toLowerCase() : '';
  if (mime.startsWith('image/')) {
    return true;
  }
  const name = typeof file.name === 'string' ? file.name.toLowerCase() : '';
  return /\.(png|jpe?g|webp|gif|bmp|svg)$/.test(name);
}

function extractCurrentImage(props) {
  if (!props || typeof props !== 'object') {
    return null;
  }
  const attachment = getFileAttachment(props, 'image');
  if (attachment) {
    const descriptor = props.image && typeof props.image === 'object' ? props.image : null;
    const view = buildImageViewFromAttachment(attachment, descriptor);
    if (view) {
      return view;
    }
  }
  const image = props.image;
  if (image && typeof image === 'object') {
    const mode = typeof image.mode === 'string'
      ? image.mode
      : (typeof image.fileId === 'string'
        ? 'file'
        : (typeof image.url === 'string' ? 'url' : null));
    if (mode === 'file' && typeof image.fileId === 'string' && image.fileId) {
      const attachmentLike = {
        fileId: image.fileId,
        url: typeof image.url === 'string' ? image.url : '',
        name: typeof image.name === 'string' ? image.name : '',
        mimeType: typeof image.mimeType === 'string' ? image.mimeType : null,
        byteSize: Number.isFinite(image.byteSize) ? Number(image.byteSize) : null,
        sizeLabel: typeof image.sizeLabel === 'string' ? image.sizeLabel : null,
        checksum: typeof image.checksum === 'string' ? image.checksum : null,
        updatedAt: Number.isFinite(image.updatedAt) ? Number(image.updatedAt) : null,
        kind: 'user',
      };
      return buildImageViewFromAttachment(attachmentLike, image);
    }
    if (mode === 'url') {
      return {
        mode: 'url',
        url: typeof image.url === 'string' ? image.url : '',
        source: typeof image.source === 'string' ? image.source : 'url',
      };
    }
  }
  const legacyUrl = typeof props.imageUrl === 'string' ? props.imageUrl.trim() : '';
  if (legacyUrl) {
    return {
      mode: 'url',
      url: legacyUrl,
      source: 'url',
    };
  }
  return null;
}

function buildImageDataFromFile(file) {
  const attachment = mapFileToAttachment(file, { role: 'image', meta: { source: 'upload' } });
  if (!attachment) {
    return null;
  }
  const descriptor = {
    source: 'upload',
    updatedAt: Number.isFinite(attachment.updatedAt) ? Number(attachment.updatedAt) : nowSeconds(),
  };
  const view = buildImageViewFromAttachment(attachment, descriptor);
  if (!view) {
    return null;
  }
  return {
    mode: 'file',
    attachment,
    view,
    file,
  };
}

function buildImageViewFromAttachment(attachment, descriptor) {
  if (!attachment || typeof attachment !== 'object') {
    return null;
  }
  const view = {
    mode: 'file',
    fileId: attachment.fileId || '',
    url: typeof attachment.url === 'string' ? attachment.url : '',
    name: typeof attachment.name === 'string' ? attachment.name : '',
    mimeType: typeof attachment.mimeType === 'string' ? attachment.mimeType : null,
    byteSize: Number.isFinite(attachment.byteSize) ? Number(attachment.byteSize) : null,
    sizeLabel: typeof attachment.sizeLabel === 'string' ? attachment.sizeLabel : null,
    checksum: typeof attachment.checksum === 'string' ? attachment.checksum : null,
    source: typeof descriptor?.source === 'string'
      ? descriptor.source
      : (attachment.meta && typeof attachment.meta.source === 'string' ? attachment.meta.source : 'upload'),
    updatedAt: Number.isFinite(attachment.updatedAt)
      ? Number(attachment.updatedAt)
      : (Number.isFinite(descriptor?.updatedAt) ? Number(descriptor.updatedAt) : nowSeconds()),
  };
  return view;
}

// ————————————————————————————————————————————————————————————————
// État (footer) — rend uniquement le groupe « États » en bas de la fenêtre d'édition
// UI neutre: émet uniquement des intentions via ctx.commands
// ————————————————————————————————————————————————————————————————

async function organisationStateFooter(el, ctx) {
  const { item, commands, tags } = ctx || {};
  const nodeId = item?.id;
  if (!nodeId || !commands) return;

  const systemTagsMeta = Array.isArray(tags?.system) ? tags.system : [];
  const stateOptions = systemTagsMeta.filter(t => String(t?.key || '').startsWith('state/'));
  const allTags = Array.isArray(item?.tags) ? item.tags : [];
  const currentKey = firstKeyWithPrefix(allTags, 'state/');

  el.innerHTML = '';
  const wrap = document.createElement('div');
  wrap.className = 'org-state-footer';
  const inner = document.createElement('div');
  inner.className = 'org-state-footer__inner';
  try {
    inner.style.display = 'flex';
    inner.style.alignItems = 'center';
    inner.style.justifyContent = 'flex-end';
    inner.style.gap = '8px';
    inner.style.marginTop = '8px';
  } catch (_) {}
  wrap.appendChild(inner);
  el.appendChild(wrap);

  if (!Array.isArray(stateOptions) || stateOptions.length === 0) {
    inner.textContent = 'Aucun état disponible.';
    return;
  }

  // Compact grouped control
  const group = document.createElement('div');
  group.className = 'org-state-footer__group';
  Object.assign(group.style, {
    display: 'inline-flex', alignItems: 'center', gap: '6px',
    padding: '2px 6px', borderRadius: '10px', border: '1px solid rgba(100,116,139,.5)'
  });
  const prevBtn = document.createElement('button');
  prevBtn.type = 'button'; prevBtn.className = 'btn ghost btn--icon'; prevBtn.textContent = '◀'; prevBtn.title = 'Précédent';
  Object.assign(prevBtn.style, { padding: '0 4px', border: 'none', background: 'transparent', color: 'inherit', opacity: '0.8', fontWeight: '500' });
  const labelSpan = document.createElement('span');
  labelSpan.className = 'org-state-footer__label';
  Object.assign(labelSpan.style, { fontWeight: '600', fontSize: '12px', textAlign: 'center' });
  const nextBtn = document.createElement('button');
  nextBtn.type = 'button'; nextBtn.className = 'btn ghost btn--icon'; nextBtn.textContent = '▶'; nextBtn.title = 'Suivant';
  Object.assign(nextBtn.style, { padding: '0 4px', border: 'none', background: 'transparent', color: 'inherit', opacity: '0.8', fontWeight: '500' });
  const clearBtn = document.createElement('button');
  clearBtn.type = 'button'; clearBtn.className = 'btn ghost btn--icon'; clearBtn.textContent = '✕'; clearBtn.title = 'Effacer l\'état';
  Object.assign(clearBtn.style, { padding: '0 6px' });
  group.append(prevBtn, labelSpan, nextBtn);
  inner.append(group, clearBtn);

  function resolveMetaFor(key) {
    return stateOptions.find(o => String(o?.key || '') === String(key)) || null;
  }
  function labelFor(key) {
    const meta = resolveMetaFor(key);
    if (meta?.ui?.badge?.label) return String(meta.ui.badge.label);
    return labelFromKey(key);
  }
  function colorFor(key) {
    const meta = resolveMetaFor(key);
    const c = meta?.ui?.badge?.color || null;
    return typeof c === 'string' && c ? c : '#64748b';
  }

  let index = Math.max(0, stateOptions.findIndex(o => String(o?.key || '') === String(currentKey)));
  function renderBadge() {
    const key = stateOptions[index]?.key || '';
    const label = labelFor(key);
    const color = colorFor(key);
    labelSpan.textContent = label || '—';
    group.style.border = `1px solid ${color}`;
    group.style.background = `${color}1A`;
    labelSpan.style.color = color;
  }

  let busy = false;
  async function setStateByIndex(nextIndex) {
    if (!Number.isFinite(nextIndex) || busy) return;
    busy = true;
    index = (nextIndex + stateOptions.length) % stateOptions.length;
    const key = stateOptions[index]?.key;
    if (!key) { busy = false; return; }
    // No-op: do nothing if state unchanged
    if (key === currentKey) { renderBadge(); busy = false; return; }
    try { await commands.addTag(nodeId, { key, kind: 'system', sys: true }); }
    catch (e) { console.error('ORG_SET_STATE_FAILED', e); }
    renderBadge();
    busy = false;
  }
  async function clearState() {
    if (busy) return;
    const key = firstKeyWithPrefix(Array.isArray(ctx?.item?.tags) ? ctx.item.tags : [], 'state/');
    if (!key) return;
    busy = true;
    try { await commands.removeTag(nodeId, key); }
    catch (e) { console.error('ORG_CLEAR_STATE_FAILED', e); }
    busy = false;
  }

  prevBtn.addEventListener('click', () => setStateByIndex(index - 1));
  nextBtn.addEventListener('click', () => setStateByIndex(index + 1));
  clearBtn.addEventListener('click', () => clearState());

  renderBadge();
}

function renderRadioGroup(host, options, currentKey, onSelect) {
  if (!host) return;
  if (!Array.isArray(options) || options.length === 0) {
    host.innerHTML = '<p class="org-empty">Aucun état disponible.</p>';
    return;
  }
  const name = `org-${Math.random().toString(36).slice(2, 8)}`;
  options.forEach(opt => {
    const key = String(opt?.key || '');
    if (!key) return;
    const id = `${name}-${key.replace(/[^a-z0-9]+/gi,'-')}`;
    const icon = typeof opt?.ui?.badge?.icon === 'string' && opt.ui.badge.icon ? `${opt.ui.badge.icon} ` : '';
    const label = typeof opt?.ui?.badge?.label === 'string' && opt.ui.badge.label
      ? opt.ui.badge.label
      : labelFromKey(key);
    const row = document.createElement('label');
    row.className = 'org-option';
    row.innerHTML = `<input type="radio" name="${name}" id="${id}" ${currentKey === key ? 'checked' : ''}/> <span>${escapeHtml(icon + label)}</span>`;
    host.appendChild(row);
    const input = row.querySelector('input');
    input.addEventListener('change', () => { if (input.checked) onSelect(key); });
  });
}

function firstKeyWithPrefix(tags, prefix) {
  for (const t of (Array.isArray(tags) ? tags : [])) {
    const k = String(t?.key ?? t?.k ?? '');
    if (k.startsWith(prefix)) return k;
  }
  return null;
}

function allKeysWithPrefix(tags, prefix) {
  const list = [];
  for (const t of (Array.isArray(tags) ? tags : [])) {
    const k = String(t?.key ?? t?.k ?? '');
    if (k.startsWith(prefix)) list.push(k);
  }
  return list;
}

function labelFromKey(key) {
  if (typeof key !== 'string') return '';
  const raw = key.includes('/') ? key.split('/').slice(1).join('/') : key;
  const cleaned = raw.replace(/[-_]+/g, ' ').trim();
  if (!cleaned) return raw;
  return cleaned.split(' ').map(word => word ? word.charAt(0).toUpperCase() + word.slice(1) : '').join(' ');
}

function buildPreviewInfo(image) {
  const parts = [];
  if (image.mode === 'file' && image.name) {
    parts.push(image.name);
  }
  if (image.sizeLabel) {
    parts.push(image.sizeLabel);
  } else if (Number.isFinite(image.byteSize)) {
    parts.push(formatByteSize(image.byteSize));
  }
  if (image.mimeType) {
    parts.push(image.mimeType);
  }
  return parts.join(' · ');
}

function formatByteSize(bytes) {
  const units = ['octets', 'Ko', 'Mo', 'Go', 'To'];
  let value = Math.max(0, Number(bytes) || 0);
  let index = 0;
  while (value >= 1024 && index < units.length - 1) {
    value /= 1024;
    index += 1;
  }
  return index === 0 ? `${Math.round(value)} ${units[index]}` : `${value.toFixed(1)} ${units[index]}`;
}

function isValidUrl(value) {
  try {
    const url = new URL(value);
    return url.protocol === 'http:' || url.protocol === 'https:';
  } catch (_) {
    return false;
  }
}
