<?php

declare(strict_types=1);

namespace Skyboard\Application\Handlers;

use Skyboard\Application\Commands\AddTagV3Command;
use Skyboard\Application\Commands\Command;
use Skyboard\Application\Commands\ProposedChange;
use Skyboard\Domain\Boards\BoardPatch;

final class AddTagHandler implements CommandHandler
{
    public function handle(Command $command): ProposedChange
    {
        if ($command instanceof AddTagV3Command) {
            $operation = [
                'op' => 'tag.add',
                'scope' => 'item',
                'targetId' => $command->nodeId(),
                'nodeId' => $command->nodeId(),
                'tag' => $this->normalizeTag($command->tag()),
            ];
            $events = [[
                'name' => 'TagWillBeAdded',
                'boardId' => $command->boardId(),
                'actorId' => $command->actorId(),
                'context' => $operation,
            ]];
            $patch = new BoardPatch($command->boardId(), [$operation]);
            return new ProposedChange($patch, $events);
        }

        throw new \InvalidArgumentException('Unsupported command');
    }

    /**
     * @param array<string,mixed> $tag
     * @return array<string,mixed>
     */
    private function normalizeTag(array $tag): array
    {
        $key = (string) ($tag['key'] ?? ($tag['k'] ?? ''));
        if ($key === '') {
            throw new \InvalidArgumentException('La clé du tag est obligatoire.');
        }

        $kind = (string) ($tag['kind'] ?? '');
        $isSystem = $kind === 'system' || (isset($tag['sys']) && (bool) $tag['sys']);

        $normalized = [
            'key' => $key,
            'k' => $key,
            'kind' => $isSystem ? 'system' : 'user',
            'sys' => $isSystem,
            'v' => $tag['value'] ?? ($tag['v'] ?? $key),
        ];

        if (isset($tag['label'])) {
            $normalized['label'] = (string) $tag['label'];
        }
        if (isset($tag['icon'])) {
            $normalized['icon'] = (string) $tag['icon'];
        }
        if (array_key_exists('color', $tag)) {
            $normalized['color'] = $tag['color'] === null ? null : (string) $tag['color'];
        }
        if (isset($tag['themeColor']) && !isset($normalized['color'])) {
            $normalized['color'] = (string) $tag['themeColor'];
        }
        if (isset($tag['category'])) {
            $normalized['category'] = (string) $tag['category'];
        }

        return $normalized;
    }
}

