<?php

declare(strict_types=1);

namespace Skyboard\Application\Handlers;

use Skyboard\Application\Commands\Command;
use Skyboard\Application\Commands\CreateNodeCommand;
use Skyboard\Application\Commands\ProposedChange;
use Skyboard\Domain\Boards\BoardPatch;
use Skyboard\Domain\Shared\Identifiers;

final class CreateNodeHandler implements CommandHandler
{
    public function handle(Command $command): ProposedChange
    {
        if (!$command instanceof CreateNodeCommand) {
            throw new \InvalidArgumentException('Unsupported command');
        }
        $payload = $command->payload();
        $nodeId = isset($payload['nodeId']) && is_string($payload['nodeId']) && $payload['nodeId'] !== ''
            ? $payload['nodeId']
            : Identifiers::new();
        $shape = 'leaf';
        if (isset($payload['sys']) && is_array($payload['sys']) && isset($payload['sys']['shape'])) {
            $candidate = (string) $payload['sys']['shape'];
            if (in_array($candidate, ['container', 'leaf'], true)) {
                $shape = $candidate;
            }
        }
        $operation = [
            'op' => 'node.create',
            'parentId' => (string) ($payload['parentId'] ?? ''),
            'index' => isset($payload['index']) ? (int) $payload['index'] : null,
            'nodeId' => $nodeId,
            'title' => (string) ($payload['title'] ?? 'Nouveau nœud'),
            'description' => $payload['description'] ?? null,
            'content' => $payload['content'] ?? null,
            'tags' => $this->sanitizeTags($payload['tags'] ?? []),
            'sys' => ['shape' => $shape],
        ];
        if ($operation['index'] === null) { unset($operation['index']); }
        $events = [[
            'name' => 'NodeWillBeCreated',
            'boardId' => $command->boardId(),
            'actorId' => $command->actorId(),
            'context' => $operation,
        ]];
        $patch = new BoardPatch($command->boardId(), [$operation]);
        return new ProposedChange($patch, $events);
    }

    private function sanitizeTags(mixed $tags): array
    {
        if (!is_array($tags)) {
            return [];
        }
        $normalized = [];
        foreach ($tags as $tag) {
            if (!is_array($tag)) {
                continue;
            }
            $key = $tag['key'] ?? ($tag['k'] ?? null);
            $value = $tag['v'] ?? $tag['value'] ?? null;
            if ($key === 'type/list' || ($key === 'type' && $value === 'list')) {
                continue;
            }
            $normalized[] = $tag;
        }
        return $normalized;
    }
}


