<?php

declare(strict_types=1);

namespace Skyboard\Application\Handlers;

use Skyboard\Application\Commands\Command;
use Skyboard\Application\Commands\ProposedChange;
use Skyboard\Application\Commands\UpdateTagFilterCommand;
use Skyboard\Domain\Boards\BoardPatch;

final class UpdateTagFilterHandler implements CommandHandler
{
    public function handle(Command $command): ProposedChange
    {
        if (!$command instanceof UpdateTagFilterCommand) {
            throw new \InvalidArgumentException('Unsupported command');
        }

        $selected = $this->normalize($command->selected());
        $operation = [
            'op' => 'tagFilter.set',
            'selected' => $selected,
        ];

        $events = [[
            'name' => 'TagFilterWillBeUpdated',
            'boardId' => $command->boardId(),
            'actorId' => $command->actorId(),
            'context' => [
                'selected' => $selected,
            ],
        ]];

        $patch = new BoardPatch($command->boardId(), [$operation]);

        return new ProposedChange($patch, $events);
    }

    /**
     * @param list<string> $selected
     * @return list<string>
     */
    private function normalize(array $selected): array
    {
        $values = [];
        foreach ($selected as $value) {
            $tag = trim((string) $value);
            if ($tag === '') {
                continue;
            }
            $values[$tag] = true;
        }

        if (isset($values['__NONE__'])) {
            return ['__NONE__'];
        }

        return array_values(array_keys($values));
    }
}
