<?php

declare(strict_types=1);

namespace Skyboard\Application\NonBoard\Handlers;

use Skyboard\Application\NonBoard\NonBoardHandler;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class UserSetCategoryFrequencyOverrideHandler implements NonBoardHandler
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    public function handle(int $userId, array $payload, Request $request): Response
    {
        $categoryId = isset($payload['categoryId']) ? (int) $payload['categoryId'] : 0;
        $kind = isset($payload['kind']) ? strtoupper((string) $payload['kind']) : '';
        $paramRaw = $payload['param'] ?? null;
        if ($categoryId <= 0 || $kind === '') {
            return Response::error('INVALID_PAYLOAD', 'categoryId et kind requis.', [], 422);
        }
        $pdo = $this->connection->pdo();
        $cat = $pdo->prepare("SELECT dispatch_mode, allow_user_override FROM notification_categories WHERE id = :id");
        $cat->execute(['id' => $categoryId]);
        $row = $cat->fetch(\PDO::FETCH_ASSOC);
        if (!$row) return Response::error('NOT_FOUND', 'Catégorie introuvable.', [], 404);
        $dispatch = strtoupper((string) $row['dispatch_mode']);
        $allowOverride = !empty($row['allow_user_override']);
        if ($dispatch !== 'PERSONALIZED' || !$allowOverride) {
            return Response::error('INVALID_PAYLOAD', 'Override non autorisé pour cette catégorie.', [], 422);
        }

        if (!in_array($kind, ['EVERY_N_DAYS','WEEKLY','MONTHLY'], true)) {
            return Response::error('INVALID_PAYLOAD', 'kind invalide pour override.', [], 422);
        }
        $param = self::validateParam($kind, $paramRaw);
        if ($param === null) {
            return Response::error('INVALID_PAYLOAD', 'paramètre invalide pour la fréquence.', [], 422);
        }

        $stmt = $pdo->prepare('INSERT INTO user_subscriptions(user_id, category_id, subscribed, override_kind, override_param, created_at)
                               VALUES(:user, :cat, 1, :kind, :param, :now)
                               ON DUPLICATE KEY UPDATE override_kind = VALUES(override_kind), override_param = VALUES(override_param), subscribed = 1');
        $stmt->execute(['user' => $userId, 'cat' => $categoryId, 'kind' => $kind, 'param' => $param, 'now' => time()]);
        return Response::ok();
    }

    private static function validateParam(string $kind, mixed $paramRaw): ?int
    {
        $param = is_numeric($paramRaw) ? (int) $paramRaw : null;
        return match ($kind) {
            'EVERY_N_DAYS' => in_array($param, [1,2,3,4,5,6,7,14,21], true) ? $param : null,
            'WEEKLY' => ($param !== null && $param >= 1 && $param <= 7) ? $param : null,
            'MONTHLY' => ($param !== null && $param >= 1 && $param <= 28) ? $param : null,
            default => null,
        };
    }
}

