<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use PDO;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class ActionResolver
{
    /** @var array<string, array{kind:string,status:string,definition:array<string,mixed>,etag:string}> */
    private array $cache = [];

    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    /**
     * @param array<string,mixed> $refEntry { ref, version, params? }
     * @return array{ok:bool, action?:array<string,mixed>, versionEtag?:string, status?:string}
     */
    public function resolveOne(array $refEntry): array
    {
        $ref = (string) ($refEntry['ref'] ?? '');
        $version = (int) ($refEntry['version'] ?? 0);
        if ($ref === '' || $version <= 0) return ['ok' => false];
        $key = $ref . ':' . $version;
        if (!isset($this->cache[$key])) {
            $pdo = $this->connection->pdo();
            $stmt = $pdo->prepare('SELECT v.status, v.definition_json, v.etag, c.kind FROM action_catalog_versions v JOIN action_catalog c ON c.action_id = v.action_id WHERE v.action_id = :id AND v.version = :ver');
            $stmt->execute(['id' => $ref, 'ver' => $version]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$row) return ['ok' => false];
            $def = json_decode((string) ($row['definition_json'] ?? '{}'), true) ?: [];
            $this->cache[$key] = [
                'kind' => (string) ($row['kind'] ?? ''),
                'status' => (string) ($row['status'] ?? 'draft'),
                'definition' => is_array($def) ? $def : [],
                'etag' => (string) ($row['etag'] ?? ''),
            ];
        }
        $e = $this->cache[$key];
        $params = is_array($refEntry['params'] ?? null) ? $refEntry['params'] : [];
        $defaults = isset($e['definition']['defaultParams']) && is_array($e['definition']['defaultParams']) ? $e['definition']['defaultParams'] : [];
        $mergedParams = $this->mergeParams($defaults, $params);
        $label = $this->pickLabel($e['definition']['label'] ?? null);
        $auditTag = isset($e['definition']['auditTag']) ? (string) $e['definition']['auditTag'] : null;

        $action = [
            'id' => $refEntry['id'] ?? $ref,
            'ref' => $ref,
            'version' => $version,
            'kind' => $e['kind'],
            'label' => $label,
            'params' => $mergedParams,
        ];
        if ($auditTag) $action['auditTag'] = $auditTag;
        if (isset($e['definition']['schemaRef'])) $action['schemaRef'] = (string) $e['definition']['schemaRef'];
        if (isset($e['definition']['followUpType'])) $action['followUpType'] = (string) $e['definition']['followUpType'];

        return ['ok' => true, 'action' => $action, 'versionEtag' => $e['etag'], 'status' => $e['status']];
    }

    /**
     * @param array<string,mixed> $snapshotLight
     * @return array{manifestVersion:int, etag:string, actions:list<array<string,mixed>>}
     */
    public function resolveForNotification(array $refsJson, array $snapshotLight = []): array
    {
        $refs = is_array($refsJson['actions'] ?? null) ? $refsJson['actions'] : [];
        $snapPolicy = is_string($snapshotLight['policy'] ?? null) ? (string) $snapshotLight['policy'] : 'snapshot';
        $snapActions = is_array($snapshotLight['actions'] ?? null) ? $snapshotLight['actions'] : [];

        /** @var list<array<string,mixed>> $resolved */
        $resolved = [];
        $etagParts = [];
        foreach ($refs as $refEntry) {
            if (!is_array($refEntry)) continue;
            $res = $this->resolveOne($refEntry);
            if (!$res['ok']) continue;

            $status = (string) ($res['status'] ?? 'draft');
            $useSnapshot = false;
            $snap = $this->matchSnapshot($snapActions, (string) ($refEntry['ref'] ?? ''), (int) ($refEntry['version'] ?? 0));
            if ($snap && isset($snap['versionEtag']) && is_string($snap['versionEtag'])) {
                if ($snapPolicy === 'snapshot' && $status !== 'disabled') {
                    $useSnapshot = true;
                }
            }
            $action = $res['action'] ?? [];
            // Always prefer the author-provided local id from snapshot when available
            if ($snap && isset($snap['id']) && is_string($snap['id']) && $snap['id'] !== '') {
                $action['id'] = $snap['id'];
            }
            if ($useSnapshot) {
                if (isset($snap['label'])) $action['label'] = $snap['label'];
                if (isset($snap['params']) && is_array($snap['params'])) $action['params'] = $snap['params'];
            } else {
                if ($status === 'disabled') {
                    // kill-switch: do not expose action
                    continue;
                }
            }
            $resolved[] = $action;
            $etagParts[] = (string) ($res['versionEtag'] ?? '');
            $etagParts[] = $this->canonicalJson($action['params'] ?? []);
        }

        sort($etagParts);
        $combined = implode('|', $etagParts);
        $etag = 'sha256:' . hash('sha256', $combined);
        $manifestVersion = count($resolved) > 0 ? 1 : 0;
        return [
            'manifestVersion' => $manifestVersion,
            'etag' => $etag,
            'actions' => $resolved,
        ];
    }

    /**
     * @param list<array<string,mixed>> $snapActions
     */
    private function matchSnapshot(array $snapActions, string $ref, int $version): ?array
    {
        foreach ($snapActions as $a) {
            if (!is_array($a)) continue;
            if ((string) ($a['ref'] ?? '') === $ref && (int) ($a['version'] ?? 0) === $version) return $a;
        }
        return null;
    }

    /**
     * @param array<string,mixed> $a
     * @param array<string,mixed> $b
     * @return array<string,mixed>
     */
    private function mergeParams(array $a, array $b): array
    {
        $out = $a;
        foreach ($b as $k => $v) $out[(string) $k] = $v;
        return $out;
    }

    /**
     * @param mixed $labelDef
     * @return array<string,string>
     */
    private function pickLabel(mixed $labelDef): array
    {
        if (is_array($labelDef)) {
            $out = [];
            foreach ($labelDef as $k => $v) {
                if (is_string($k) && (is_string($v) || is_numeric($v))) {
                    $out[$k] = (string) $v;
                }
            }
            return $out;
        }
        return ['fr' => 'Action'];
    }

    /**
     * @param mixed $value
     */
    private function canonicalJson(mixed $value): string
    {
        if (is_array($value)) {
            if ($this->isAssoc($value)) {
                ksort($value);
                $out = [];
                foreach ($value as $k => $v) {
                    $out[$k] = json_decode($this->canonicalJson($v), true);
                }
                return json_encode($out, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
            }
            $out = array_map(fn($v) => json_decode($this->canonicalJson($v), true), $value);
            return json_encode($out, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
        }
        if (is_bool($value) || is_null($value) || is_numeric($value)) {
            return json_encode($value, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
        }
        return json_encode((string) $value, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
    }

    /**
     * @param array<mixed> $arr
     */
    private function isAssoc(array $arr): bool
    {
        if ($arr === []) return false;
        return array_keys($arr) !== range(0, count($arr) - 1);
    }
}
