<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use PDO;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class ActivityReadService
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    /**
     * @return array{items:list<array<string,mixed>>, now:int}
     */
    public function listForUser(int $userId, ?int $since = null, ?int $boardId = null, int $limit = 50): array
    {
        $pdo = $this->connection->pdo();
        $since = $since !== null && $since > 0 ? $since : 0;
        $limit = $limit > 0 && $limit <= 200 ? $limit : 50;

        $sql = 'SELECT e.id, e.board_id, e.actor_id, e.kind, e.payload_json, e.severity, e.created_at,
                       aus.seen_at, aus.archived_at, aus.deleted_at
                FROM activity_events e
                LEFT JOIN activity_user_state aus
                  ON aus.user_id = :user AND aus.event_id = e.id
                WHERE e.created_at >= :since AND (aus.deleted_at IS NULL) AND (aus.archived_at IS NULL)';

        $params = [
            'user' => $userId,
            'since' => $since,
        ];
        if ($boardId !== null && $boardId > 0) {
            $sql .= ' AND e.board_id = :boardId';
            $params['boardId'] = $boardId;
        }

        $sql .= ' ORDER BY e.created_at DESC, e.id DESC LIMIT :limitRows';

        $stmt = $pdo->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
        }
        $stmt->bindValue('limitRows', $limit, PDO::PARAM_INT);
        $stmt->execute();

        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $items = [];
        foreach ($rows as $row) {
            $payload = json_decode((string) ($row['payload_json'] ?? 'null'), true);
            if (!is_array($payload)) $payload = null;

            $items[] = [
                'id' => (int) $row['id'],
                'boardId' => (int) ($row['board_id'] ?? 0),
                'actorId' => isset($row['actor_id']) ? (int) $row['actor_id'] : null,
                'kind' => (string) ($row['kind'] ?? ''),
                'payload' => $payload,
                'severity' => (int) ($row['severity'] ?? 0),
                'createdAt' => (int) ($row['created_at'] ?? 0),
                'seen' => !empty($row['seen_at']),
                'archived' => !empty($row['archived_at']),
                'deleted' => !empty($row['deleted_at']),
            ];
        }

        return [
            'items' => $items,
            'now' => time(),
        ];
    }
}
