<?php

declare(strict_types=1);

namespace Skyboard\Application\Services\Admin;

use PDO;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class NotificationRichAdminService
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    /**
     * @return list<array<string,mixed>>
     */
    public function list(?int $categoryId = null): array
    {
        $pdo = $this->connection->pdo();
        if ($categoryId && $categoryId > 0) {
            $stmt = $pdo->prepare('SELECT id, category_id, title, emitter, weight, active, created_at FROM notifications_rich WHERE category_id = :cat ORDER BY active DESC, weight DESC, created_at DESC');
            $stmt->execute(['cat' => $categoryId]);
        } else {
            $stmt = $pdo->query('SELECT id, category_id, title, emitter, weight, active, created_at FROM notifications_rich ORDER BY active DESC, weight DESC, created_at DESC');
        }
        $rows = $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
        return array_map([$this, 'hydrate'], $rows ?: []);
    }

    /**
     * @return array<string,mixed>
     */
    public function create(array $input): array
    {
        $payload = $this->normalize($input, true);
        $pdo = $this->connection->pdo();

        // Ensure sequence_index is set (strict order). If absent, compute next.
        if (!isset($payload['sequence_index']) || !is_int($payload['sequence_index']) || $payload['sequence_index'] <= 0) {
            $stmtSeq = $pdo->prepare('SELECT COALESCE(MAX(sequence_index), 0) AS max_seq FROM notifications_rich WHERE category_id = :cat');
            $stmtSeq->execute(['cat' => $payload['category_id']]);
            $rowSeq = $stmtSeq->fetch(PDO::FETCH_ASSOC);
            $next = (int) ($rowSeq['max_seq'] ?? 0) + 1;
            $payload['sequence_index'] = max(1, $next);
        }

        $now = time();
        $stmt = $pdo->prepare(
            'INSERT INTO notifications_rich(category_id, title, emitter, weight, active, content_html, content_css, created_at, sequence_index, actions_ref_json, actions_snapshot_json)
             VALUES(:category_id, :title, :emitter, :weight, :active, :content_html, :content_css, :created_at, :sequence_index, :actions_ref_json, :actions_snapshot_json)'
        );
        // Normalize JSON fields: accept both array and already-encoded string
        $actionsRef = $payload['actions_ref_json'] ?? ['actions' => []];
        if (is_array($actionsRef)) $actionsRef = json_encode($actionsRef, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
        $actionsSnap = $payload['actions_snapshot_json'] ?? ['version' => 1, 'policy' => 'snapshot', 'actions' => []];
        if (is_array($actionsSnap)) $actionsSnap = json_encode($actionsSnap, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);

        $stmt->execute([
            'category_id' => $payload['category_id'],
            'title' => $payload['title'],
            'emitter' => $payload['emitter'],
            'weight' => $payload['weight'],
            'active' => $payload['active'],
            'content_html' => $payload['content_html'],
            'content_css' => $payload['content_css'],
            'created_at' => $now,
            'sequence_index' => $payload['sequence_index'],
            'actions_ref_json' => is_string($actionsRef) ? $actionsRef : json_encode(['actions' => []], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
            'actions_snapshot_json' => is_string($actionsSnap) ? $actionsSnap : json_encode(['version' => 1, 'policy' => 'snapshot', 'actions' => []], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
        ]);
        $id = (int) $this->connection->pdo()->lastInsertId();

        // IMMEDIATE materialization: write NUS rows for eligible users with available_at=created_at
        $this->maybeMaterializeImmediate($payload['category_id'], $id, $now);
        return $this->get($id);
    }

    /**
     * @return array<string,mixed>
     */
    public function update(int $id, array $input): array
    {
        $payload = $this->normalize($input, false);
        if ($payload === []) return $this->get($id);
        $sets = [];
        foreach ($payload as $field => $_) {
            $sets[] = sprintf('%s = :%s', $field, $field);
        }
        $sql = 'UPDATE notifications_rich SET ' . implode(', ', $sets) . ' WHERE id = :id';
        $payload['id'] = $id;
        $stmt = $this->connection->pdo()->prepare($sql);
        $stmt->execute($payload);
        return $this->get($id);
    }

    public function delete(int $id): void
    {
        $stmt = $this->connection->pdo()->prepare('DELETE FROM notifications_rich WHERE id = :id');
        $stmt->execute(['id' => $id]);
    }

    /**
     * @return array<string,mixed>
     */
    public function get(int $id): array
    {
        $stmt = $this->connection->pdo()->prepare('SELECT id, category_id, title, emitter, weight, active, content_html, content_css, created_at, sequence_index, actions_ref_json, actions_snapshot_json FROM notifications_rich WHERE id = :id');
        $stmt->execute(['id' => $id]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) throw new \RuntimeException('NOTIFICATION_NOT_FOUND');
        return $this->hydrate($row);
    }

    /**
     * @param array<string,mixed> $row
     * @return array<string,mixed>
     */
    private function hydrate(array $row): array
    {
        return [
            'id' => (int) $row['id'],
            'category_id' => (int) $row['category_id'],
            'title' => (string) $row['title'],
            'emitter' => $row['emitter'] ?? null,
            'weight' => (int) $row['weight'],
            'active' => (bool) $row['active'],
            'sequence_index' => isset($row['sequence_index']) ? (int) $row['sequence_index'] : null,
            'content_html' => (string) ($row['content_html'] ?? ''),
            'content_css' => (string) ($row['content_css'] ?? ''),
            
            'created_at' => (int) $row['created_at'],
            'actions_ref_json' => isset($row['actions_ref_json']) ? (is_string($row['actions_ref_json']) ? $row['actions_ref_json'] : json_encode($row['actions_ref_json'])) : null,
            'actions_snapshot_json' => isset($row['actions_snapshot_json']) ? (is_string($row['actions_snapshot_json']) ? $row['actions_snapshot_json'] : json_encode($row['actions_snapshot_json'])) : null,
        ];
    }

    /**
     * @return array<string,mixed>
     */
    private function normalize(array $input, bool $requireAll): array
    {
        $allowed = ['category_id','title','emitter','weight','active','content_html','content_css','sequence_index','actions_ref_json','actions_snapshot_json'];
        $out = [];
        foreach ($allowed as $field) {
            if (array_key_exists($field, $input)) {
                $out[$field] = $input[$field];
            } elseif ($requireAll) {
                // Champs optionnels en création: sequence_index (auto-assigné),
                // et actions_ref_json / actions_snapshot_json (valeurs par défaut si absents)
                if ($field === 'sequence_index' || $field === 'actions_ref_json' || $field === 'actions_snapshot_json') {
                    continue;
                }
                throw new \InvalidArgumentException(sprintf('FIELD_%s_REQUIRED', strtoupper($field)));
            }
        }
        if (isset($out['category_id'])) $out['category_id'] = (int) $out['category_id'];
        if (isset($out['title'])) {
            $out['title'] = trim((string) $out['title']);
            if ($out['title'] === '') throw new \InvalidArgumentException('FIELD_TITLE_REQUIRED');
        }
        if (isset($out['emitter'])) $out['emitter'] = $out['emitter'] === null ? null : trim((string) $out['emitter']);
        if (isset($out['weight'])) $out['weight'] = (int) $out['weight'];
        if (isset($out['active'])) $out['active'] = !empty($out['active']) ? 1 : 0;
        // layout removed
        foreach (['content_html','content_css'] as $k) {
            if (isset($out[$k])) $out[$k] = (string) $out[$k];
        }
        // Sanitize HTML to strip scripts and inline handlers
        if (isset($out['content_html'])) {
            $out['content_html'] = $this->sanitizeHtml($out['content_html']);
        }
        if (isset($out['sequence_index'])) {
            $si = (int) $out['sequence_index'];
            if ($si <= 0) unset($out['sequence_index']);
            else $out['sequence_index'] = $si;
        }
        // JSON fields normalization
        if (isset($out['actions_ref_json']) && is_array($out['actions_ref_json'])) {
            $out['actions_ref_json'] = json_encode($out['actions_ref_json'], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
        }
        if (isset($out['actions_snapshot_json']) && is_array($out['actions_snapshot_json'])) {
            $snap = $out['actions_snapshot_json'];
            if (!isset($snap['policy'])) $snap['policy'] = 'snapshot';
            if (!isset($snap['version'])) $snap['version'] = 1;
            $out['actions_snapshot_json'] = json_encode($snap, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
        }
        return $out;
    }

    private function sanitizeHtml(string $html): string
    {
        // Remove <script>...</script>
        $html = preg_replace('#<\s*script\b[^>]*>.*?<\s*/\s*script\s*>#is', '', $html) ?? $html;
        // Remove inline event handler attributes on*
        $html = preg_replace('/\son[\w-]+\s*=\s*("[^"]*"|\'[^\']*\'|[^\s>]+)/i', '', $html) ?? $html;
        return $html;
    }

    private function maybeMaterializeImmediate(int $categoryId, int $notificationId, int $createdAt): void
    {
        $pdo = $this->connection->pdo();
        $stmt = $pdo->prepare('SELECT audience_mode, dispatch_mode, frequency_kind FROM notification_categories WHERE id = :id');
        $stmt->execute(['id' => $categoryId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) return;
        $freq = strtoupper((string) ($row['frequency_kind'] ?? 'IMMEDIATE'));
        if ($freq !== 'IMMEDIATE') return; // only immediate is materialized at creation
        $aud = strtoupper((string) ($row['audience_mode'] ?? 'EVERYONE'));

        if ($aud === 'EVERYONE') {
            $sql = 'INSERT INTO notification_user_state (user_id, notification_id, available_at)
                    SELECT u.id, :nid, :ts
                    FROM users u
                    ON DUPLICATE KEY UPDATE available_at = LEAST(VALUES(available_at), notification_user_state.available_at)';
            $ins = $pdo->prepare($sql);
            $ins->execute(['nid' => $notificationId, 'ts' => $createdAt]);
            return;
        }

        // SUBSCRIBERS only
        $sql = 'INSERT INTO notification_user_state (user_id, notification_id, available_at)
                SELECT us.user_id, :nid, :ts
                FROM user_subscriptions us
                WHERE us.category_id = :cat AND us.subscribed = 1
                ON DUPLICATE KEY UPDATE available_at = LEAST(VALUES(available_at), notification_user_state.available_at)';
        $ins = $pdo->prepare($sql);
        $ins->execute(['nid' => $notificationId, 'ts' => $createdAt, 'cat' => $categoryId]);
    }
}
