<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use InvalidArgumentException;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;
use Throwable;

final class BoardRulesService
{
    public function __construct(
        private readonly DatabaseConnection $connection,
        private readonly BoardCatalog $catalog
    ) {
    }

    /**
     * @return list<array<string,mixed>>
     */
    public function export(int $boardId, int $userId): array
    {
        if (!$this->catalog->exists($boardId, $userId)) {
            throw new InvalidArgumentException('BOARD_NOT_FOUND');
        }

        $stmt = $this->connection->pdo()->prepare('SELECT id, scope, priority, triggers_json, conditions_json, actions_json, on_veto_json, created_at FROM board_rules WHERE board_id = :board ORDER BY priority DESC, id ASC');
        $stmt->execute(['board' => $boardId]);

        $rules = [];
        while ($row = $stmt->fetch()) {
            $rules[] = [
                'id' => (int) $row['id'],
                'scope' => $row['scope'],
                'priority' => (int) $row['priority'],
                'triggers' => json_decode((string) $row['triggers_json'], true) ?: [],
                'conditions' => json_decode((string) $row['conditions_json'], true) ?: [],
                'actions' => json_decode((string) $row['actions_json'], true) ?: [],
                'onVeto' => $row['on_veto_json'] ? json_decode((string) $row['on_veto_json'], true) : null,
                'createdAt' => $row['created_at'] ? (int) $row['created_at'] : null,
            ];
        }

        return $rules;
    }

    /**
     * @param list<array<string,mixed>> $rules
     */
    public function replace(int $boardId, int $userId, array $rules): void
    {
        if (!$this->catalog->exists($boardId, $userId)) {
            throw new InvalidArgumentException('BOARD_NOT_FOUND');
        }

        $normalized = $this->validate($rules);

        $pdo = $this->connection->pdo();
        $pdo->beginTransaction();
        try {
            $delete = $pdo->prepare('DELETE FROM board_rules WHERE board_id = :board');
            $delete->execute(['board' => $boardId]);

            $insert = $pdo->prepare('INSERT INTO board_rules(board_id, scope, priority, triggers_json, conditions_json, actions_json, on_veto_json, created_at) VALUES(:board, :scope, :priority, :triggers, :conditions, :actions, :onVeto, :createdAt)');
            $now = time();
            foreach ($normalized as $rule) {
                $insert->execute([
                    'board' => $boardId,
                    'scope' => $rule['scope'],
                    'priority' => $rule['priority'],
                    'triggers' => json_encode($rule['triggers'], JSON_UNESCAPED_SLASHES | JSON_THROW_ON_ERROR),
                    'conditions' => json_encode($rule['conditions'], JSON_UNESCAPED_SLASHES | JSON_THROW_ON_ERROR),
                    'actions' => json_encode($rule['actions'], JSON_UNESCAPED_SLASHES | JSON_THROW_ON_ERROR),
                    'onVeto' => $rule['onVeto'] === null ? null : json_encode($rule['onVeto'], JSON_UNESCAPED_SLASHES | JSON_THROW_ON_ERROR),
                    'createdAt' => $now,
                ]);
            }
            $pdo->commit();
        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }

    /**
     * @param list<array<string,mixed>> $rules
     * @return list<array{scope:string,priority:int,triggers:array,conditions:array,actions:array,onVeto:array<string,mixed>|null}>
     */
    private function validate(array $rules): array
    {
        $normalized = [];
        foreach ($rules as $index => $rule) {
            if (!is_array($rule)) {
                throw new InvalidArgumentException('INVALID_RULE_AT_' . $index);
            }
            $scope = (string) ($rule['scope'] ?? 'board');
            if ($scope === '') {
                throw new InvalidArgumentException('RULE_SCOPE_REQUIRED_AT_' . $index);
            }
            $priority = (int) ($rule['priority'] ?? 0);
            $triggers = $rule['triggers'] ?? [];
            $conditions = $rule['conditions'] ?? [];
            $actions = $rule['actions'] ?? [];

            if (!is_array($triggers) || !is_array($conditions) || !is_array($actions)) {
                throw new InvalidArgumentException('RULE_SHAPE_INVALID_AT_' . $index);
            }

            $onVeto = $rule['onVeto'] ?? null;
            if ($onVeto !== null && !is_array($onVeto)) {
                throw new InvalidArgumentException('RULE_ON_VETO_INVALID_AT_' . $index);
            }

            $normalized[] = [
                'scope' => $scope,
                'priority' => $priority,
                'triggers' => array_values($triggers),
                'conditions' => array_values($conditions),
                'actions' => array_values($actions),
                'onVeto' => $onVeto,
            ];
        }

        return $normalized;
    }
}
