<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use PDO;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;
use Skyboard\Infrastructure\Mail\Mailer;
use function Skyboard\sb_env;

final class EmailVerificationService
{
    public function __construct(
        private readonly DatabaseConnection $connection,
        private readonly Mailer $mailer,
    )
    {
    }

    /**
     * @return array{token:string,expiresAt:int,sent:bool}
     */
    public function request(int $userId): array
    {
        $token = bin2hex(random_bytes(8));
        $expiresAt = time() + 86400;

        $stmt = $this->connection->pdo()->prepare('UPDATE users SET verify_token = :token, verify_expires_at = :expiresAt WHERE id = :id');
        $stmt->execute([
            'token' => $token,
            'expiresAt' => $expiresAt,
            'id' => $userId,
        ]);

        // Send email (templates simples TXT/HTML). Ne pas logguer le contenu.
        $userStmt = $this->connection->pdo()->prepare('SELECT email FROM users WHERE id = :id');
        $userStmt->execute(['id' => $userId]);
        $email = (string) ($userStmt->fetchColumn() ?: '');

        $appUrl = rtrim((string) (sb_env('APP_URL', '') ?? ''), '/');
        $from = (string) (sb_env('MAIL_FROM', 'noreply@localhost') ?? 'noreply@localhost');
        $subject = 'Votre code de vérification';
        $text = "Bonjour,\n\nVoici votre code de vérification: {$token}\nIl expire dans 24h.\n\n— Skyboard";
        $html = '<p>Bonjour,</p><p>Voici votre code de vérification: <strong>' . htmlspecialchars($token, ENT_QUOTES) . '</strong><br/>Il expire dans 24h.</p><p>— Skyboard</p>';
        $headers = ['From' => $from, 'X-Skyboard-Mail' => 'verification'];
        $sent = false;
        if ($email !== '') {
            try {
                $sent = $this->mailer->send($email, $subject, $text, $html, $headers);
            } catch (\Throwable $e) {
                $sent = false;
            }
        }

        return [
            'token' => $token,
            'expiresAt' => $expiresAt,
            'sent' => $sent,
        ];
    }

    public function verify(int $userId, string $token): bool
    {
        $stmt = $this->connection->pdo()->prepare('SELECT verify_token, verify_expires_at FROM users WHERE id = :id');
        $stmt->execute(['id' => $userId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) {
            return false;
        }

        $expected = (string) ($row['verify_token'] ?? '');
        $expiresAt = isset($row['verify_expires_at']) ? (int) $row['verify_expires_at'] : 0;

        if ($expected === '' || !hash_equals($expected, $token) || $expiresAt < time()) {
            return false;
        }

        $update = $this->connection->pdo()->prepare('UPDATE users SET email_verified_at = :verifiedAt, verify_token = NULL, verify_expires_at = NULL WHERE id = :id');
        $update->execute([
            'verifiedAt' => time(),
            'id' => $userId,
        ]);

        return true;
    }
}
