<?php

declare(strict_types=1);

namespace Skyboard\Application\Services\Scheduler;

use DateTimeImmutable;
use DateTimeZone;

final class Slot
{
    /**
     * Compute next slot timestamp (UTC) for the given scheduling parameters.
     * - kind: IMMEDIATE | EVERY_N_DAYS | WEEKLY | MONTHLY
     * - param: n (for EVERY_N_DAYS), 1..7 for WEEKLY (1=Mon..7=Sun), 1..28 for MONTHLY
     * - anchorUtc: optional UTC timestamp used to align EVERY_N_DAYS cadence
     * - tz: IANA timezone for calendar logic (default: Europe/Paris)
     */
    public static function nextSlot(int $nowUtc, string $kind, ?int $param, ?int $anchorUtc = null, string $tz = 'Europe/Paris'): ?int
    {
        $kind = strtoupper($kind);
        if ($kind === 'IMMEDIATE') {
            return null;
        }

        $tzLocal = new DateTimeZone($tz);
        $tzUtc = new DateTimeZone('UTC');
        $nowLocal = (new DateTimeImmutable('@' . $nowUtc))->setTimezone($tzLocal);
        $midnightNow = $nowLocal->setTime(0, 0, 0);

        if ($kind === 'EVERY_N_DAYS') {
            $n = max(1, (int) ($param ?? 1));
            $anchorLocal = $anchorUtc
                ? (new DateTimeImmutable('@' . $anchorUtc))->setTimezone($tzLocal)->setTime(0, 0, 0)
                : $midnightNow;
            $slot = $anchorLocal;
            if ($slot < $nowLocal) {
                // Iterate by n days until slot >= now (DST-safe)
                while ($slot < $nowLocal) {
                    $slot = $slot->modify('+' . $n . ' day');
                }
            }
            return $slot->setTimezone($tzUtc)->getTimestamp();
        }

        if ($kind === 'WEEKLY') {
            $dowTarget = max(1, min(7, (int) ($param ?? 1))); // 1=Mon..7=Sun
            $base = $midnightNow;
            $nowIsMidnight = $nowLocal->format('H:i:s') === '00:00:00';
            $dow = (int) $base->format('N');
            if ($dow === $dowTarget) {
                $slot = $nowIsMidnight ? $base : $base->modify('+7 day');
            } else {
                $delta = ($dowTarget - $dow + 7) % 7;
                if ($delta === 0) $delta = 7;
                $slot = $base->modify('+' . $delta . ' day');
            }
            return $slot->setTimezone($tzUtc)->getTimestamp();
        }

        if ($kind === 'MONTHLY') {
            $day = max(1, min(28, (int) ($param ?? 1)));
            $y = (int) $nowLocal->format('Y');
            $m = (int) $nowLocal->format('m');
            $slot = (new DateTimeImmutable(sprintf('%04d-%02d-%02d 00:00:00', $y, $m, $day), $tzLocal));
            if ($slot < $nowLocal) {
                $slot = $slot->modify('+1 month');
            }
            return $slot->setTimezone($tzUtc)->getTimestamp();
        }

        return null;
    }
}

