<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use Skyboard\Domain\Boards\BoardPatch;
use Skyboard\Domain\Boards\BoardState;
use Skyboard\Domain\Boards\BoardTraversal;

use function is_array;

final class StructureTagReactor
{
    private const STRUCTURAL_TAG_KEY = 'type/list';

    public function __construct(private readonly bool $enabled)
    {
    }

    public function augment(?BoardState $board, BoardPatch $patch): BoardPatch
    {
        if (!$this->enabled || $board === null) {
            return $patch;
        }

        $state = $board->toArray();
        $operations = $patch->operations();
        $augmented = $operations;
        $mutated = false;

        foreach ($operations as $operation) {
            $opName = (string) ($operation['op'] ?? '');
            if ($opName === 'tag.add') {
                $nodeId = $this->resolveNodeId($operation);
                if ($nodeId === null) {
                    continue;
                }
                if (!$this->isSystemListTag($operation['tag'] ?? null)) {
                    continue;
                }
                $node = BoardTraversal::findNode($state, $nodeId);
                if ($node === null) {
                    continue;
                }
                if (($node['sys']['shape'] ?? null) === 'container') {
                    continue;
                }
                if ($this->hasScheduledShape($augmented, $nodeId, 'container')) {
                    continue;
                }
                $augmented[] = [
                    'op' => 'node.update',
                    'nodeId' => $nodeId,
                    'sys' => ['shape' => 'container'],
                ];
                $mutated = true;
                continue;
            }

            if ($opName === 'tag.remove') {
                $nodeId = $this->resolveNodeId($operation);
                if ($nodeId === null) {
                    continue;
                }
                if (!$this->removesSystemList($state, $nodeId, $operation)) {
                    continue;
                }
                $node = BoardTraversal::findNode($state, $nodeId);
                if ($node === null) {
                    continue;
                }
                if (($node['sys']['shape'] ?? null) === 'leaf') {
                    continue;
                }
                if ($this->nodeHasChildren($state, $nodeId)) {
                    throw new StructureTagReactionRefused(
                        'list.not_empty',
                        'Impossible de retirer le type liste tant que des éléments sont présents.',
                        ['nodeId' => $nodeId]
                    );
                }
                if ($this->hasScheduledShape($augmented, $nodeId, 'leaf')) {
                    continue;
                }
                $augmented[] = [
                    'op' => 'node.update',
                    'nodeId' => $nodeId,
                    'sys' => ['shape' => 'leaf'],
                ];
                $mutated = true;
            }
        }

        if (!$mutated) {
            return $patch;
        }

        return $patch->withOperations(array_values($augmented));
    }

    private function resolveNodeId(array $operation): ?string
    {
        $candidate = $operation['nodeId'] ?? ($operation['targetId'] ?? null);
        if (!is_string($candidate) || $candidate === '') {
            return null;
        }
        return $candidate;
    }

    private function isSystemListTag(mixed $tag): bool
    {
        if (!is_array($tag)) {
            return false;
        }
        $key = (string) ($tag['key'] ?? ($tag['k'] ?? ''));
        if ($key === '') {
            return false;
        }
        $kind = (string) ($tag['kind'] ?? '');
        $isSystem = $kind === 'system' || (isset($tag['sys']) && (bool) $tag['sys']);
        if (!$isSystem) {
            return false;
        }
        return $key === self::STRUCTURAL_TAG_KEY;
    }

    private function removesSystemList(array $state, string $nodeId, array $operation): bool
    {
        $key = (string) ($operation['key'] ?? '');
        if ($key === '') {
            $tag = $operation['tag'] ?? null;
            if (is_array($tag)) {
                $key = (string) ($tag['key'] ?? ($tag['k'] ?? ''));
            }
        }
        if ($key !== self::STRUCTURAL_TAG_KEY) {
            return false;
        }

        $node = BoardTraversal::findNode($state, $nodeId);
        if ($node === null) {
            return false;
        }
        $tags = $node['tags'] ?? [];
        if (!is_array($tags)) {
            return false;
        }
        foreach ($tags as $tag) {
            if ($this->isSystemListTag($tag)) {
                return true;
            }
        }
        return false;
    }

    private function hasScheduledShape(array $operations, string $nodeId, string $shape): bool
    {
        foreach ($operations as $operation) {
            if (($operation['op'] ?? null) !== 'node.update') {
                continue;
            }
            if ((string) ($operation['nodeId'] ?? '') !== $nodeId) {
                continue;
            }
            $sys = $operation['sys'] ?? null;
            if (!is_array($sys)) {
                continue;
            }
            $targetShape = (string) ($sys['shape'] ?? '');
            if ($targetShape === $shape) {
                return true;
            }
        }
        return false;
    }

    private function nodeHasChildren(array $state, string $nodeId): bool
    {
        $node = BoardTraversal::findNode($state, $nodeId);
        if ($node === null) {
            return false;
        }
        $children = $node['children'] ?? [];
        return is_array($children) && $children !== [];
    }
}
