<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use Skyboard\Application\Commands\AddTagV3Command;
use Skyboard\Application\Commands\ProposedChange;
use Skyboard\Domain\Boards\BoardPatch;
use Skyboard\Domain\Boards\BoardState;
use Skyboard\Domain\Boards\BoardTraversal;
use Skyboard\Domain\Rules\RulesEngine as DomainRulesEngine;
use Skyboard\Domain\Rules\RulesEngineResult;
use Skyboard\Infrastructure\Packs\SystemTagRegistry;

final class TagSuggestionService
{
    public function __construct(
        private readonly SystemTagRegistry $systemTagRegistry,
        private readonly DomainRulesEngine $rules
    ) {
    }

    /**
     * @return list<array{kind:string,key:string,label:string,icon?:string,color?:string|null}>
     */
    public function allowedTags(BoardState $boardState, string $boardId, string $actorId, string $scope, string $targetId): array
    {
        $board = $boardState->toArray();
        $existing = BoardTraversal::tagsForTarget($board, $scope, $targetId);
        $existingKeys = [];
        foreach ($existing as $tag) {
            $existingKeys[$this->hash((string) ($tag['kind'] ?? ''), (string) ($tag['key'] ?? ''))] = true;
        }

        $metaMap = $this->collectSystemTags();
        foreach (BoardTraversal::collectTags($board) as $tag) {
            $metaMap[$this->hash((string) ($tag['kind'] ?? ''), (string) ($tag['key'] ?? ''))] = [
                'kind' => (string) ($tag['kind'] ?? 'user'),
                'key' => (string) ($tag['key'] ?? ''),
                'label' => (string) ($tag['label'] ?? $tag['key'] ?? ''),
                'icon' => $tag['icon'] ?? '',
                'color' => $tag['color'] ?? null,
            ];
        }

        $candidates = [];
        foreach ($metaMap as $hash => $tag) {
            if (isset($existingKeys[$hash])) {
                continue;
            }
            $candidates[$hash] = $tag;
        }

        $approved = $this->rules->filter(
            $boardState,
            $candidates,
            function (array $candidate) use ($boardId, $actorId, $scope, $targetId): array {
                $operation = [
                    'op' => 'tag.add',
                    'scope' => 'item',
                    'targetId' => $targetId,
                    'nodeId' => $targetId,
                    'tag' => [
                        'kind' => $candidate['kind'],
                        'key' => $candidate['key'],
                    ],
                ];
                $command = new AddTagV3Command($boardId, $actorId, $targetId, $operation['tag']);
                $change = new ProposedChange(
                    new BoardPatch($boardId, [$operation]),
                    [[
                        'name' => 'TagWillBeAdded',
                        'boardId' => $boardId,
                        'actorId' => $actorId,
                        'context' => $operation,
                    ]]
                );

                return [
                    'command' => $command,
                    'change' => $change,
                ];
            },
            function (array $candidate, RulesEngineResult $result) use (&$metaMap, $targetId) {
                $finalTag = $this->extractFinalTag($result, $targetId) ?? [
                    'kind' => $candidate['kind'],
                    'key' => $candidate['key'],
                ];

                return $this->hydrateTag($metaMap, $finalTag, $candidate);
            }
        );

        usort($approved, static fn(array $a, array $b): int => strcmp($a['label'], $b['label']));

        return $approved;
    }

    /**
     * @return array<string,array{kind:string,key:string,label:string,icon?:string,color?:string|null}>
     */
    private function collectSystemTags(): array
    {
        $tags = [];
        foreach ($this->systemTagRegistry->getAll() as $tagDef) {
            $ui = $tagDef->ui;
            $badge = $ui['badge'] ?? [];

            $tags[$this->hash('system', $tagDef->shortKey)] = [
                'kind' => 'system',
                'key' => $tagDef->shortKey,
                'label' => $badge['label'] ?? $tagDef->shortKey,
                'icon' => $badge['icon'] ?? '',
                'color' => $badge['color'] ?? null,
                'category' => $tagDef->category,
            ];
        }
        return $tags;
    }

    /**
     * @return array{kind:string,key:string}|null
     */
    private function extractFinalTag(RulesEngineResult $result, string $targetId): ?array
    {
        foreach ($result->patch()->operations() as $operation) {
            if (($operation['op'] ?? null) !== 'tag.add') {
                continue;
            }
            if ((string) ($operation['targetId'] ?? '') !== $targetId) {
                continue;
            }
            $tag = $operation['tag'] ?? null;
            if (!is_array($tag)) {
                continue;
            }
            return [
                'kind' => (string) ($tag['kind'] ?? 'user'),
                'key' => (string) ($tag['key'] ?? ''),
                'label' => (string) ($tag['label'] ?? $tag['key'] ?? ''),
                'icon' => $tag['icon'] ?? '',
                'color' => $tag['color'] ?? null,
            ];
        }

        return null;
    }

    private function hydrateTag(array &$metaMap, array $tag, array $fallback): array
    {
        $key = $this->hash((string) ($tag['kind'] ?? ''), (string) ($tag['key'] ?? ''));

        if (!isset($metaMap[$key])) {
            $metaMap[$key] = [
                'kind' => (string) ($tag['kind'] ?? 'user'),
                'key' => (string) ($tag['key'] ?? ''),
                'label' => (string) ($tag['label'] ?? $fallback['label'] ?? $tag['key'] ?? ''),
                'icon' => $tag['icon'] ?? $fallback['icon'] ?? '',
                'color' => $tag['color'] ?? $fallback['color'] ?? null,
                'category' => $tag['category'] ?? $fallback['category'] ?? null,
            ];
        } else {
            $metaMap[$key]['label'] = (string) ($metaMap[$key]['label'] ?? $tag['label'] ?? $fallback['label'] ?? $tag['key'] ?? '');
            $metaMap[$key]['icon'] = $metaMap[$key]['icon'] ?? $tag['icon'] ?? $fallback['icon'] ?? '';
            $metaMap[$key]['color'] = $metaMap[$key]['color'] ?? $tag['color'] ?? $fallback['color'] ?? null;
            if (!isset($metaMap[$key]['category'])) {
                $metaMap[$key]['category'] = $tag['category'] ?? $fallback['category'] ?? null;
            }
        }

        return $metaMap[$key];
    }

    private function hash(string $kind, string $key): string
    {
        return $kind . ':' . $key;
    }
}

