<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use Skyboard\Domain\UserFiles\UserFile;
use Skyboard\Domain\UserFiles\UserFileFolder;

use function Skyboard\storage_path;

final class UserFileService
{
    private const DEFAULT_MAX_FILE_SIZE = 10_485_760; // 10 MiB

    private UserFileRepository $repository;
    private ?UserFileFolderRepository $folderRepository = null;
    private int $maxFileSize;
    private string $storageRoot;

    public function __construct(UserFileRepository $repository, ?int $maxFileSize = null, ?UserFileFolderRepository $folderRepository = null)
    {
        $this->repository = $repository;
        $this->maxFileSize = $maxFileSize && $maxFileSize > 0 ? $maxFileSize : self::DEFAULT_MAX_FILE_SIZE;
        $this->storageRoot = rtrim(storage_path('user-files'), '/');
        $this->folderRepository = $folderRepository;
        if (!is_dir($this->storageRoot)) {
            @mkdir($this->storageRoot, 0775, true);
        }
    }

    /**
     * @return list<array<string,mixed>>
     */
    public function listForUser(int $userId): array
    {
        $files = $this->repository->listForUser($userId);
        return array_map(fn(UserFile $file) => $this->present($file), $files);
    }

    /**
     * @param array{name:string,type:string,tmp_name:string,error:int,size:int} $spec
     * @return array<string,mixed>
     */
    public function uploadFromSpec(int $userId, array $spec): array
    {
        $error = (int) ($spec['error'] ?? 0);
        $uploadOk = \defined('UPLOAD_ERR_OK') ? UPLOAD_ERR_OK : 0;
        if ($error !== $uploadOk) {
            throw new UserFileException('UPLOAD_FAILED', $this->resolveUploadErrorMessage($error), $this->resolveUploadErrorStatus($error));
        }

        $tmpName = (string) ($spec['tmp_name'] ?? '');
        if ($tmpName === '' || !file_exists($tmpName)) {
            throw new UserFileException('UPLOAD_TMP_MISSING', 'Fichier temporaire introuvable.', 400);
        }

        if ($this->shouldAssertUploadedFile() && !is_uploaded_file($tmpName)) {
            throw new UserFileException('UPLOAD_INVALID_SOURCE', 'La source du fichier est invalide.', 400);
        }

        $size = max(0, (int) ($spec['size'] ?? 0));
        if ($size > $this->maxFileSize) {
            throw new UserFileException(
                'FILE_TOO_LARGE',
                sprintf('Le fichier dépasse la taille maximale autorisée (%s).', $this->formatByteSize($this->maxFileSize)),
                413
            );
        }

        $originalName = $this->sanitizeOriginalName((string) ($spec['name'] ?? ''));
        $mimeType = $this->detectMimeType($tmpName, $spec['type'] ?? null);
        $checksum = $this->computeChecksum($tmpName);
        $publicId = $this->generatePublicId();
        $extension = $this->guessExtension($originalName, $mimeType);
        $storedName = $extension !== '' ? $publicId . '.' . $extension : $publicId;

        $directory = $this->directoryFor($userId);
        $this->ensureDirectory($directory);
        $destination = $directory . DIRECTORY_SEPARATOR . $storedName;

        if (!$this->moveUploadedFile($tmpName, $destination)) {
            throw new UserFileException('FILE_PERSIST_FAILED', 'Impossible de sauvegarder le fichier.', 500);
        }
        @chmod($destination, 0664);

        clearstatcache(true, $destination);
        $actualSize = filesize($destination);
        if (is_int($actualSize) && $actualSize >= 0) {
            $size = $actualSize;
        }

        $timestamp = time();
        $file = UserFile::createNew(
            $userId,
            $publicId,
            $storedName,
            $originalName,
            $mimeType,
            $size,
            $checksum,
            $timestamp
        );
        $file = $this->repository->insert($file);

        return $this->present($file);
    }

    public function delete(int $userId, string $publicId): void
    {
        $file = $this->repository->findByPublicId($userId, $publicId);
        if ($file === null) {
            throw new UserFileException('FILE_NOT_FOUND', 'Fichier introuvable.', 404);
        }

        $path = $this->pathFor($file);
        if (is_file($path)) {
            @unlink($path);
        }

        $this->repository->delete($file);
    }

    /**
     * @return array<string,mixed>
     */
    public function rename(int $userId, string $publicId, string $newName): array
    {
        $file = $this->repository->findByPublicId($userId, $publicId);
        if ($file === null) {
            throw new UserFileException('FILE_NOT_FOUND', 'Fichier introuvable.', 404);
        }
        $sanitized = $this->sanitizeOriginalName($newName);
        if ($sanitized === '') {
            throw new UserFileException('INVALID_FILE_NAME', 'Nom de fichier invalide.', 422);
        }
        $updated = $file->withOriginalName($sanitized, time());
        $this->repository->update($updated);
        return $this->present($updated);
    }

    /**
     * @return list<array<string,mixed>>
     */
    public function listFolders(int $userId): array
    {
        $repo = $this->requireFolderRepository();
        $folders = $repo->listForUser($userId);
        // Build map internal id -> public id for parent resolution
        $idToPublic = [];
        foreach ($folders as $f) {
            if ($f->id() !== null) {
                $idToPublic[$f->id()] = $f->publicId();
            }
        }
        $out = [];
        foreach ($folders as $f) {
            $parentPublic = null;
            $pid = $f->parentId();
            if ($pid !== null && isset($idToPublic[$pid])) {
                $parentPublic = $idToPublic[$pid];
            }
            $out[] = $this->presentFolder($f, $parentPublic);
        }
        return $out;
    }

    /**
     * @return array<string,mixed>
     */
    public function createFolder(int $userId, string $name, ?string $parentPublicId = null): array
    {
        $repo = $this->requireFolderRepository();
        $sanitized = $this->sanitizeFolderName($name);
        if ($sanitized === '') {
            throw new UserFileException('INVALID_FOLDER_NAME', 'Nom de dossier invalide.', 422);
        }
        $parentId = null;
        $parentPublic = null;
        if ($parentPublicId !== null && $parentPublicId !== '') {
            $parent = $repo->findByPublicId($userId, $parentPublicId);
            if ($parent === null) {
                throw new UserFileException('FOLDER_NOT_FOUND', 'Dossier parent introuvable.', 404);
            }
            $parentId = $parent->id();
            $parentPublic = $parent->publicId();
        }
        $timestamp = time();
        $folder = UserFileFolder::createNew($userId, $this->generatePublicId(), $sanitized, $parentId, $timestamp);
        $folder = $repo->insert($folder);
        return $this->presentFolder($folder, $parentPublic);
    }

    /**
     * Déplacer un fichier dans un dossier (ou à la racine si $destFolderPublicId est null/"")
     * @return array<string,mixed>
     */
    public function moveFile(int $userId, string $filePublicId, ?string $destFolderPublicId): array
    {
        $file = $this->repository->findByPublicId($userId, $filePublicId);
        if ($file === null) {
            throw new UserFileException('FILE_NOT_FOUND', 'Fichier introuvable.', 404);
        }
        $folderId = null;
        $folderPublicId = null;
        if ($destFolderPublicId !== null && $destFolderPublicId !== '') {
            $repo = $this->requireFolderRepository();
            $folder = $repo->findByPublicId($userId, $destFolderPublicId);
            if ($folder === null) {
                throw new UserFileException('FOLDER_NOT_FOUND', 'Dossier de destination introuvable.', 404);
            }
            $folderId = $folder->id();
            $folderPublicId = $folder->publicId();
        }
        $updated = $file->withFolderRef($folderId, $folderPublicId, time());
        $this->repository->update($updated);
        return $this->present($updated);
    }

    /**
     * @return array<string,mixed>
     */
    public function renameFolder(int $userId, string $folderPublicId, string $name): array
    {
        $repo = $this->requireFolderRepository();
        $folder = $repo->findByPublicId($userId, $folderPublicId);
        if ($folder === null) {
            throw new UserFileException('FOLDER_NOT_FOUND', 'Dossier introuvable.', 404);
        }
        $sanitized = $this->sanitizeFolderName($name);
        if ($sanitized === '') {
            throw new UserFileException('INVALID_FOLDER_NAME', 'Nom de dossier invalide.', 422);
        }
        $updated = $folder->withName($sanitized, time());
        $repo->update($updated);
        // parent stays the same; expose parent public id via list mapping
        return $this->presentFolder($updated, null);
    }

    public function deleteFolder(int $userId, string $folderPublicId): void
    {
        $repo = $this->requireFolderRepository();
        $folder = $repo->findByPublicId($userId, $folderPublicId);
        if ($folder === null) {
            throw new UserFileException('FOLDER_NOT_FOUND', 'Dossier introuvable.', 404);
        }
        $fid = $folder->id();
        if ($fid === null) {
            throw new UserFileException('FOLDER_NOT_FOUND', 'Dossier introuvable.', 404);
        }
        $childrenFiles = $repo->countFilesInFolder($userId, $fid);
        $childrenFolders = $repo->countSubFolders($userId, $fid);
        if ($childrenFiles > 0 || $childrenFolders > 0) {
            throw new UserFileException('FOLDER_NOT_EMPTY', 'Le dossier n’est pas vide.', 422);
        }
        $repo->delete($folder);
    }

    /**
     * Supprime un dossier et tout son sous-arbre (fichiers + sous-dossiers).
     * @return array{files:int,folders:int}
     */
    public function deleteFolderCascade(int $userId, string $folderPublicId): array
    {
        $repo = $this->requireFolderRepository();
        $root = $repo->findByPublicId($userId, $folderPublicId);
        if ($root === null) {
            throw new UserFileException('FOLDER_NOT_FOUND', 'Dossier introuvable.', 404);
        }
        $rootId = $root->id();
        if ($rootId === null) {
            throw new UserFileException('FOLDER_NOT_FOUND', 'Dossier introuvable.', 404);
        }

        // 1) Récupérer la liste complète des dossiers de l'utilisateur
        $allFolders = $repo->listForUser($userId);
        $byParent = [];
        $byId = [];
        foreach ($allFolders as $f) {
            $id = $f->id();
            if ($id === null) continue;
            $byId[$id] = $f;
            $p = $f->parentId();
            $byParent[$p ?? 0][] = $id; // parent null → clé 0
        }

        // 2) Construire l'ensemble des descendants (BFS)
        $stack = [$rootId];
        $desc = [];
        while ($stack) {
            $id = array_pop($stack);
            $desc[] = $id;
            $children = $byParent[$id] ?? [];
            foreach ($children as $childId) {
                $stack[] = $childId;
            }
        }

        // 3) Supprimer d'abord les fichiers appartenant à ces dossiers
        $filesDeleted = 0;
        $files = $this->repository->listForUser($userId);
        foreach ($files as $file) {
            $fid = $file->folderId();
            if ($fid !== null && in_array($fid, $desc, true)) {
                $path = $this->pathFor($file);
                if (is_file($path)) {
                    @unlink($path);
                }
                $this->repository->delete($file);
                $filesDeleted++;
            }
        }

        // 4) Supprimer les dossiers de la feuille à la racine (ordre inverse)
        $foldersDeleted = 0;
        for ($i = count($desc) - 1; $i >= 0; $i--) {
            $id = $desc[$i];
            $folder = $byId[$id] ?? null;
            if ($folder !== null) {
                $repo->delete($folder);
                $foldersDeleted++;
            }
        }

        return ['files' => $filesDeleted, 'folders' => $foldersDeleted];
    }

    /**
     * @param array{name?:string,content?:string,kind?:string,format?:string,mimeType?:string} $payload
     * @return array{file:array<string,mixed>,content:string}
     */
    public function compose(int $userId, array $payload): array
    {
        $kind = strtolower((string) ($payload['kind'] ?? 'text'));
        if ($kind !== 'text') {
            throw new UserFileException('UNSUPPORTED_FILE_KIND', 'Type de fichier non pris en charge pour la composition.', 415);
        }

        $format = strtolower((string) ($payload['format'] ?? 'markdown'));
        if ($format !== 'markdown' && $format !== 'md' && $format !== 'text' && $format !== 'txt') {
            throw new UserFileException('UNSUPPORTED_FILE_FORMAT', 'Format de fichier non pris en charge.', 415);
        }

        $mimeType = $payload['mimeType'] ?? null;
        if (!is_string($mimeType) || $mimeType === '') {
            $mimeType = null;
        }

        return $this->createTextFile(
            $userId,
            (string) ($payload['name'] ?? ''),
            (string) ($payload['content'] ?? ''),
            $mimeType
        );
    }

    /**
     * @param array{content?:string,checksum?:string,kind?:string,format?:string,mimeType?:string} $payload
     * @return array{file:array<string,mixed>,content:string}
     */
    public function writeContent(int $userId, string $publicId, array $payload): array
    {
        $kind = strtolower((string) ($payload['kind'] ?? 'text'));
        if ($kind !== 'text') {
            throw new UserFileException('UNSUPPORTED_FILE_KIND', 'Type de fichier non pris en charge pour l’écriture.', 415);
        }

        $format = strtolower((string) ($payload['format'] ?? 'markdown'));
        if ($format !== 'markdown' && $format !== 'md' && $format !== 'text' && $format !== 'txt') {
            throw new UserFileException('UNSUPPORTED_FILE_FORMAT', 'Format de fichier non pris en charge.', 415);
        }

        $checksum = $payload['checksum'] ?? null;
        if (!is_string($checksum) || $checksum === '') {
            $checksum = null;
        }

        $mimeType = $payload['mimeType'] ?? null;
        if (!is_string($mimeType) || $mimeType === '') {
            $mimeType = null;
        }

        return $this->writeTextFile(
            $userId,
            $publicId,
            (string) ($payload['content'] ?? ''),
            $checksum,
            $mimeType
        );
    }

    /**
     * @return array{file:array<string,mixed>,content:string}
     */
    public function createTextFile(int $userId, string $name, string $content, ?string $preferredMimeType = null): array
    {
        $sanitized = $this->sanitizeOriginalName($name);
        if ($sanitized === '') {
            throw new UserFileException('INVALID_FILE_NAME', 'Nom de fichier invalide.', 422);
        }

        $finalName = $this->ensureMarkdownExtension($sanitized);
        $normalized = $this->normalizeTextContent($content);
        $length = strlen($normalized);
        if ($length > $this->maxFileSize) {
            throw new UserFileException(
                'FILE_TOO_LARGE',
                sprintf('Le fichier dépasse la taille maximale autorisée (%s).', $this->formatByteSize($this->maxFileSize)),
                413
            );
        }

        $publicId = $this->generatePublicId();
        $extension = strtolower((string) pathinfo($finalName, PATHINFO_EXTENSION));
        $storedName = $extension !== '' ? $publicId . '.' . $extension : $publicId;

        $directory = $this->directoryFor($userId);
        $this->ensureDirectory($directory);
        $path = $directory . DIRECTORY_SEPARATOR . $storedName;

        if (@file_put_contents($path, $normalized) === false) {
            throw new UserFileException('FILE_PERSIST_FAILED', 'Impossible de sauvegarder le fichier.', 500);
        }
        @chmod($path, 0664);

        clearstatcache(true, $path);
        $byteSize = $this->resolveFileSize($path, $length);
        $checksum = $this->computeChecksum($path);
        $timestamp = time();
        $mimeType = $this->resolveTextMimeType($preferredMimeType);

        $file = UserFile::createNew(
            $userId,
            $publicId,
            $storedName,
            $finalName,
            $mimeType,
            $byteSize,
            $checksum,
            $timestamp
        );
        $file = $this->repository->insert($file);

        return [
            'file' => $this->present($file),
            'content' => $normalized,
        ];
    }

    /**
     * @return array{file:array<string,mixed>,content:string}
     */
    public function readTextFile(int $userId, string $publicId): array
    {
        $file = $this->repository->findByPublicId($userId, $publicId);
        if ($file === null) {
            throw new UserFileException('FILE_NOT_FOUND', 'Fichier introuvable.', 404);
        }

        $path = $this->pathFor($file);
        if (!is_file($path)) {
            throw new UserFileException('FILE_NOT_FOUND', 'Fichier introuvable.', 404);
        }

        $this->assertMarkdownCompatible($file, $path);

        $body = @file_get_contents($path);
        if ($body === false) {
            throw new UserFileException('FILE_READ_FAILED', 'Impossible de lire le fichier.', 500);
        }

        $length = strlen($body);
        if ($length > $this->maxFileSize) {
            throw new UserFileException(
                'FILE_TOO_LARGE',
                sprintf('Le fichier dépasse la taille maximale autorisée (%s).', $this->formatByteSize($this->maxFileSize)),
                413
            );
        }

        $normalized = $this->normalizeTextContent($body);

        return [
            'file' => $this->present($file),
            'content' => $normalized,
        ];
    }

    /**
     * @return array{file:array<string,mixed>,content:string}
     */
    public function writeTextFile(
        int $userId,
        string $publicId,
        string $content,
        ?string $expectedChecksum = null,
        ?string $preferredMimeType = null
    ): array {
        $file = $this->repository->findByPublicId($userId, $publicId);
        if ($file === null) {
            throw new UserFileException('FILE_NOT_FOUND', 'Fichier introuvable.', 404);
        }

        $path = $this->pathFor($file);
        if (!is_file($path)) {
            throw new UserFileException('FILE_NOT_FOUND', 'Fichier introuvable.', 404);
        }

        if ($expectedChecksum !== null && $expectedChecksum !== '' && $file->checksum() !== null) {
            if (!hash_equals((string) $file->checksum(), (string) $expectedChecksum)) {
                throw new UserFileException('FILE_CONFLICT', 'Le fichier a été modifié depuis la dernière lecture.', 409);
            }
        }

        $this->assertMarkdownCompatible($file, $path);

        $normalized = $this->normalizeTextContent($content);
        $length = strlen($normalized);
        if ($length > $this->maxFileSize) {
            throw new UserFileException(
                'FILE_TOO_LARGE',
                sprintf('Le fichier dépasse la taille maximale autorisée (%s).', $this->formatByteSize($this->maxFileSize)),
                413
            );
        }

        if (@file_put_contents($path, $normalized) === false) {
            throw new UserFileException('FILE_PERSIST_FAILED', 'Impossible d’écrire dans le fichier.', 500);
        }
        @chmod($path, 0664);

        clearstatcache(true, $path);
        $byteSize = $this->resolveFileSize($path, $length);
        $checksum = $this->computeChecksum($path);
        $timestamp = time();

        $mimeType = $file->mimeType();
        if ($mimeType === null || !str_starts_with(strtolower($mimeType), 'text/')) {
            $mimeType = $this->resolveTextMimeType($preferredMimeType);
        }

        $updated = $file->withUpdatedMetadata($file->storedName(), $mimeType, $byteSize, $checksum, $timestamp);
        $this->repository->update($updated);

        return [
            'file' => $this->present($updated),
            'content' => $normalized,
        ];
    }

    /**
     * @return array{file:UserFile,path:string}|null
     */
    public function resolveForDownload(int $userId, string $publicId): ?array
    {
        $file = $this->repository->findByPublicId($userId, $publicId);
        if ($file === null) {
            return null;
        }
        $path = $this->pathFor($file);
        if (!is_file($path)) {
            return null;
        }
        return ['file' => $file, 'path' => $path];
    }

    public function buildContentUrl(UserFile $file): string
    {
        return '/api/files/' . rawurlencode($file->publicId()) . '/content';
    }

    public function buildDownloadName(UserFile $file): string
    {
        return $file->originalName();
    }

    /**
     * @return array<string,mixed>
     */
    public function buildAttachment(UserFile $file, ?array $meta = null): array
    {
        $attachment = [
            'kind' => 'user',
            'origin' => 'user-files',
            'fileId' => $file->publicId(),
            'name' => $file->originalName(),
            'byteSize' => $file->byteSize(),
            'sizeLabel' => $this->formatByteSize($file->byteSize()),
            'url' => $this->buildContentUrl($file),
            'downloadName' => $this->buildDownloadName($file),
            'updatedAt' => $file->updatedAt(),
        ];

        $mime = $file->mimeType();
        if (is_string($mime) && $mime !== '') {
            $attachment['mimeType'] = $mime;
        }

        $checksum = $file->checksum();
        if (is_string($checksum) && $checksum !== '') {
            $attachment['checksum'] = $checksum;
        }

        if ($meta !== null) {
            $attachment['meta'] = $meta;
        }

        return $attachment;
    }

    private function resolveTextMimeType(?string $preferred): string
    {
        if ($preferred !== null) {
            $candidate = strtolower(trim($preferred));
            if ($candidate !== '' && (str_starts_with($candidate, 'text/') || $candidate === 'application/markdown')) {
                return $candidate === 'application/markdown' ? 'text/markdown' : $candidate;
            }
        }
        return 'text/markdown';
    }

    private function present(UserFile $file): array
    {
        $attachment = $this->buildAttachment($file);

        return [
            'id' => $file->publicId(),
            'fileId' => $file->publicId(),
            'name' => $file->originalName(),
            'byteSize' => $file->byteSize(),
            'sizeLabel' => $attachment['sizeLabel'],
            'mimeType' => $file->mimeType(),
            'checksum' => $file->checksum(),
            'createdAt' => $file->createdAt(),
            'updatedAt' => $file->updatedAt(),
            'url' => $this->buildContentUrl($file),
            'downloadName' => $this->buildDownloadName($file),
            'folderId' => $file->folderPublicId(),
            'kind' => 'user',
            'origin' => 'user-files',
            'attachment' => $attachment,
        ];
    }

    /**
     * @return array<string,mixed>
     */
    private function presentFolder(UserFileFolder $folder, ?string $parentPublicId = null): array
    {
        return [
            'id' => $folder->publicId(),
            'name' => $folder->name(),
            'parentId' => $parentPublicId,
            'createdAt' => $folder->createdAt(),
            'updatedAt' => $folder->updatedAt(),
        ];
    }

    private function resolveFileSize(string $path, int $fallback): int
    {
        $actualSize = filesize($path);
        if (is_int($actualSize) && $actualSize >= 0) {
            return $actualSize;
        }
        return max(0, $fallback);
    }

    private function directoryFor(int $userId): string
    {
        $bucket = (int) floor($userId / 1000);
        return $this->storageRoot
            . DIRECTORY_SEPARATOR . sprintf('%03d', $bucket)
            . DIRECTORY_SEPARATOR . (string) $userId;
    }

    private function ensureDirectory(string $directory): void
    {
        if (is_dir($directory)) {
            return;
        }
        if (!@mkdir($directory, 0775, true) && !is_dir($directory)) {
            throw new UserFileException('STORAGE_UNAVAILABLE', 'Impossible de préparer le dossier de stockage.', 500);
        }
    }

    private function pathFor(UserFile $file): string
    {
        return $this->directoryFor($file->userId()) . DIRECTORY_SEPARATOR . $file->storedName();
    }

    private function sanitizeOriginalName(string $name): string
    {
        $trimmed = trim($name);
        $trimmed = preg_replace('/[\\x00-\\x1f\\x7f]+/u', '', $trimmed) ?? '';
        $sanitized = str_replace(
            ['\\', '/', ':', '*', '?', '"', '<', '>', '|'],
            '_',
            $trimmed
        );
        $sanitized = preg_replace('/\s+/', ' ', $sanitized) ?? '';
        $sanitized = trim($sanitized);
        if ($sanitized === '') {
            return 'fichier';
        }
        return mb_substr($sanitized, 0, 160);
    }

    private function sanitizeFolderName(string $name): string
    {
        $n = $this->sanitizeOriginalName($name);
        return $n === 'fichier' ? 'dossier' : $n;
    }

    private function ensureMarkdownExtension(string $name): string
    {
        $ext = strtolower((string) pathinfo($name, PATHINFO_EXTENSION));
        if (in_array($ext, ['md', 'markdown', 'mdown', 'mkd', 'txt'], true)) {
            return $name;
        }
        return $name . '.md';
    }

    private function detectMimeType(string $path, mixed $fallback): ?string
    {
        try {
            if (\function_exists('finfo_open')) {
                $finfo = finfo_open(FILEINFO_MIME_TYPE);
                if ($finfo !== false) {
                    $detected = finfo_file($finfo, $path);
                    finfo_close($finfo);
                    if (is_string($detected) && $detected !== '') {
                        return $detected;
                    }
                }
            }
        } catch (\Throwable) {
            // ignore detection errors
        }
        $candidate = is_string($fallback) ? trim($fallback) : '';
        return $candidate !== '' ? $candidate : null;
    }

    private function computeChecksum(string $path): ?string
    {
        try {
            $hash = hash_file('sha256', $path);
            return $hash !== false ? $hash : null;
        } catch (\Throwable) {
            return null;
        }
    }

    private function generatePublicId(): string
    {
        try {
            return bin2hex(random_bytes(16));
        } catch (\Throwable) {
            return bin2hex(random_bytes(8));
        }
    }

    private function guessExtension(string $name, ?string $mimeType): string
    {
        $ext = strtolower((string) pathinfo($name, PATHINFO_EXTENSION));
        $ext = preg_replace('/[^a-z0-9]+/', '', $ext ?? '') ?? '';
        if ($ext !== '') {
            return $ext;
        }
        return match ($mimeType) {
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'image/svg+xml' => 'svg',
            'application/pdf' => 'pdf',
            'text/plain' => 'txt',
            default => '',
        };
    }

    private function formatByteSize(int $bytes): string
    {
        $units = ['octets', 'Ko', 'Mo', 'Go', 'To'];
        $value = max($bytes, 0);
        $idx = 0;
        while ($value >= 1024 && $idx < count($units) - 1) {
            $value /= 1024;
            $idx++;
        }
        if ($idx === 0) {
            return sprintf('%d %s', $value, $units[$idx]);
        }
        return sprintf('%.1f %s', $value, $units[$idx]);
    }

    private function moveUploadedFile(string $source, string $destination): bool
    {
        if (@move_uploaded_file($source, $destination)) {
            return true;
        }
        return @rename($source, $destination);
    }

    private function normalizeTextContent(mixed $value): string
    {
        $string = '';
        if (is_string($value)) {
            $string = $value;
        } elseif (is_scalar($value)) {
            $string = (string) $value;
        }
        $string = str_replace(["\r\n", "\r"], "\n", $string);
        if (strpos($string, "\0") !== false) {
            throw new UserFileException('FILE_BINARY_CONTENT', 'Le contenu fourni contient des données binaires.', 422);
        }
        return $string;
    }

    private function assertMarkdownCompatible(UserFile $file, string $path): void
    {
        $mimeType = $file->mimeType();
        $storedExt = strtolower((string) pathinfo($file->storedName(), PATHINFO_EXTENSION));
        $originalExt = strtolower((string) pathinfo($file->originalName(), PATHINFO_EXTENSION));
        $allowedExt = ['md', 'markdown', 'mdown', 'mkd', 'txt'];
        if ($mimeType === null || $mimeType === '') {
            $mimeType = $this->detectMimeType($path, null) ?? '';
        }
        $isText = $mimeType !== '' ? str_starts_with(strtolower($mimeType), 'text/') : false;
        if (!$isText) {
            $isText = in_array($storedExt, $allowedExt, true) || in_array($originalExt, $allowedExt, true);
        }
        if (!$isText) {
            throw new UserFileException('FILE_NOT_MARKDOWN', 'Le fichier sélectionné n’est pas un fichier texte compatible.', 422);
        }
    }

    private function shouldAssertUploadedFile(): bool
    {
        if (!\function_exists('is_uploaded_file')) {
            return false;
        }
        $sapi = PHP_SAPI ?? '';
        return !in_array($sapi, ['cli', 'phpdbg', 'cli-server'], true);
    }

    private function resolveUploadErrorMessage(int $error): string
    {
        return match ($error) {
            UPLOAD_ERR_INI_SIZE, UPLOAD_ERR_FORM_SIZE => sprintf('Le fichier dépasse la taille maximale autorisée (%s).', $this->formatByteSize($this->maxFileSize)),
            UPLOAD_ERR_PARTIAL => 'Le téléchargement du fichier est incomplet.',
            UPLOAD_ERR_NO_FILE => 'Aucun fichier fourni.',
            UPLOAD_ERR_NO_TMP_DIR => 'Le dossier temporaire est indisponible.',
            UPLOAD_ERR_CANT_WRITE => 'Impossible d’écrire le fichier sur le disque.',
            UPLOAD_ERR_EXTENSION => 'Le téléchargement a été bloqué par une extension PHP.',
            default => 'Le téléchargement du fichier a échoué.',
        };
    }

    private function resolveUploadErrorStatus(int $error): int
    {
        return match ($error) {
            UPLOAD_ERR_INI_SIZE, UPLOAD_ERR_FORM_SIZE => 413,
            UPLOAD_ERR_NO_FILE => 422,
            default => 400,
        };
    }

    private function requireFolderRepository(): UserFileFolderRepository
    {
        if (!$this->folderRepository) {
            throw new UserFileException('FOLDER_UNSUPPORTED', 'La gestion des dossiers n’est pas activée.', 501);
        }
        return $this->folderRepository;
    }
}
