<?php

declare(strict_types=1);

namespace Skyboard\Domain\Boards;

use Skyboard\Domain\Shared\Identifiers;

final class BoardStateApplier
{
    public function apply(BoardState $board, BoardPatch $patch): BoardState
    {
        $state = $board->toArray();
        foreach ($patch->operations() as $operation) {
            $state = $this->applyOperation($state, $operation);
        }
        return $board->withState($state);
    }

    /** @param array<string,mixed> $state */
    private function applyOperation(array $state, array $operation): array
    {
        return match ($operation['op']) {
            // v3 generic node/tag operations
            'node.create' => $this->opNodeCreate($state, $operation),
            'node.update' => $this->opNodeUpdate($state, $operation),
            'node.move'   => $this->opNodeMove($state, $operation),
            'node.delete' => $this->opNodeDelete($state, $operation),
            'tag.add'     => $this->opTagAdd($state, $operation),
            'tag.remove'  => $this->opTagRemove($state, $operation),
            // filters persist (renommé)
            'filters.set' => $this->opFiltersSet($state, $operation),
            'tagFilter.set' => $this->opTagFilterSet($state, $operation),
            'workspace.rename' => $this->opWorkspaceRename($state, $operation),
            'column.rename' => $this->opColumnRename($state, $operation),
            default => throw new BoardInvariantViolation(
                'patch.unsupported_op',
                'Unsupported operation: ' . (string) ($operation['op'] ?? '')
            ),
        };
    }

    private function opNodeCreate(array $state, array $op): array
    {
        $nodes = &$state['nodes'];
        $parentId = (string) $op['parentId'];
        $index = isset($op['index']) ? (int) $op['index'] : null;
        $nodeId = $op['nodeId'] ?? Identifiers::new();
        if (!isset($nodes[$parentId])) return $state;

        $parent = $nodes[$parentId];
        if (($parent['sys']['shape'] ?? null) !== 'container') {
            throw new BoardInvariantViolation('shape.leaf_has_children', 'Cannot append child to leaf node');
        }

        $shape = $this->extractShape($op['sys'] ?? null);
        $tags = $op['tags'] ?? [];
        if (!is_array($tags)) {
            $tags = [];
        }

        $node = [
            'id' => $nodeId,
            'parentId' => $parentId,
            'children' => [],
            'order' => 0,
            'tags' => $tags,
            'title' => $op['title'] ?? 'Nouveau',
            'description' => $op['description'] ?? null,
            'content' => $op['content'] ?? null,
            'props' => $this->normalizeProps($op['props'] ?? null),
            'sys' => ['shape' => $shape],
            'updatedAt' => time(),
        ];
        $nodes[$nodeId] = $node;
        $arr = &$nodes[$parentId]['children'];
        $pos = $index === null ? count($arr) : max(0, min(count($arr), $index));
        array_splice($arr, $pos, 0, [$nodeId]);
        foreach ($arr as $i => $cid) $nodes[$cid]['order'] = $i;
        $state['updatedAt'] = time();
        return $state;
    }

    private function opNodeUpdate(array $state, array $op): array
    {
        $nodes = &$state['nodes'];
        $id = (string) $op['nodeId'];
        if (!isset($nodes[$id])) return $state;
        $fields = array_intersect_key($op, array_flip(['title','description','content','tags']));
        if (array_key_exists('tags', $fields) && !is_array($fields['tags'])) {
            $fields['tags'] = [];
        }
        foreach ($fields as $k => $v) $nodes[$id][$k] = $v;
        if (array_key_exists('collapsed', $op)) {
            $nodes[$id]['collapsed'] = (bool) $op['collapsed'];
        }
        if (isset($op['sys'])) {
            $shape = $this->extractShape($op['sys']);
            $children = $nodes[$id]['children'] ?? [];
            if ($shape === 'leaf' && is_array($children) && count($children) > 0) {
                throw new BoardInvariantViolation('shape.leaf_has_children', 'Cannot mark node as leaf while it has children');
            }
            $nodes[$id]['sys'] = ['shape' => $shape];
            if ($shape === 'leaf') {
                $nodes[$id]['children'] = [];
            }
        }
        if (array_key_exists('props', $op)) {
            $nodes[$id]['props'] = $this->normalizeProps($op['props']);
        }
        $nodes[$id]['updatedAt'] = time();
        $state['updatedAt'] = time();
        return $state;
    }

    private function opNodeMove(array $state, array $op): array
    {
        $nodes = &$state['nodes'];
        $id  = (string) $op['nodeId'];
        $pid = (string) $op['toParentId'];
        $idx = (int) $op['toIndex'];
        if (!isset($nodes[$id], $nodes[$pid])) return $state;

        if (($nodes[$pid]['sys']['shape'] ?? null) !== 'container') {
            throw new BoardInvariantViolation('shape.leaf_has_children', 'Cannot move node under leaf');
        }

        $fromPid = $nodes[$id]['parentId'];
        if ($fromPid !== null && isset($nodes[$fromPid])) {
            $arr = &$nodes[$fromPid]['children'];
            $pos = array_search($id, $arr, true);
            if ($pos !== false) array_splice($arr, $pos, 1);
            foreach ($arr as $i => $cid) $nodes[$cid]['order'] = $i;
        }

        $nodes[$id]['parentId'] = $pid;
        $arr = &$nodes[$pid]['children'];
        $idx = max(0, min($idx, count($arr)));
        array_splice($arr, $idx, 0, [$id]);
        foreach ($arr as $i => $cid) $nodes[$cid]['order'] = $i;
        $nodes[$id]['updatedAt'] = time();
        $state['updatedAt'] = time();
        return $state;
    }

    private function opNodeDelete(array $state, array $op): array
    {
        $nodes = &$state['nodes'];
        $id = (string) $op['nodeId'];
        if (!isset($nodes[$id])) return $state;

        $removeSubtree = function (string $nid) use (&$nodes, &$removeSubtree): void {
            foreach (($nodes[$nid]['children'] ?? []) as $cid) {
                $removeSubtree($cid);
            }
            unset($nodes[$nid]);
        };

        $pid = $nodes[$id]['parentId'];
        if ($pid !== null && isset($nodes[$pid])) {
            $arr = &$nodes[$pid]['children'];
            $pos = array_search($id, $arr, true);
            if ($pos !== false) array_splice($arr, $pos, 1);
            foreach ($arr as $i => $cid) $nodes[$cid]['order'] = $i;
        }

        $removeSubtree($id);
        $state['updatedAt'] = time();
        return $state;
    }

    private function opTagAdd(array $state, array $op): array
    {
        $nodes = &$state['nodes'];
        $id = (string) ($op['nodeId'] ?? ($op['targetId'] ?? ''));
        if ($id === '' || !isset($nodes[$id])) return $state;

        $normalized = $this->normalizeIncomingTag($op['tag'] ?? null);
        if ($normalized === null) {
            return $state;
        }

        $tags = $nodes[$id]['tags'] ?? [];
        $key = $normalized['key'];
        $tags = array_values(array_filter($tags, fn(array $t): bool => $this->extractTagKey($t) !== $key));
        $tags[] = $normalized;
        $nodes[$id]['tags'] = $tags;
        $nodes[$id]['updatedAt'] = time();
        $state['updatedAt'] = time();
        return $state;
    }

    private function opTagRemove(array $state, array $op): array
    {
        $nodes = &$state['nodes'];
        $id = (string) ($op['nodeId'] ?? ($op['targetId'] ?? ''));
        if ($id === '' || !isset($nodes[$id])) return $state;

        $key = (string) ($op['key'] ?? '');
        if ($key === '') {
            $normalized = $this->normalizeIncomingTag($op['tag'] ?? null);
            if ($normalized !== null) {
                $key = $normalized['key'];
            }
        }
        if ($key === '') {
            return $state;
        }

        $tags = $nodes[$id]['tags'] ?? [];
        $tags = array_values(array_filter($tags, fn(array $t): bool => $this->extractTagKey($t) !== $key));
        $nodes[$id]['tags'] = $tags;
        $nodes[$id]['updatedAt'] = time();
        $state['updatedAt'] = time();
        return $state;
    }

    private function opFiltersSet(array $state, array $op): array
    {
        $filters = $op['filters'] ?? [];
        if (!is_array($filters)) $filters = [];
        $state['filters'] = $filters;
        return $state;
    }

    private function opTagFilterSet(array $state, array $op): array
    {
        $selected = $op['selected'] ?? [];
        if (!is_array($selected)) {
            $selected = [];
        }
        $normalized = [];
        foreach ($selected as $value) {
            if (is_string($value) && $value !== '') {
                $normalized[] = $value;
            }
        }
        $state['tagFilter'] = ['selected' => $normalized];
        return $state;
    }

    private function opWorkspaceRename(array $state, array $op): array
    {
        $workspaceId = isset($op['workspaceId']) ? (string) $op['workspaceId'] : '';
        if ($workspaceId === '' || !isset($state['nodes'][$workspaceId])) {
            return $state;
        }
        $title = isset($op['title']) ? (string) $op['title'] : null;
        if ($title !== null && $title !== '') {
            $state['nodes'][$workspaceId]['title'] = $title;
        }
        $state['nodes'][$workspaceId]['updatedAt'] = time();
        $state['updatedAt'] = time();
        return $state;
    }

    private function opColumnRename(array $state, array $op): array
    {
        $columnId = isset($op['columnId']) ? (string) $op['columnId'] : '';
        if ($columnId === '' || !isset($state['nodes'][$columnId])) {
            return $state;
        }
        $title = isset($op['title']) ? (string) $op['title'] : null;
        if ($title !== null && $title !== '') {
            $state['nodes'][$columnId]['title'] = $title;
        }
        $state['nodes'][$columnId]['updatedAt'] = time();
        $state['updatedAt'] = time();
        return $state;
    }

    private function normalizeIncomingTag(mixed $tag): ?array
    {
        if (!is_array($tag)) {
            return null;
        }
        $key = (string) ($tag['key'] ?? ($tag['k'] ?? ''));
        if ($key === '') {
            return null;
        }
        $kind = (string) ($tag['kind'] ?? '');
        $isSystem = $kind === 'system' || (isset($tag['sys']) && (bool) $tag['sys']);

        $normalized = [
            'key' => $key,
            'k' => $key,
            'kind' => $isSystem ? 'system' : 'user',
            'sys' => $isSystem,
        ];

        if (isset($tag['label'])) {
            $normalized['label'] = (string) $tag['label'];
        }
        if (isset($tag['icon'])) {
            $normalized['icon'] = (string) $tag['icon'];
        }
        if (array_key_exists('color', $tag)) {
            $normalized['color'] = $tag['color'] === null ? null : (string) $tag['color'];
        }
        if (isset($tag['themeColor']) && !isset($normalized['color'])) {
            $normalized['color'] = (string) $tag['themeColor'];
        }
        if (isset($tag['category'])) {
            $normalized['category'] = (string) $tag['category'];
        }
        if (isset($tag['v'])) {
            $normalized['v'] = $tag['v'];
        }
        if (isset($tag['value']) && !isset($normalized['v'])) {
            $normalized['v'] = $tag['value'];
        }

        return $normalized;
    }

    private function extractTagKey(array $tag): string
    {
        return (string) ($tag['key'] ?? ($tag['k'] ?? ''));
    }

    /**
     * @param mixed $sys
     */
    private function extractShape($sys): string
    {
        $shape = null;
        if (is_array($sys)) {
            $shape = $sys['shape'] ?? null;
        }
        if (!is_string($shape)) {
            throw new BoardInvariantViolation('shape.invalid', 'Node shape must be provided');
        }
        if (!in_array($shape, ['container', 'leaf'], true)) {
            throw new BoardInvariantViolation('shape.invalid', 'Unsupported node shape');
        }
        return $shape;
    }

    /**
     * @param mixed $props
     * @return array<string,mixed>
     */
    private function normalizeProps($props, int $depth = 0): array
    {
        if (!is_array($props) || $depth > 3) {
            return [];
        }
        $normalized = [];
        foreach ($props as $key => $value) {
            if (!is_string($key) || $key === '') {
                continue;
            }
            if ($key === 'files') {
                $normalized[$key] = $this->normalizeFileAttachments($value);
                continue;
            }
            $normalized[$key] = $this->normalizePropValue($value, $depth + 1);
        }
        return $normalized;
    }

    /**
     * @param mixed $value
     * @return array<string,array<string,mixed>>
     */
    private function normalizeFileAttachments($value): array
    {
        if (!is_array($value)) {
            return [];
        }
        $attachments = [];
        foreach ($value as $slot => $attachment) {
            if (!is_string($slot) || $slot === '') {
                continue;
            }
            $normalized = $this->normalizeFileAttachment($attachment);
            if ($normalized !== null) {
                $attachments[$slot] = $normalized;
            }
        }
        return $attachments;
    }

    /**
     * @param mixed $value
     * @return array<string,mixed>|null
     */
    private function normalizeFileAttachment($value): ?array
    {
        if (!is_array($value)) {
            return null;
        }
        $fileId = $value['fileId'] ?? ($value['id'] ?? null);
        if (!is_string($fileId) || $fileId === '') {
            return null;
        }

        $attachment = [
            'kind' => $this->normalizeAttachmentKind($value['kind'] ?? null),
            'origin' => $this->normalizeAttachmentOrigin($value['origin'] ?? null),
            'fileId' => $fileId,
        ];

        foreach (['name', 'mimeType', 'sizeLabel', 'checksum', 'url', 'downloadName'] as $field) {
            $raw = $value[$field] ?? null;
            if (is_string($raw) && $raw !== '') {
                $attachment[$field] = $raw;
            }
        }

        if (array_key_exists('byteSize', $value) && is_numeric($value['byteSize'])) {
            $attachment['byteSize'] = (int) $value['byteSize'];
        }

        if (array_key_exists('updatedAt', $value) && is_numeric($value['updatedAt'])) {
            $attachment['updatedAt'] = (int) $value['updatedAt'];
        }

        if (isset($value['meta']) && is_array($value['meta'])) {
            $meta = $this->normalizeProps($value['meta'], 1);
            if ($meta !== []) {
                $attachment['meta'] = $meta;
            }
        }

        return $attachment;
    }

    private function normalizeAttachmentKind($value): string
    {
        if (is_string($value) && $value !== '') {
            return $value;
        }
        return 'user';
    }

    private function normalizeAttachmentOrigin($value): string
    {
        if (is_string($value) && $value !== '') {
            return $value;
        }
        return 'user-files';
    }

    private function normalizePropValue(mixed $value, int $depth): mixed
    {
        if (is_string($value) || is_int($value) || is_float($value) || is_bool($value) || $value === null) {
            return $value;
        }
        if (is_array($value)) {
            if ($depth > 3) {
                return [];
            }
            if (array_is_list($value)) {
                $list = [];
                foreach ($value as $item) {
                    if (is_string($item) || is_int($item) || is_float($item) || is_bool($item) || $item === null) {
                        $list[] = $item;
                        continue;
                    }
                    if (is_array($item)) {
                        $list[] = $this->normalizeProps($item, $depth + 1);
                    }
                }
                return $list;
            }
            return $this->normalizeProps($value, $depth + 1);
        }
        return null;
    }
}
