<?php

declare(strict_types=1);

namespace Skyboard\Domain\Boards;

final class BoardTraversal
{
    /**
     * v3: Renvoie un node par id depuis l'arbre nodes{}
     * @param array<string,mixed> $board
     */
    public static function findNode(array $board, string $nodeId): ?array
    {
        $nodes = $board['nodes'] ?? null;
        if (!is_array($nodes)) return null;
        $n = $nodes[$nodeId] ?? null;
        return is_array($n) ? $n : null;
    }

    /**
     * v3: Collecte les tags uniques présents sur tous les nodes
     * @param array<string,mixed> $board
     * @return list<array<string,mixed>>
     */
    public static function collectTags(array $board): array
    {
        $nodes = $board['nodes'] ?? null;
        if (!is_array($nodes)) return [];
        $result = [];
        $seen = [];
        foreach ($nodes as $node) {
            foreach (($node['tags'] ?? []) as $tag) {
                $kind = (string) ($tag['kind'] ?? ((isset($tag['sys']) && (bool) $tag['sys']) ? 'system' : 'user'));
                $key = (string) ($tag['key'] ?? ($tag['k'] ?? ''));
                if ($key === '') continue;
                $hash = self::hash($kind, $key);
                if (isset($seen[$hash])) continue;
                $seen[$hash] = true;
                $result[] = ['kind' => $kind, 'key' => $key];
            }
        }
        return $result;
    }

    /**
     * v3: Retourne les tags pour une cible
     * @param array<string,mixed> $board
     * @return list<array<string,mixed>>
     */
    public static function tagsForTarget(array $board, string $scope, string $targetId): array
    {
        // En v3, scope pertinent = 'item' (un node), 'list' n'est plus structurel; on mappe vers nodeId
        $node = self::findNode($board, $targetId);
        if ($node && isset($node['tags']) && is_array($node['tags'])) {
            // Normalise en {kind,key}
            $out = [];
            foreach ($node['tags'] as $tag) {
                $kind = (string) ($tag['kind'] ?? ((isset($tag['sys']) && (bool) $tag['sys']) ? 'system' : 'user'));
                $key = (string) ($tag['key'] ?? ($tag['k'] ?? ''));
                if ($key === '') continue;
                $out[] = ['kind' => $kind, 'key' => $key];
            }
            return $out;
        }
        return [];
    }

    private static function hash(string $kind, string $key): string
    {
        return $kind . ':' . $key;
    }
}
