<?php

declare(strict_types=1);

namespace Skyboard\Domain\Rules;

use Skyboard\Domain\Boards\BoardTraversal;

final class RuleActionExecutor
{
    /**
     * @param list<array<string,mixed>> $operations
     * @param array<string,mixed>|null $board
     * @param array<string,mixed> $context
     */
    public function apply(
        RuleDefinition $rule,
        array $operations,
        int $index,
        ?array $board,
        array $context
    ): RuleActionOutcome {
        $mutations = 0;
        $rejection = null;

        foreach ($rule->actions() as $action) {
            $type = (string) ($action['type'] ?? '');
            if ($type === '') {
                continue;
            }

            switch ($type) {
                case 'ensureOnlyOneOf':
                    [$operations, $changed] = $this->ensureOnlyOneOf($operations, $index, $action, $board);
                    $mutations += $changed;
                    break;
                case 'ensure_unique_tag':
                    [$operations, $changed] = $this->ensureOnlyOneOf($operations, $index, $action, $board);
                    $mutations += $changed;
                    break;
                case 'remove_conflicting_tags':
                    [$operations, $changed] = $this->ensureOnlyOneOf($operations, $index, $action, $board);
                    $mutations += $changed;
                    break;
                case 'replaceTag':
                case 'replace_tag':
                    [$operations, $changed] = $this->replaceTag($operations, $index, $action);
                    $mutations += $changed;
                    break;
                case 'addTag':
                    [$operations, $changed] = $this->addTag($operations, $index, $action);
                    $mutations += $changed;
                    break;
                case 'removeTag':
                    [$operations, $changed] = $this->removeTag($operations, $index, $action, $board);
                    $mutations += $changed;
                    break;
                case 'ensureAbsent':
                    [$operations, $changed] = $this->ensureAbsent($operations, $index, $action, $board);
                    $mutations += $changed;
                    break;
                case 'aggregate':
                    [$operations, $changed] = $this->aggregate($operations, $index, $action, $context);
                    $mutations += $changed;
                    break;
                case 'setField':
                case 'set_field':
                    [$operations, $changed] = $this->setField($operations, $index, $action);
                    $mutations += $changed;
                    break;
                case 'increment':
                    [$operations, $changed] = $this->incrementField($operations, $index, $action);
                    $mutations += $changed;
                    break;
                case 'reject':
                case 'veto':
                    $rejection = [
                        'reasonId' => $action['reasonId'] ?? null,
                        'message' => $action['message'] ?? null,
                    ];
                    break 2;
                default:
                    // extensibility: ignore unknown actions
                    break;
            }
        }

        return new RuleActionOutcome($operations, $mutations, $rejection);
    }

    /**
     * @param list<array<string,mixed>> $operations
     * @return array{0:list<array<string,mixed>>,1:int}
     */
    private function ensureOnlyOneOf(array $operations, int $index, array $action, ?array $board): array
    {
        if (!isset($operations[$index])) {
            return [$operations, 0];
        }

        $operation = $operations[$index];
        $opName = (string) ($operation['op'] ?? '');
        if ($opName !== 'tag.add') {
            return [$operations, 0];
        }

        $targetId = (string) ($operation['targetId'] ?? '');
        $scope = (string) ($operation['scope'] ?? 'item');
        $tag = $operation['tag'] ?? null;
        if (!is_array($tag) || $targetId === '') {
            return [$operations, 0];
        }

        $kind = (string) ($action['kind'] ?? ($tag['kind'] ?? 'user'));
        $keys = (array) ($action['keys'] ?? []);
        if ($keys === []) {
            $keys = [$tag['key'] ?? ''];
        }
        $keys = array_filter(array_map('strval', $keys), static fn(string $key): bool => $key !== '');
        if ($keys === []) {
            return [$operations, 0];
        }

        $mutations = 0;
        // Remove conflicting adds already present in patch
        foreach ($operations as $cursor => $candidate) {
            if (($candidate['op'] ?? null) !== 'tag.add') {
                continue;
            }
            if ((string) ($candidate['targetId'] ?? '') !== $targetId) {
                continue;
            }
            if ((string) ($candidate['scope'] ?? 'item') !== $scope) {
                continue;
            }
            $candidateTag = $candidate['tag'] ?? null;
            if (!is_array($candidateTag)) {
                continue;
            }
            if ((string) ($candidateTag['kind'] ?? '') !== $kind) {
                continue;
            }
            if (!in_array((string) ($candidateTag['key'] ?? ''), $keys, true)) {
                continue;
            }
            if ($cursor === $index) {
                continue;
            }
            unset($operations[$cursor]);
            $mutations++;
        }

        // Remove existing tags already on board
        if ($board !== null) {
            $existing = BoardTraversal::tagsForTarget($board, $scope, $targetId);
            foreach ($existing as $current) {
                if ((string) ($current['kind'] ?? '') !== $kind) {
                    continue;
                }
                $key = (string) ($current['key'] ?? '');
                if (!in_array($key, $keys, true) || $key === (string) ($tag['key'] ?? '')) {
                    continue;
                }
                $definition = [
                    'kind' => $kind,
                    'key' => $key,
                ];
                if ($this->hasTagOperation($operations, 'tag.remove', $scope, $targetId, $definition)) {
                    continue;
                }
                $operations[] = [
                    'op' => 'tag.remove',
                    'scope' => $scope,
                    'targetId' => $targetId,
                    'nodeId' => $targetId,
                    'tag' => $definition,
                ];
                $mutations++;
            }
        }

        return [array_values($operations), $mutations];
    }

    /**
     * @param list<array<string,mixed>> $operations
     * @return array{0:list<array<string,mixed>>,1:int}
     */
    private function replaceTag(array $operations, int $index, array $action): array
    {
        if (!isset($operations[$index])) {
            return [$operations, 0];
        }

        $operation = $operations[$index];
        if (($operation['op'] ?? null) !== 'tag.add') {
            return [$operations, 0];
        }

        $tag = $operation['tag'] ?? null;
        if (!is_array($tag)) {
            return [$operations, 0];
        }

        $match = $action['match'] ?? ($action['from'] ?? []);
        $set = $action['set'] ?? ($action['to'] ?? []);
        if (!is_array($match) || !is_array($set)) {
            return [$operations, 0];
        }

        $kind = (string) ($match['kind'] ?? ($tag['kind'] ?? 'user'));
        $key = (string) ($match['key'] ?? ($tag['key'] ?? ''));
        if ($kind === '' || $key === '') {
            return [$operations, 0];
        }

        if ((string) ($tag['kind'] ?? '') !== $kind || (string) ($tag['key'] ?? '') !== $key) {
            return [$operations, 0];
        }

        $operations[$index]['tag'] = [
            'kind' => (string) ($set['kind'] ?? $kind),
            'key' => (string) ($set['key'] ?? $key),
        ];

        return [$operations, 1];
    }

    /**
     * @param list<array<string,mixed>> $operations
     * @return array{0:list<array<string,mixed>>,1:int}
     */
    private function addTag(array $operations, int $index, array $action): array
    {
        if (!isset($operations[$index])) {
            return [$operations, 0];
        }

        $base = $operations[$index];
        $scope = (string) ($action['scope'] ?? ($base['scope'] ?? 'item'));
        $targetId = (string) ($action['targetId'] ?? ($base['targetId'] ?? ''));
        $definition = $action['tag'] ?? null;
        if (!is_array($definition) || $targetId === '') {
            return [$operations, 0];
        }

        $tag = [
            'kind' => (string) ($definition['kind'] ?? 'user'),
            'key' => (string) ($definition['key'] ?? ''),
        ];
        if ($tag['key'] === '') {
            return [$operations, 0];
        }

        if ($this->hasTagOperation($operations, 'tag.add', $scope, $targetId, $tag)) {
            return [$operations, 0];
        }

        $operations[] = [
            'op' => 'tag.add',
            'scope' => $scope,
            'targetId' => $targetId,
            'nodeId' => $targetId,
            'tag' => $tag,
        ];

        return [$operations, 1];
    }

    /**
     * @param list<array<string,mixed>> $operations
     * @return array{0:list<array<string,mixed>>,1:int}
     */
    private function removeTag(array $operations, int $index, array $action, ?array $board): array
    {
        if (!isset($operations[$index])) {
            return [$operations, 0];
        }

        $base = $operations[$index];
        $scope = (string) ($action['scope'] ?? ($base['scope'] ?? 'item'));
        $targetId = (string) ($action['targetId'] ?? ($base['targetId'] ?? ''));
        $tag = $action['tag'] ?? null;
        if (!is_array($tag) || $targetId === '') {
            return [$operations, 0];
        }
        $definition = [
            'kind' => (string) ($tag['kind'] ?? 'user'),
            'key' => (string) ($tag['key'] ?? ''),
        ];
        if ($definition['key'] === '') {
            return [$operations, 0];
        }

        if ($this->hasTagOperation($operations, 'tag.remove', $scope, $targetId, $definition)) {
            return [$operations, 0];
        }

        if ($board !== null) {
            $existing = BoardTraversal::tagsForTarget($board, $scope, $targetId);
            $present = array_filter($existing, fn (array $current): bool => $this->sameTag($current, $definition));
            if ($present === []) {
                return [$operations, 0];
            }
        }

        $operations[] = [
            'op' => 'tag.remove',
            'scope' => $scope,
            'targetId' => $targetId,
            'nodeId' => $targetId,
            'tag' => $definition,
        ];

        return [$operations, 1];
    }

    /**
     * @param list<array<string,mixed>> $operations
     * @return array{0:list<array<string,mixed>>,1:int}
     */
    private function ensureAbsent(array $operations, int $index, array $action, ?array $board): array
    {
        [$operations, $changed] = $this->removeTag($operations, $index, $action, $board);
        if ($changed > 0) {
            return [$operations, $changed];
        }

        if (!isset($operations[$index])) {
            return [$operations, 0];
        }

        $scope = (string) ($action['scope'] ?? ($operations[$index]['scope'] ?? 'item'));
        $targetId = (string) ($action['targetId'] ?? ($operations[$index]['targetId'] ?? ''));
        $tag = $action['tag'] ?? null;
        if (!is_array($tag) || $targetId === '') {
            return [$operations, 0];
        }
        $definition = [
            'kind' => (string) ($tag['kind'] ?? 'user'),
            'key' => (string) ($tag['key'] ?? ''),
        ];

        if ($definition['key'] === '') {
            return [$operations, 0];
        }

        $mutations = 0;
        foreach ($operations as $cursor => $candidate) {
            if (($candidate['op'] ?? null) !== 'tag.add') {
                continue;
            }
            if ((string) ($candidate['targetId'] ?? '') !== $targetId) {
                continue;
            }
            if ((string) ($candidate['scope'] ?? 'item') !== $scope) {
                continue;
            }
            $candidateTag = $candidate['tag'] ?? null;
            if (!is_array($candidateTag)) {
                continue;
            }
            if ($this->sameTag($candidateTag, $definition)) {
                unset($operations[$cursor]);
                $mutations++;
            }
        }

        return [array_values($operations), $mutations];
    }

    /**
     * @param list<array<string,mixed>> $operations
     * @return array{0:list<array<string,mixed>>,1:int}
     */
    private function aggregate(array $operations, int $index, array $action, array $context): array
    {
        $targetPath = (string) ($action['targetPath'] ?? ($action['into'] ?? ''));
        if ($targetPath === '' || !isset($operations[$index])) {
            return [$operations, 0];
        }

        $valuesPath = (string) ($action['valuesPath'] ?? ($action['from'] ?? ''));
        $values = $valuesPath !== '' ? $this->resolvePath($context, $valuesPath) : null;
        if (!is_array($values)) {
            return [$operations, 0];
        }

        $using = (string) ($action['using'] ?? $action['mode'] ?? 'count');
        $result = null;
        switch ($using) {
            case 'count':
                $result = count($values);
                break;
            case 'sum':
                $result = 0;
                foreach ($values as $value) {
                    if (is_numeric($value)) {
                        $result += (float) $value;
                    }
                }
                break;
            case 'avg':
            case 'average':
                $numbers = array_filter($values, 'is_numeric');
                if ($numbers === []) {
                    return [$operations, 0];
                }
                $result = array_sum(array_map('floatval', $numbers)) / count($numbers);
                break;
            case 'min':
                $numbers = array_filter($values, 'is_numeric');
                if ($numbers === []) {
                    return [$operations, 0];
                }
                $result = min(array_map('floatval', $numbers));
                break;
            case 'max':
                $numbers = array_filter($values, 'is_numeric');
                if ($numbers === []) {
                    return [$operations, 0];
                }
                $result = max(array_map('floatval', $numbers));
                break;
            default:
                return [$operations, 0];
        }

        $operation = $operations[$index];
        $existing = $this->resolvePath($operation, $targetPath);
        if ($existing === $result) {
            return [$operations, 0];
        }

        if (!$this->setPath($operation, $targetPath, $result)) {
            return [$operations, 0];
        }

        $operations[$index] = $operation;
        return [$operations, 1];
    }

    /**
     * @param list<array<string,mixed>> $operations
     * @return array{0:list<array<string,mixed>>,1:int}
     */
    private function setField(array $operations, int $index, array $action): array
    {
        if (!isset($operations[$index])) {
            return [$operations, 0];
        }
        $path = (string) ($action['path'] ?? '');
        if ($path === '') {
            return [$operations, 0];
        }
        $operation = $operations[$index];
        if (!$this->setPath($operation, $path, $action['value'] ?? null)) {
            return [$operations, 0];
        }
        $operations[$index] = $operation;
        return [$operations, 1];
    }

    /**
     * @param list<array<string,mixed>> $operations
     * @return array{0:list<array<string,mixed>>,1:int}
     */
    private function incrementField(array $operations, int $index, array $action): array
    {
        if (!isset($operations[$index])) {
            return [$operations, 0];
        }
        $path = (string) ($action['path'] ?? '');
        if ($path === '') {
            return [$operations, 0];
        }
        $amount = (float) ($action['by'] ?? 1);
        $operation = $operations[$index];
        $current = $this->resolvePath($operation, $path);
        if (!is_numeric($current)) {
            $current = 0;
        }
        if (!$this->setPath($operation, $path, $current + $amount)) {
            return [$operations, 0];
        }
        $operations[$index] = $operation;
        return [$operations, 1];
    }

    private function sameTag(array $left, array $right): bool
    {
        return (string) ($left['kind'] ?? '') === (string) ($right['kind'] ?? '')
            && (string) ($left['key'] ?? '') === (string) ($right['key'] ?? '');
    }

    /**
     * @param list<array<string,mixed>> $operations
     */
    private function hasTagOperation(array $operations, string $op, string $scope, string $targetId, array $tag): bool
    {
        foreach ($operations as $candidate) {
            if (($candidate['op'] ?? null) !== $op) {
                continue;
            }
            if ((string) ($candidate['scope'] ?? 'item') !== $scope) {
                continue;
            }
            if ((string) ($candidate['targetId'] ?? '') !== $targetId) {
                continue;
            }
            $candidateTag = $candidate['tag'] ?? null;
            if (!is_array($candidateTag)) {
                continue;
            }
            if ($this->sameTag($candidateTag, $tag)) {
                return true;
            }
        }
        return false;
    }

    /**
     * @param array<string,mixed> $target
     */
    private function setPath(array &$target, string $path, mixed $value): bool
    {
        $segments = explode('.', $path);
        if ($segments === []) {
            return false;
        }

        $cursor =& $target;
        $lastIndex = count($segments) - 1;
        foreach ($segments as $index => $segment) {
            if ($segment === '' || !is_array($cursor)) {
                return false;
            }

            if ($index === $lastIndex) {
                $cursor[$segment] = $value;
                return true;
            }

            if (!array_key_exists($segment, $cursor)) {
                $cursor[$segment] = [];
            } elseif (!is_array($cursor[$segment])) {
                return false;
            }

            $cursor =& $cursor[$segment];
        }

        return false;
    }

    private function resolvePath(array $target, string $path): mixed
    {
        $segments = explode('.', $path);
        $cursor = $target;
        foreach ($segments as $segment) {
            if ($segment === '') {
                return null;
            }
            if (is_array($cursor) && array_key_exists($segment, $cursor)) {
                $cursor = $cursor[$segment];
                continue;
            }
            return null;
        }
        return $cursor;
    }
}
