<?php

declare(strict_types=1);

namespace Skyboard\Domain\Rules;

final class RuleConditionEvaluator
{
    /**
     * @param list<array<string,mixed>> $conditions
     * @param array<string,mixed> $context
     */
    public function matches(array $conditions, array $context): bool
    {
        foreach ($conditions as $condition) {
            if (!$this->evaluate((array) $condition, $context)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param array<string,mixed> $condition
     * @param array<string,mixed> $context
     */
    private function evaluate(array $condition, array $context): bool
    {
        if (isset($condition['all']) && is_array($condition['all'])) {
            return $this->matches((array) $condition['all'], $context);
        }

        if (isset($condition['any']) && is_array($condition['any'])) {
            foreach ($condition['any'] as $nested) {
                if ($this->evaluate((array) $nested, $context)) {
                    return true;
                }
            }
            return false;
        }

        if (isset($condition['none']) && is_array($condition['none'])) {
            foreach ($condition['none'] as $nested) {
                if ($this->evaluate((array) $nested, $context)) {
                    return false;
                }
            }
            return true;
        }

        $path = $condition['when'] ?? ($condition['path'] ?? null);
        $value = $path !== null ? $this->resolvePath($context, (string) $path) : null;

        if (array_key_exists('equals', $condition) && $value !== $condition['equals']) {
            return false;
        }
        if (array_key_exists('notEquals', $condition) && $value === $condition['notEquals']) {
            return false;
        }
        if (array_key_exists('in', $condition)) {
            $in = (array) $condition['in'];
            if (!in_array($value, $in, true)) {
                return false;
            }
        }
        if (array_key_exists('notIn', $condition)) {
            $notIn = (array) $condition['notIn'];
            if (in_array($value, $notIn, true)) {
                return false;
            }
        }
        if (array_key_exists('exists', $condition)) {
            $shouldExist = (bool) $condition['exists'];
            $exists = $value !== null;
            if ($exists !== $shouldExist) {
                return false;
            }
        }
        if (array_key_exists('isTrue', $condition) && ((bool) $condition['isTrue']) !== ($value === true)) {
            return false;
        }
        if (array_key_exists('isFalse', $condition) && ((bool) $condition['isFalse']) !== ($value === false)) {
            return false;
        }
        if (array_key_exists('isEmpty', $condition)) {
            $isEmpty = $value === null || $value === '' || (is_array($value) && $value === []);
            if ((bool) $condition['isEmpty'] !== $isEmpty) {
                return false;
            }
        }
        if (array_key_exists('notEmpty', $condition)) {
            $isEmpty = $value === null || $value === '' || (is_array($value) && $value === []);
            if ((bool) $condition['notEmpty'] === $isEmpty) {
                return false;
            }
        }
        if (array_key_exists('contains', $condition)) {
            $needle = $condition['contains'];
            if (is_array($value)) {
                if (!in_array($needle, $value, true)) {
                    return false;
                }
            } elseif (is_string($value)) {
                if (!is_string($needle) || !str_contains($value, $needle)) {
                    return false;
                }
            } else {
                return false;
            }
        }
        if (array_key_exists('notContains', $condition)) {
            $needle = $condition['notContains'];
            if (is_array($value) && in_array($needle, $value, true)) {
                return false;
            }
            if (is_string($value) && is_string($needle) && str_contains($value, $needle)) {
                return false;
            }
        }
        if (array_key_exists('greaterThan', $condition) && !$this->compareNumeric($value, $condition['greaterThan'], '>')) {
            return false;
        }
        if (array_key_exists('lessThan', $condition) && !$this->compareNumeric($value, $condition['lessThan'], '<')) {
            return false;
        }
        if (array_key_exists('greaterOrEquals', $condition) && !$this->compareNumeric($value, $condition['greaterOrEquals'], '>=')) {
            return false;
        }
        if (array_key_exists('lessOrEquals', $condition) && !$this->compareNumeric($value, $condition['lessOrEquals'], '<=')) {
            return false;
        }
        if (array_key_exists('startsWith', $condition)) {
            $prefix = (string) $condition['startsWith'];
            if (!is_string($value) || !str_starts_with($value, $prefix)) {
                return false;
            }
        }
        if (array_key_exists('endsWith', $condition)) {
            $suffix = (string) $condition['endsWith'];
            if (!is_string($value) || !str_ends_with($value, $suffix)) {
                return false;
            }
        }
        if (array_key_exists('matches', $condition)) {
            $pattern = (string) $condition['matches'];
            if (!is_string($value) || @preg_match($pattern, $value) !== 1) {
                return false;
            }
        }
        if (array_key_exists('length', $condition)) {
            $length = is_countable($value) ? count((array) $value) : (is_string($value) ? mb_strlen($value) : null);
            if ($length === null) {
                return false;
            }
            $expected = $condition['length'];
            if (is_array($expected)) {
                if (isset($expected['equals']) && $length !== (int) $expected['equals']) {
                    return false;
                }
                if (isset($expected['min']) && $length < (int) $expected['min']) {
                    return false;
                }
                if (isset($expected['max']) && $length > (int) $expected['max']) {
                    return false;
                }
            } elseif ($length !== (int) $expected) {
                return false;
            }
        }

        return true;
    }

    private function resolvePath(array $context, string $path): mixed
    {
        $segments = explode('.', $path);
        $value = $context;
        foreach ($segments as $segment) {
            if ($segment === '') {
                return null;
            }
            $candidate = $segment;
            $index = null;
            if (ctype_digit($segment)) {
                $index = (int) $segment;
            }
            if ($index !== null) {
                if (is_array($value) && array_key_exists($index, $value)) {
                    $value = $value[$index];
                    continue;
                }
                return null;
            }
            if (is_array($value) && array_key_exists($candidate, $value)) {
                $value = $value[$candidate];
                continue;
            }
            return null;
        }
        return $value;
    }

    private function compareNumeric(mixed $left, mixed $right, string $operator): bool
    {
        if (!is_numeric($left) || !is_numeric($right)) {
            return false;
        }
        $l = (float) $left;
        $r = (float) $right;
        return match ($operator) {
            '>' => $l > $r,
            '<' => $l < $r,
            '>=' => $l >= $r,
            '<=' => $l <= $r,
            default => false,
        };
    }
}
