<?php

declare(strict_types=1);

namespace Skyboard\Domain\Rules;

use Skyboard\Application\Commands\Command;
use Skyboard\Application\Commands\ProposedChange;

final class RuleContextBuilder
{
    /**
     * @param list<RuleContextProjector> $projectors
     */
    public function __construct(private array $projectors = [])
    {
    }

    public function register(RuleContextProjector $projector): void
    {
        $this->projectors[] = $projector;
    }

    /**
     * @param array<string,mixed>|null $board
     * @param list<array<string,mixed>> $operations
     * @return array<string,mixed>
     */
    public function build(?array $board, Command $command, ProposedChange $change, array $operations, int $index): array
    {
        $context = [
            'board' => $board,
            'operation' => $operations[$index] ?? [],
            'operations' => $operations,
            'operationIndex' => $index,
            'events' => $change->events(),
            'postCommitEvents' => $change->postCommitEvents(),
            'command' => [
                'type' => $command::class,
                'actorId' => $command->actorId(),
                'boardId' => $command->boardId(),
            ],
            'patch' => [
                'boardId' => $change->patch()->boardId(),
            ],
        ];

        foreach ($this->projectors as $projector) {
            $addition = $projector->project($context, $board, $command, $change, $operations, $index);
            if ($addition === []) {
                continue;
            }
            $context = $this->merge($context, $addition);
        }

        return $context;
    }

    /**
     * @param array<string,mixed> $base
     * @param array<string,mixed> $addition
     * @return array<string,mixed>
     */
    private function merge(array $base, array $addition): array
    {
        foreach ($addition as $key => $value) {
            if (is_array($value) && isset($base[$key]) && is_array($base[$key])) {
                $base[$key] = $this->merge((array) $base[$key], $value);
                continue;
            }
            $base[$key] = $value;
        }

        return $base;
    }
}
