<?php

declare(strict_types=1);

namespace Skyboard\Domain\UserFiles;

final class UserFile
{
    private ?int $id;
    private int $userId;
    private string $publicId;
    private string $storedName;
    private string $originalName;
    private ?string $mimeType;
    private int $byteSize;
    private ?string $checksum;
    private ?int $folderId = null;
    private ?string $folderPublicId = null;
    private int $createdAt;
    private int $updatedAt;

    public function __construct(
        ?int $id,
        int $userId,
        string $publicId,
        string $storedName,
        string $originalName,
        ?string $mimeType,
        int $byteSize,
        ?string $checksum,
        int $createdAt,
        int $updatedAt,
        ?int $folderId = null,
        ?string $folderPublicId = null
    ) {
        $this->id = $id;
        $this->userId = $userId;
        $this->publicId = $publicId;
        $this->storedName = $storedName;
        $this->originalName = $originalName;
        $this->mimeType = $mimeType;
        $this->byteSize = $byteSize;
        $this->checksum = $checksum;
        $this->createdAt = $createdAt;
        $this->updatedAt = $updatedAt;
        $this->folderId = $folderId;
        $this->folderPublicId = $folderPublicId;
    }

    public static function createNew(
        int $userId,
        string $publicId,
        string $storedName,
        string $originalName,
        ?string $mimeType,
        int $byteSize,
        ?string $checksum,
        int $timestamp
    ): self {
        return new self(
            null,
            $userId,
            $publicId,
            $storedName,
            $originalName,
            $mimeType,
            $byteSize,
            $checksum,
            $timestamp,
            $timestamp,
            null,
            null
        );
    }

    public function withId(int $id): self
    {
        $clone = clone $this;
        $clone->id = $id;
        return $clone;
    }

    public function withOriginalName(string $originalName, int $timestamp): self
    {
        $clone = clone $this;
        $clone->originalName = $originalName;
        $clone->updatedAt = $timestamp;
        return $clone;
    }

    public function withUpdatedMetadata(string $storedName, ?string $mimeType, int $byteSize, ?string $checksum, int $timestamp): self
    {
        $clone = clone $this;
        $clone->storedName = $storedName;
        $clone->mimeType = $mimeType;
        $clone->byteSize = $byteSize;
        $clone->checksum = $checksum;
        $clone->updatedAt = $timestamp;
        return $clone;
    }

    public function touch(int $timestamp): self
    {
        $clone = clone $this;
        $clone->updatedAt = $timestamp;
        return $clone;
    }

    public function withFolderRef(?int $folderId, ?string $folderPublicId, ?int $timestamp = null): self
    {
        $clone = clone $this;
        $clone->folderId = $folderId;
        $clone->folderPublicId = $folderPublicId;
        if ($timestamp !== null) {
            $clone->updatedAt = $timestamp;
        }
        return $clone;
    }

    public function id(): ?int
    {
        return $this->id;
    }

    public function userId(): int
    {
        return $this->userId;
    }

    public function publicId(): string
    {
        return $this->publicId;
    }

    public function storedName(): string
    {
        return $this->storedName;
    }

    public function originalName(): string
    {
        return $this->originalName;
    }

    public function mimeType(): ?string
    {
        return $this->mimeType;
    }

    public function byteSize(): int
    {
        return $this->byteSize;
    }

    public function checksum(): ?string
    {
        return $this->checksum;
    }

    public function folderId(): ?int
    {
        return $this->folderId;
    }

    public function folderPublicId(): ?string
    {
        return $this->folderPublicId;
    }

    public function createdAt(): int
    {
        return $this->createdAt;
    }

    public function updatedAt(): int
    {
        return $this->updatedAt;
    }

    /**
     * @return array<string,mixed>
     */
    public function toRecord(): array
    {
        return [
            'id' => $this->id,
            'user_id' => $this->userId,
            'public_id' => $this->publicId,
            'stored_name' => $this->storedName,
            'original_name' => $this->originalName,
            'mime_type' => $this->mimeType,
            'byte_size' => $this->byteSize,
            'checksum' => $this->checksum,
            'folder_id' => $this->folderId,
            'created_at' => $this->createdAt,
            'updated_at' => $this->updatedAt,
        ];
    }
}
