<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Http;

final class Response
{
    public function __construct(
        public readonly int $status,
        public readonly array $headers,
        public readonly string $body
    ) {
    }

    public static function json(array $payload, int $status = 200, array $extraHeaders = []): self
    {
        $headers = array_merge(['Content-Type' => 'application/json; charset=utf-8'], $extraHeaders);
        $flags = JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_THROW_ON_ERROR;
        // Plus de normalisation automatique: on renvoie tel quel.
        $data = $payload;
        try {
            $body = json_encode($data, $flags);
            return new self($status, $headers, $body);
        } catch (\JsonException $encodeError) {
            error_log('JSON_ENCODE_FAILED: ' . $encodeError->getMessage());
            $sanitized = self::sanitizeUtf8($data);
            try {
                $body = json_encode($sanitized, $flags);
                return new self($status, $headers, $body);
            } catch (\JsonException $secondError) {
                error_log('JSON_ENCODE_FAILED_FALLBACK: ' . $secondError->getMessage());
                $fallback = [
                    'ok' => false,
                    'error' => 'JSON_ENCODE_FAILED',
                    'code' => 'JSON_ENCODE_FAILED',
                    'message' => 'Impossible de sérialiser la réponse.',
                ];
                $body = json_encode($fallback, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                return new self(500, $headers, $body === false ? '{"ok":false,"code":"JSON_ENCODE_FAILED","message":"Impossible de sérialiser la réponse."}' : $body);
            }
        }
    }

    /**
     * Convenience helper for canonical success responses.
     * Always enforces ok:true and delegates to json().
     *
     * @param array<string,mixed> $payload
     */
    public static function ok(array $payload = [], int $status = 200, array $extraHeaders = []): self
    {
        $payload['ok'] = true;
        return self::json($payload, $status, $extraHeaders);
    }

    /**
     * Convenience helper for canonical error responses.
     * Always enforces ok:false with code/message and optional details.
     *
     * @param array<string,mixed> $details
     */
    public static function error(string $code, string $message, array $details = [], int $status = 400, array $extraHeaders = []): self
    {
        $payload = [
            'ok' => false,
            'code' => $code,
            'message' => $message,
        ];
        if ($details !== []) {
            $payload['details'] = $details;
        }
        return self::json($payload, $status, $extraHeaders);
    }

    public static function text(string $body, int $status = 200, array $extraHeaders = []): self
    {
        $headers = array_merge(['Content-Type' => 'text/plain; charset=utf-8'], $extraHeaders);
        return new self($status, $headers, $body);
    }

    // (Removed older duplicate enforceEnvelope version above)

    public static function binary(string $body, string $contentType = 'application/octet-stream', int $status = 200, array $extraHeaders = []): self
    {
        $baseHeaders = [
            'Content-Type' => $contentType,
            'Content-Length' => (string) strlen($body),
        ];
        $headers = array_merge($baseHeaders, $extraHeaders);
        return new self($status, $headers, $body);
    }

    public function send(): void
    {
        http_response_code($this->status);
        foreach ($this->headers as $name => $value) {
            header($name . ': ' . $value);
        }
        echo $this->body;
    }

    /**
     * @param mixed $value
     * @return mixed
     */
    private static function sanitizeUtf8(mixed $value): mixed
    {
        if (is_array($value)) {
            $clean = [];
            foreach ($value as $key => $item) {
                $clean[$key] = self::sanitizeUtf8($item);
            }
            return $clean;
        }

        if (is_string($value)) {
            if (mb_detect_encoding($value, 'UTF-8', true) !== false) {
                return $value;
            }
            $converted = @iconv('UTF-8', 'UTF-8//IGNORE', $value);
            if ($converted === false) {
                return mb_convert_encoding($value, 'UTF-8', 'UTF-8');
            }
            return $converted;
        }

        return $value;
    }

    /**
     * Enforce a minimal public envelope on JSON responses without breaking existing keys.
     * - Ensures `ok` is present and boolean.
     * - On errors, ensures `code` and `message` are present (deriving from `error`/status if needed).
     * Success payloads are left mostly untouched.
     *
     * @param array<string,mixed> $payload
     * @return array<string,mixed>
     */
    // Ancienne normalisation d’enveloppe retirée (pas de parachute)
}
