<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Http;

use Closure;

final class Router
{
    /** @var array<string,list<array{path:string,handler:Closure,regex:string,variables:list<string>}>> */
    private array $routes = [];

    public function add(string $method, string $path, Closure $handler): void
    {
        $method = strtoupper($method);
        [$regex, $variables] = $this->compile($path);
        $this->routes[$method][] = [
            'path' => $path,
            'handler' => $handler,
            'regex' => $regex,
            'variables' => $variables,
        ];
    }

    public function dispatch(Request $request): ?Response
    {
        $method = strtoupper($request->method);
        $candidates = $this->routes[$method] ?? [];
        foreach ($candidates as $route) {
            if ($route['variables'] === [] && $route['path'] === $request->path) {
                return ($route['handler'])($request);
            }

            if ($route['variables'] === []) {
                continue;
            }

            if (!preg_match($route['regex'], $request->path, $matches)) {
                continue;
            }

            $enriched = $request;
            foreach ($route['variables'] as $name) {
                if (isset($matches[$name])) {
                    $enriched = $enriched->withAttribute($name, $matches[$name]);
                }
            }

            return ($route['handler'])($enriched);
        }

        return null;
    }

    /**
     * @return array{0:string,1:list<string>}
     */
    private function compile(string $path): array
    {
        $trimmed = trim($path, '/');
        if ($trimmed === '') {
            return ['#^/?$#', []];
        }

        $segments = explode('/', $trimmed);
        $variables = [];
        $parts = [];

        foreach ($segments as $segment) {
            if ($segment !== '' && $segment[0] === ':') {
                $name = substr($segment, 1);
                if ($name === '') {
                    continue;
                }
                $variables[] = $name;
                $parts[] = '(?P<' . $name . '>[^/]+)';
            } else {
                $parts[] = preg_quote($segment, '#');
            }
        }

        $pattern = '#^/' . implode('/', $parts) . '/?$#';
        return [$pattern, $variables];
    }
}
