<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Packs;

final class DatasetStore
{
    /** @var array<string, DatasetDefinition> */
    private array $datasets = [];

    public function register(string $packId, string $packsDir, array $datasetDefinitions): void
    {
        foreach ($datasetDefinitions as $datasetDef) {
            $id = $datasetDef['id'] ?? '';
            if ($id === '') {
                continue;
            }

            $path = $packsDir . '/' . $packId . '/' . ($datasetDef['path'] ?? '');
            $this->datasets[$id] = new DatasetDefinition(
                $id,
                $packId,
                $datasetDef['kind'] ?? 'json',
                $datasetDef['schema'] ?? [],
                $path
            );
        }
    }

    public function get(string $id): ?DatasetDefinition
    {
        return $this->datasets[$id] ?? null;
    }

    public function getAll(): array
    {
        return $this->datasets;
    }

    public function load(string $id): ?array
    {
        $dataset = $this->get($id);
        if (!$dataset || !file_exists($dataset->path)) {
            return null;
        }

        $content = file_get_contents($dataset->path);
        if ($content === false) {
            return null;
        }

        return json_decode($content, true);
    }

    public function save(string $id, array $data): bool
    {
        $dataset = $this->get($id);
        if (!$dataset) {
            return false;
        }

        $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        if ($json === false) {
            return false;
        }

        $directory = dirname($dataset->path);
        if (!is_dir($directory)) {
            @mkdir($directory, 0775, true);
        }

        $tmpPath = $dataset->path . '.' . uniqid('tmp', true);
        $bytes = @file_put_contents($tmpPath, $json . PHP_EOL, LOCK_EX);
        if ($bytes === false) {
            @unlink($tmpPath);
            return false;
        }
        @chmod($tmpPath, 0664);
        if (!@rename($tmpPath, $dataset->path)) {
            @unlink($tmpPath);
            return false;
        }
        @chmod($dataset->path, 0664);
        return true;
    }

    /**
     * Compute a weak ETag for a dataset based on file contents, or empty if unavailable.
     */
    public function getEtag(string $id): string
    {
        $dataset = $this->get($id);
        if (!$dataset || !file_exists($dataset->path)) {
            return '';
        }
        $content = file_get_contents($dataset->path);
        if ($content === false) {
            return '';
        }
        return 'W/"' . substr(sha1($content), 0, 32) . '"';
    }

    /**
     * Minimal schema validation supporting a subset used by packs (object + patternProperties + required + enum + string types).
     * Returns a list of error messages. Empty array means valid.
     */
    public function validateData(string $id, array $data): array
    {
        $def = $this->get($id);
        if (!$def) {
            return ["Dataset not found: $id"];
        }
        $schema = $def->schema;
        if (!$schema) return [];
        if (($schema['type'] ?? null) === 'object') {
            if (!is_array($data)) return ['Dataset must be an object'];
            $required = isset($schema['required']) && is_array($schema['required']) ? $schema['required'] : [];
            foreach ($required as $field) {
                if (!array_key_exists($field, $data)) {
                    return ["Dataset missing required field: $field"];
                }
            }
            $properties = isset($schema['properties']) && is_array($schema['properties']) ? $schema['properties'] : [];
            foreach ($properties as $name => $propSchema) {
                if (!array_key_exists($name, $data)) {
                    continue;
                }
                $errors = $this->validateAgainstSchema($data[$name], $propSchema, (string) $name);
                if (!empty($errors)) {
                    return $errors;
                }
            }
            $patternProps = $schema['patternProperties'] ?? null;
            if (is_array($patternProps) && count($patternProps) >= 1) {
                foreach ($patternProps as $pattern => $childSchema) {
                    $regex = '#' . $pattern . '#';
                    foreach ($data as $key => $value) {
                        if (!preg_match($regex, (string) $key)) continue;
                        if (!is_array($value)) return ["Entry '$key' must be an object"];
                        $errors = $this->validateEntry($value, $childSchema, (string) $key);
                        if (!empty($errors)) return $errors;
                    }
                }
            }
        }
        return [];
    }

    /**
     * @param mixed $value
     * @param array<string,mixed> $schema
     */
    private function validateAgainstSchema(mixed $value, array $schema, string $path): array
    {
        $type = $schema['type'] ?? null;
        if ($type === 'object') {
            if (!is_array($value)) {
                return ["$path must be an object"];
            }
            $required = isset($schema['required']) && is_array($schema['required']) ? $schema['required'] : [];
            foreach ($required as $field) {
                if (!array_key_exists($field, $value)) {
                    return ["$path missing required field: $field"];
                }
            }
            $properties = isset($schema['properties']) && is_array($schema['properties']) ? $schema['properties'] : [];
            foreach ($properties as $name => $childSchema) {
                if (!array_key_exists($name, $value)) {
                    continue;
                }
                $errors = $this->validateAgainstSchema($value[$name], $childSchema, $path . '.' . $name);
                if (!empty($errors)) {
                    return $errors;
                }
            }
            $patternProps = $schema['patternProperties'] ?? null;
            if (is_array($patternProps)) {
                foreach ($patternProps as $pattern => $childSchema) {
                    $regex = '#' . $pattern . '#';
                    foreach ($value as $key => $entry) {
                        if (!preg_match($regex, (string) $key)) {
                            continue;
                        }
                        if (!is_array($entry)) {
                            return ["$path.$key must be an object"];
                        }
                        $errors = $this->validateEntry($entry, $childSchema, $path . '.' . $key);
                        if (!empty($errors)) {
                            return $errors;
                        }
                    }
                }
            }
            return [];
        }

        if ($type === 'array') {
            if (!is_array($value)) {
                return ["$path must be an array"];
            }
            $itemSchema = $schema['items'] ?? null;
            if (is_array($itemSchema)) {
                foreach ($value as $index => $entry) {
                    $errors = $this->validateAgainstSchema($entry, $itemSchema, $path . '[' . $index . ']');
                    if (!empty($errors)) {
                        return $errors;
                    }
                }
            }
            return [];
        }

        if ($type === 'string' && !is_string($value)) {
            return ["$path must be a string"];
        }
        if ($type === 'integer' && !is_int($value)) {
            return ["$path must be an integer"];
        }
        if ($type === 'number' && !is_numeric($value)) {
            return ["$path must be a number"];
        }
        if (isset($schema['enum']) && is_array($schema['enum']) && !in_array($value, $schema['enum'], true)) {
            return ["$path not in enum"];
        }

        return [];
    }

    private function validateEntry(array $entry, array $schema, string $entryKey): array
    {
        $required = $schema['required'] ?? [];
        foreach ($required as $field) {
            if (!array_key_exists($field, $entry)) {
                return ["Entry '$entryKey' missing required field: $field"]; 
            }
        }
        $props = $schema['properties'] ?? [];
        foreach ($props as $name => $propSchema) {
            if (!array_key_exists($name, $entry)) continue;
            $val = $entry[$name];
            $type = $propSchema['type'] ?? null;
            if ($type === 'string' && !is_string($val)) {
                return ["Entry '$entryKey' field '$name' must be a string"]; 
            }
            if (isset($propSchema['enum']) && is_array($propSchema['enum'])) {
                if (!in_array($val, $propSchema['enum'], true)) {
                    return ["Entry '$entryKey' field '$name' not in enum"];
                }
            }
        }
        return [];
    }
}

final class DatasetDefinition
{
    public function __construct(
        public readonly string $id,
        public readonly string $packId,
        public readonly string $kind,
        public readonly array $schema,
        public readonly string $path
    ) {
    }
}
