<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Packs;

final class MccValidator
{
    private array $schema;

    public function __construct()
    {
        $schemaPath = \ROOT_PATH . '/config/mcc.schema.json';
        $schemaContent = file_get_contents($schemaPath);
        if ($schemaContent === false) {
            throw new \RuntimeException("Cannot load MCC schema from: $schemaPath");
        }
        $this->schema = json_decode($schemaContent, true);
        if ($this->schema === null) {
            throw new \RuntimeException("Invalid JSON schema");
        }
    }

    public function validate(array $mcc): array
    {
        $errors = [];
        
        // Validation basique des champs requis
        if (!isset($mcc['pack'])) {
            $errors[] = "Missing required field: pack";
            return $errors;
        }

        if (!isset($mcc['pack']['id'])) {
            $errors[] = "Missing required field: pack.id";
        }

        if (!isset($mcc['pack']['version'])) {
            $errors[] = "Missing required field: pack.version";
        }

        // Validation des tags
        if (isset($mcc['tags'])) {
            foreach ($mcc['tags'] as $index => $tag) {
                if (!isset($tag['key'])) {
                    $errors[] = "Tag at index $index missing required field: key";
                }
                if (!isset($tag['category'])) {
                    $errors[] = "Tag at index $index missing required field: category";
                }
            }
        }

        // Validation des règles
        if (isset($mcc['rules'])) {
            foreach ($mcc['rules'] as $index => $rule) {
                if (!is_array($rule)) {
                    $errors[] = "Rule at index $index is not an object";
                    continue;
                }
                if (!isset($rule['id'])) {
                    $errors[] = "Rule at index $index missing required field: id";
                }
                if (!isset($rule['scope'])) {
                    $errors[] = "Rule at index $index missing required field: scope";
                }
                if (!isset($rule['priority'])) {
                    $errors[] = "Rule at index $index missing required field: priority";
                }
                if (!isset($rule['triggers']) || !is_array($rule['triggers']) || $rule['triggers'] === []) {
                    $errors[] = "Rule at index $index missing required field: triggers";
                }
                if (!isset($rule['conditions']) || !is_array($rule['conditions'])) {
                    $errors[] = "Rule at index $index missing required field: conditions";
                }
                if (!isset($rule['actions']) || !is_array($rule['actions']) || $rule['actions'] === []) {
                    $errors[] = "Rule at index $index missing required field: actions";
                }
            }
        }

        // Validation des UI slots (format strict minimal)
        if (isset($mcc['ui']['slots'])) {
            foreach (($mcc['ui']['slots'] ?? []) as $index => $slot) {
                if (!is_array($slot)) {
                    $errors[] = "UI slot at index $index is not an object";
                    continue;
                }
                if (!isset($slot['id']) || !is_string($slot['id'])) {
                    $errors[] = "UI slot at index $index missing required field: id";
                }
                if (!isset($slot['component']) || !is_string($slot['component'])) {
                    $errors[] = "UI slot at index $index missing required field: component";
                } else {
                    if (!preg_match('#^/assets/packs/[a-z0-9-]+/.+\\.js$#', $slot['component'])) {
                        $errors[] = "UI slot at index $index has invalid component path";
                    }
                }
                if (isset($slot['export']) && !is_string($slot['export'])) {
                    $errors[] = "UI slot at index $index has invalid export (must be string)";
                }
                if (isset($slot['order']) && !is_int($slot['order'])) {
                    $errors[] = "UI slot at index $index has invalid order (must be int)";
                }
            }
        }

        return $errors;
    }

    public function isValid(array $mcc): bool
    {
        return empty($this->validate($mcc));
    }
}
