<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Packs;

final class UiSlotRegistry
{
    /** @var list<UiSlot> */
    private array $slots = [];

    public function register(string $packId, array $uiDefinitions): void
    {
        // Nouveau format MCC: ui.slots[] (source unique)
        foreach (($uiDefinitions['slots'] ?? []) as $slotDef) {
            if (!is_array($slotDef)) {
                continue;
            }
            $slotId = $slotDef['id'] ?? $slotDef['slot'] ?? null;
            $component = $slotDef['component'] ?? $slotDef['module'] ?? null;
            $type = $slotDef['type'] ?? null;
            if (!is_string($slotId) || !is_string($component)) {
                continue;
            }
            $this->slots[] = new UiSlot(
                $slotId,
                $type ?? 'generic',
                $packId,
                $component,
                $slotDef
            );
        }
    }

    /**
     * @return list<UiSlot>
     */
    public function getSlots(): array
    {
        return $this->slots;
    }

    /**
     * @return list<UiSlot>
     */
    public function getEnabledSlots(array $enabledPacks): array
    {
        return array_values(array_filter(
            $this->slots,
            fn(UiSlot $slot) => in_array($slot->packId, $enabledPacks, true)
        ));
    }

    public function getSlotsByType(string $type): array
    {
        return array_values(array_filter(
            $this->slots,
            fn(UiSlot $slot) => $slot->type === $type
        ));
    }

    public function getSlotsByPack(string $packId): array
    {
        return array_values(array_filter(
            $this->slots,
            fn(UiSlot $slot) => $slot->packId === $packId
        ));
    }
}

final class UiSlot
{
    public function __construct(
        public readonly string $id,
        public readonly string $type,
        public readonly string $packId,
        public readonly string $component,
        public readonly array $config
    ) {
    }
}
