<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Persistence;

use PDO;
use Skyboard\Application\Services\UserFileRepository;
use Skyboard\Domain\UserFiles\UserFile;
final class MySqlUserFileRepository implements UserFileRepository
{
    private PDO $pdo;

    public function __construct(DatabaseConnection $connection)
    {
        $this->pdo = $connection->pdo();
    }

    /**
     * @return list<UserFile>
     */
    public function listForUser(int $userId): array
    {
        $stmt = $this->pdo->prepare('SELECT uf.*, uff.id AS folder_id, uff.public_id AS folder_public_id
            FROM user_files uf
            LEFT JOIN user_file_folders uff ON uff.id = uf.folder_id
            WHERE uf.user_id = :user
            ORDER BY uf.created_at DESC, uf.id DESC');
        $stmt->execute(['user' => $userId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if (!$rows) {
            return [];
        }

        $files = [];
        foreach ($rows as $row) {
            $mapped = $this->mapRow($row);
            if ($mapped !== null) {
                $files[] = $mapped;
            }
        }
        return $files;
    }

    public function insert(UserFile $file): UserFile
    {
        $stmt = $this->pdo->prepare(
            'INSERT INTO user_files (user_id, public_id, stored_name, original_name, mime_type, byte_size, checksum, created_at, updated_at)
             VALUES (:user_id, :public_id, :stored_name, :original_name, :mime_type, :byte_size, :checksum, :created_at, :updated_at)'
        );
        $record = $file->toRecord();
        $stmt->execute([
            'user_id' => $record['user_id'],
            'public_id' => $record['public_id'],
            'stored_name' => $record['stored_name'],
            'original_name' => $record['original_name'],
            'mime_type' => $record['mime_type'],
            'byte_size' => $record['byte_size'],
            'checksum' => $record['checksum'],
            'created_at' => $record['created_at'],
            'updated_at' => $record['updated_at'],
        ]);

        $id = (int) $this->pdo->lastInsertId();
        return $file->withId($id);
    }

    public function findByPublicId(int $userId, string $publicId): ?UserFile
    {
        $stmt = $this->pdo->prepare('SELECT uf.*, uff.id AS folder_id, uff.public_id AS folder_public_id
            FROM user_files uf
            LEFT JOIN user_file_folders uff ON uff.id = uf.folder_id
            WHERE uf.user_id = :user AND uf.public_id = :public_id
            LIMIT 1');
        $stmt->execute([
            'user' => $userId,
            'public_id' => $publicId,
        ]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row || !is_array($row)) {
            return null;
        }
        return $this->mapRow($row);
    }

    public function update(UserFile $file): void
    {
        $record = $file->toRecord();
        $stmt = $this->pdo->prepare(
            'UPDATE user_files
             SET stored_name = :stored_name,
                 original_name = :original_name,
                 mime_type = :mime_type,
                 byte_size = :byte_size,
                 checksum = :checksum,
                 folder_id = :folder_id,
                 updated_at = :updated_at
             WHERE user_id = :user_id AND public_id = :public_id'
        );
        $stmt->execute([
            'stored_name' => $record['stored_name'],
            'original_name' => $record['original_name'],
            'mime_type' => $record['mime_type'],
            'byte_size' => $record['byte_size'],
            'checksum' => $record['checksum'],
            'folder_id' => $record['folder_id'],
            'updated_at' => $record['updated_at'],
            'user_id' => $record['user_id'],
            'public_id' => $record['public_id'],
        ]);
    }

    public function delete(UserFile $file): void
    {
        $stmt = $this->pdo->prepare('DELETE FROM user_files WHERE user_id = :user AND public_id = :public');
        $stmt->execute([
            'user' => $file->userId(),
            'public' => $file->publicId(),
        ]);
    }

    /**
     * @param array<string,mixed> $row
     */
    private function mapRow(array $row): ?UserFile
    {
        try {
            $file = new UserFile(
                isset($row['id']) ? (int) $row['id'] : null,
                isset($row['user_id']) ? (int) $row['user_id'] : 0,
                (string) ($row['public_id'] ?? ''),
                (string) ($row['stored_name'] ?? ''),
                (string) ($row['original_name'] ?? ''),
                $row['mime_type'] !== null ? (string) $row['mime_type'] : null,
                isset($row['byte_size']) ? (int) $row['byte_size'] : 0,
                $row['checksum'] !== null ? (string) $row['checksum'] : null,
                isset($row['created_at']) ? (int) $row['created_at'] : time(),
                isset($row['updated_at']) ? (int) $row['updated_at'] : time(),
                isset($row['folder_id']) ? (int) $row['folder_id'] : null,
                isset($row['folder_public_id']) && $row['folder_public_id'] !== null ? (string) $row['folder_public_id'] : null,
            );
            return $file;
        } catch (\Throwable) {
            return null;
        }
    }
}
