<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Security;

use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class RateLimiter implements RateLimiterInterface
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    public function hit(string $ip, int $threshold, int $banSeconds): void
    {
        $pdo = $this->connection->pdo();
        $stmt = $pdo->prepare('SELECT fails, banned_until FROM ip_rec WHERE ip = :ip');
        $stmt->execute(['ip' => $ip]);
        $row = $stmt->fetch();
        $now = time();
        if ($row) {
            if (($row['banned_until'] ?? 0) > $now) {
                return;
            }
            $fails = ((int) $row['fails']) + 1;
            $banned = $fails >= $threshold ? $now + $banSeconds : 0;
            $pdo->prepare('UPDATE ip_rec SET fails = :fails, banned_until = :banned WHERE ip = :ip')
                ->execute(['fails' => $fails, 'banned' => $banned, 'ip' => $ip]);
        } else {
            $pdo->prepare('INSERT INTO ip_rec(ip, fails, banned_until) VALUES(:ip, 1, 0)')
                ->execute(['ip' => $ip]);
        }
    }

    public function clear(string $ip): void
    {
        $this->connection->pdo()->prepare('DELETE FROM ip_rec WHERE ip = :ip')->execute(['ip' => $ip]);
    }

    public function isBanned(string $ip): bool
    {
        $stmt = $this->connection->pdo()->prepare('SELECT banned_until FROM ip_rec WHERE ip = :ip');
        $stmt->execute(['ip' => $ip]);
        $row = $stmt->fetch();
        if (!$row) {
            return false;
        }
        return (int) $row['banned_until'] > time();
    }
}

