<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Security;

use Skyboard\Infrastructure\Persistence\DatabaseConnection;
use Skyboard\Domain\Shared\Identifiers;

final class SessionManager
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    public function create(int $userId): array
    {
        $token = bin2hex(random_bytes(32));
        $csrf = bin2hex(random_bytes(32));
        $now = time();
        $stmt = $this->connection->pdo()->prepare('INSERT INTO sessions(token, user_id, csrf_token, created_at, last_seen) VALUES(:token, :user, :csrf, :created, :last)');
        $stmt->execute([
            'token' => $token,
            'user' => $userId,
            'csrf' => $csrf,
            'created' => $now,
            'last' => $now,
        ]);
        return ['token' => $token, 'csrf' => $csrf];
    }

    public function destroy(string $token): void
    {
        $stmt = $this->connection->pdo()->prepare('DELETE FROM sessions WHERE token = :token');
        $stmt->execute(['token' => $token]);
    }

    public function get(string $token): ?array
    {
        $stmt = $this->connection->pdo()->prepare('SELECT token, user_id, csrf_token FROM sessions WHERE token = :token');
        $stmt->execute(['token' => $token]);
        $row = $stmt->fetch();
        if (!$row) {
            return null;
        }
        $this->connection->pdo()->prepare('UPDATE sessions SET last_seen = :ts WHERE token = :token')
            ->execute(['ts' => time(), 'token' => $token]);
        return $row;
    }

    public function regenerateCsrf(string $token): ?string
    {
        $csrf = bin2hex(random_bytes(32));
        $stmt = $this->connection->pdo()->prepare('UPDATE sessions SET csrf_token = :csrf WHERE token = :token');
        $stmt->execute(['csrf' => $csrf, 'token' => $token]);
        return $csrf;
    }
}

